# -*- coding: utf-8 -*-
import time
import multiprocessing
import datetime
import simplejson
import socket
from dateutil.relativedelta import *
try:
    from PySide import QtCore, QtGui
except ImportError:
    from PyQt4 import QtCore, QtGui

from config import Config
from qaikuconnection import *
from channel import Channel
from person import Person
from message import Message

from sql import *


class QaikuSynchronization():
    def __init__(self, status_queue):
        self.execute = True
        self.status_queue = status_queue
        self.QC = QaikuConnection()
        self.sections_enabled = { "channels" : True, "users" : True, "post" : True }
        self.total = 1.0
        self.progress = 0.0
        

    def addItem(self, name, id):
        self.status_queue.put([name, id])
        
    def setStatusMajor(self, value):
        self.status_queue.put(["major", value])

        
    def setStatusMinor(self, value):
        self.status_queue.put(["minor", value])


    def setWait(self, value):
        self.status_queue.put(["wait", value])


    def setProgress(self, value):
        self.status_queue.put(["progress", value])
        

    def itemReady(self):
        self.progress += 1.0
        progress = 100.0 * self.progress / self.total
        self.status_queue.put(["progress", progress])
        

    def enableSection(self, section, value):
        self.sections_enabled[section] = value


    def synchronize(self, ret = True):
        self.total = 1.0
        self.progress = 0.0
        conn = SQLConnection()
        if self.sections_enabled["channels"]:
            self.total += len(Channel.channelGetAll()) + 2.0
        if self.sections_enabled["users"]:
            self.total += 1.0
        if self.sections_enabled["post"]:
            self.total += 1.0

        try:
            if self.sections_enabled["channels"]:
                self.channelsUpdate()
            if self.sections_enabled["users"]:
                self.personsUpdate()
                self.itemReady()
            if self.sections_enabled["post"]:
                self.changesToQaiku()
                self.itemReady()
        except socket.error, (code, message):
            print code, message
            self.setStatusMinor("")
            self.setStatusMajor(message)
            self.addItem("error", message)
            time.sleep(2)
            
        conn.commit(True)
        self.itemReady()
        self.addItem("ready", ret)


    def channelsUpdate(self):
        self.setStatusMinor("")
        self.setStatusMajor("Channels")
        channels_c = len(Channel.channelGetAll())
        self.QC.qaikuGetChannels(callback = self.channelsReceiveChannel)
        channels = Channel.channelGetAll()
        self.total += len(channels) - channels_c
        self.itemReady()

        for channel in channels:
            self.addItem("channel", channel.id)

        self.setStatusMinor("updating users stream")
        page = 0
        while True:
            request = self.QC.qaikuGetChannelMessages(callback = self.channelsUpdateMessages)
            page += 1
            if len(request["channel_messages"].json) < 1 or page > 3:
                break
        self.itemReady()

        i = 1
        for channel in channels:
            self.setStatusMinor("updating messages on channel #%s (%d / %d)" % (channel.getName(), i, len(channels)))
            page = 0
            while True:
                request = self.QC.qaikuGetChannelMessages(channel = channel.getName(), callback = self.channelsUpdateMessages)
                page += 1
                if len(request["channel_messages"].json) < 1 or page > 3:
                    break
            i += 1
            self.itemReady()


    def channelsUpdateMessages(self, request):
        if request["status"] != 200:
            print "failed",request["status"],request["reason"],request["host"],request["path"]
            return False

        for v in request["channel_messages"].json:
            #if self.execute == False:
                #break
            if len(v["in_reply_to_status_id"]) > 0:
                continue
            #print v["text"]
            message = Message(None, None, v)
            page = 0
            while True:
                request = self.QC.qaikuGetMessageComments(message = v["id"], callback = self.messagesUpdateComments, page = page, userdata = message)
                page += 1
                if len(request["message_comments"].json) < 1 or page > 3:
                    break
            
        return True


    def channelsUpdateLogo(self, request):
        if request["status"] != 200:
            return False

        channel = request["userdata"]["channel"]
        channel.writeLogoData(request["data"])


    def channelsReceiveChannel(self, request):
        if request["status"] != 200:
            print "failed",request["status"],request["reason"]
            return False

        #Config().sync.setStatusMinor("updating")
        channels = Channel.channelGetAll()
        channels_existing = {}
        for channel in channels:
            channels_existing[channel.getName()] = channel
        qchannels = request["channels"]
        i = 1
        for channel, data in qchannels.channels.items():
            if channels_existing.has_key(channel):
                channels_existing[channel] = None
            newchannel = Channel.channelCreate(channel)
            data["channel"] = newchannel
            self.QC.qaikuGet(data["logo_url"], userdata = data, callback = self.channelsUpdateLogo)
            self.setStatusMinor("updating %d%%" % (int(i * 100 / len(qchannels.channels))))
            i += 1
        for name, channel in channels_existing.items():
            if channel != None:
                channel.delete()


    def messagesUpdateComments(self, request):
        parent = request["userdata"]
        for v in request["message_comments"].json:
            #if self.execute == False:
                #break
            if len(v["in_reply_to_status_id"]) < 1:
                continue
            message = Message(None, parent, v)
        self.addItem("message", parent.id)
            

    def personsUpdate(self):
        self.setStatusMinor("")
        self.setStatusMajor("Users")
        persons = Person.personGetAll()
        for person in persons:
            #if self.execute == False:
                #break
            if person.avatarupdate == 1:
                self.QC.qaikuGet(person.avatarurl, callback = self.personsUpdateAvatar, userdata = person)
                person.avatarupdate = 0
                person.update()


    def personsUpdateAvatar(self, request):
        if request["status"] != 200:
            return False

        person = request["userdata"]
        try:
            person.writeAvatarData(request["data"])
        except:
            pass


    def changesToQaiku(self):
        self.setStatusMinor("")
        self.setStatusMajor("Post to Qaiku")
        q = SQLQuery("sync")
        posts = q.execute()

        i = 1
        for o in posts:
            print "post",o.objectclassname,o.operation
            if o.operation != "create":
                continue
            self.setStatusMajor("Post to Qaiku %s" % (100 * i / len(posts)))
            try:
                if o.objectclassname == "message":
                    m = Message(int(o.objectid))
                    if m.objectSync(self.QC):
                        o.delete()
            except Exception, e:
                print "exception caught",e
            i += 1
        

    def authenticate(self, user = None, password = None):
        request = self.QC.qaikuAuthenticate(username = user, password = password)
        return self.authenticated(request)


    def authenticated(self, request):
        print str(request["status"])+" "+str(request["reason"])
        if request["status"] != 200:
            self.status_queue.put(["error", str(request["status"])+" "+str(request["reason"])])
        self.status_queue.put(["auth", request["data"]])
            #return str(request["status"])+" "+str(request["reason"])
        #quser = QaikuUser(None, request["data"])
        #person = Person(quser.json)
        #return person
        

def QaikuSynchronizationProcess(command_queue, status_queue):
    if SQLConnection.dbtype() == "midgard":
        conn = SQLConnection("/home/user/qaikuclient.sqlite", "schema_midgard.cfg", "/home/user/qaikublobs")
        #conn.create_database()
    else:
        conn = SQLConnection("/home/user/qaikuclient.sqlite", "schema.cfg", "/home/user/qaikublobs")
        conn.create_database()
    #conn.set_commit_limit(1000)
    sync = QaikuSynchronization(status_queue)
    
    while sync.execute == True:
        try:
            command = command_queue.get(True, 1.0)
        except:
            continue
        
        action = command[0]
        if action == "host":
            sync.QC.setHost(command[1])
        elif action == "apikey":
            sync.QC.setApiKey(command[1])
        elif action == "since":
            sync.QC.setSince(command[1])
        elif action == "section":
            sync.enableSection(command[1], command[2])
        elif action == "authenticate":
            sync.authenticate(command[1], command[2])
        elif action == "synchronize":
            sync.setWait(True)
            sync.setStatusMajor("Starting synchronization")
            sync.setStatusMinor("")
            sync.setProgress(0.0)
            time.sleep(1.0)
            sync.synchronize(command[1])
            time.sleep(1.0)
            sync.setWait(False)
            sync.setStatusMajor("Synchronization ready")
            sync.setStatusMinor("")
            sync.setProgress(100.0)
        elif action == "quit":
            sync.execute = False
    print "exit sync process"


class QaikuSynchronize():
    def __init__(self):
        self.status = { "wait" : False, "progress" : 0, "major" : "", "minor" : "", "auth" : False }
        self.execute = True
        self.command_queue = multiprocessing.Queue()
        self.status_queue = multiprocessing.Queue()
        self.process = multiprocessing.Process(target = QaikuSynchronizationProcess, args = (self.command_queue, self.status_queue,))
        self.process.daemon = True
        self.process.start()


    def stop(self):
        self.command_queue.put(["quit"])
        self.process.join()
        #self.process.terminate()


    def readStatus(self):
        items = []
        try:
            while True:
                item = self.status_queue.get_nowait()
                if self.status.has_key(item[0]):
                    self.status[item[0]] = item[1]
                else:
                  items.append(item)  
        except:
            pass
        
        return items


    def getStatus(self):
        return self.status


    def setHost(self, value):
        self.command_queue.put(["host", value])


    def setApiKey(self, value):
        self.command_queue.put(["apikey", value])


    def setSince(self, value):
        self.command_queue.put(["since", value])


    def synchronize(self, ret = True):
        self.command_queue.put(["synchronize", ret])


    def enableSection(self, section, value):
        self.command_queue.put(["section", section, value])


    def authenticate(self, user = None, password = None):
        try:
            while True:
                item = self.status_queue.get_nowait()
                self.status[item[0]] = item[1]
        except:
            pass
        self.command_queue.put(["authenticate", user, password])
        item = self.status_queue.get()
        if item[0] != "auth":
            return item[1]
        quser = QaikuUser(None, item[1])
        person = Person(quser.json)
        return person
        

if __name__ == '__main__':
    sync = QaikuSynchronize()
    
    sync.setHost("www.beta.qaiku.com")
    sync.setApiKey("3470c9ac7e1d45c9a39239808a191417")
#    sync.setSince(datetime.datetime.now() + relativedelta(weeks=-4))
#    sync.setSince()
    sync.synchronize()
    start = datetime.datetime.now()
    try:
        while True:
            sync.readStatus()
            status = sync.getStatus()
            print status
            time.sleep(0.5)
            if status["progress"] >= 100.0:
                break
    except:
        pass
    end = datetime.datetime.now()
    print (end - start)
    sync.stop()
    sys.exit(0)

