#ifdef RUN_STANDALONE_QUERY_TEXT
#	define QUERY_TEXT_WIDGET_TYPE GTK_TYPE_TEXT_VIEW
#else /* !RUN_STANDALONE_QUERY_TEXT */
#	include <hildon/hildon.h>
#	define QUERY_TEXT_WIDGET_TYPE HILDON_TYPE_TEXT_VIEW
#endif /* RUN_STANDALONE_QUERY_TEXT */

#include "query-text.h"
#include "hsl2rgb.h"

typedef struct {
	GArray *marks;
	GtkTextTag *highlight_tag;
	GtkTextTag *uneditable_tag;
} QueryTextInfo;

static QueryTextInfo *
query_text_info_new()
{
	QueryTextInfo *qti = g_new0(QueryTextInfo, 1);

	qti->marks = g_array_new(FALSE, TRUE, sizeof(GtkTextMark *));

	return qti;
}

static void
query_text_info_free(QueryTextInfo *qti)
{
	g_array_free(qti->marks, TRUE);
	g_free(qti);
}

#define CLR_TO_HEX(clr) \
	((((clr).red   >> 8) << 16 & 0xff0000) | \
	 (((clr).green >> 8) <<  8 & 0x00ff00) | \
	 (((clr).blue  >> 8)       & 0x0000ff))

static void
set_highlight_tag_clr(GtkWidget *text, GtkStyle *prev_style_may_be_null, gpointer null)
{
	GtkStyle *style = gtk_widget_get_style(text);
	GdkColor clr = style->bg[GTK_STATE_SELECTED], style_clr = {.red = 0, .green = 0, .blue = 0, .pixel = 0};
	QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");

	if (gtk_style_lookup_color(gtk_widget_get_style(text), "ActiveTextColor", &style_clr))
		clr = style_clr;

	RGBToHSL(&clr);
	/* Brighten a dark highlight and darken a bright highlight */
	clr.blue += 0x4000 * (clr.blue > 0x8000 ? -1 : 1);
	HSLToRGB(&clr);

	g_object_set(G_OBJECT(qti->highlight_tag),
		"background-gdk", &clr,
		"background-set", TRUE,
		NULL);
}

static void
buffer_changed(GtkTextBuffer *buffer, GtkTextView *text)
{
	int Nix;
	QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");
	GtkTextIter text_itr_beg, text_itr_end, mark_itr_beg, mark_itr_end;

	gtk_text_buffer_get_start_iter(buffer, &text_itr_beg);
	gtk_text_buffer_get_end_iter(buffer, &text_itr_end);
	gtk_text_tag_table_remove(gtk_text_buffer_get_tag_table(buffer), qti->highlight_tag); qti->highlight_tag = NULL;
	gtk_text_tag_table_remove(gtk_text_buffer_get_tag_table(buffer), qti->uneditable_tag); qti->uneditable_tag = NULL;
	qti->highlight_tag = gtk_text_buffer_create_tag(buffer, NULL, NULL);
	qti->uneditable_tag = gtk_text_buffer_create_tag(buffer, NULL, NULL);
	set_highlight_tag_clr(GTK_WIDGET(text), NULL, NULL);

	if (qti->marks->len > 1)
		for (Nix = 0 ; Nix < qti->marks->len ; Nix += 2) {
			gtk_text_buffer_get_iter_at_mark(buffer, &mark_itr_beg, g_array_index(qti->marks, GtkTextMark *, Nix));
			gtk_text_buffer_get_iter_at_mark(buffer, &mark_itr_end, g_array_index(qti->marks, GtkTextMark *, Nix + 1));
			gtk_text_buffer_apply_tag(buffer, qti->uneditable_tag, &text_itr_beg, &mark_itr_beg);
			gtk_text_buffer_apply_tag(buffer, qti->highlight_tag, &mark_itr_beg, &mark_itr_end);
			text_itr_beg = mark_itr_end;
		}

	gtk_text_buffer_apply_tag(buffer, qti->uneditable_tag, &text_itr_beg, &text_itr_end);
}

static void
buffer_insert_text(GtkTextBuffer *buffer, GtkTextIter *location, gchar *str, gint len, GtkTextView *text)
{
	int Nix;
	GtkTextIter beg, end;
	QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");

	for (Nix = 0 ; Nix < qti->marks->len ; Nix += 2) {
		gtk_text_buffer_get_iter_at_mark(buffer, &beg, g_array_index(qti->marks, GtkTextMark *, Nix));
		gtk_text_buffer_get_iter_at_mark(buffer, &end, g_array_index(qti->marks, GtkTextMark *, Nix + 1));
		if (gtk_text_iter_get_offset(location) >= gtk_text_iter_get_offset(&beg) &&
		    gtk_text_iter_get_offset(location) <= gtk_text_iter_get_offset(&end))
			return;
	}

	g_signal_stop_emission_by_name(buffer, "insert-text");
}

/*
Given:   |-----------------------|
[ ] |-|
[ ] |----|
[X] |----------------|
[X] |----------------------------|
[X] |--------------------------------|
[X]      |-----------|
[X]      |-----------------------|
[X] 	   |---------------------------|
[ ] 	                           |---|
[ ]                                |-|
*/
static gboolean
get_has_editable_bits(GtkTextBuffer *buffer, GtkTextIter *beg, GtkTextIter *end, GArray *marks)
{
	int Nix, beg_offset, end_offset, mark_beg_offset, mark_end_offset;
	GtkTextIter mark_beg, mark_end;

	beg_offset = gtk_text_iter_get_offset(beg);
	end_offset = gtk_text_iter_get_offset(end);

	for (Nix = 0 ; Nix < marks->len ; Nix += 2) {
		gtk_text_buffer_get_iter_at_mark(buffer, &mark_beg, g_array_index(marks, GtkTextMark *, Nix));
		gtk_text_buffer_get_iter_at_mark(buffer, &mark_end, g_array_index(marks, GtkTextMark *, Nix + 1));

		mark_beg_offset = gtk_text_iter_get_offset(&mark_beg);
		mark_end_offset = gtk_text_iter_get_offset(&mark_end);

		if (mark_beg_offset != mark_end_offset)
			if (!(((mark_beg_offset <  beg_offset) && (mark_end_offset <= beg_offset)) ||
		      	((mark_beg_offset >= end_offset) && (mark_end_offset >  end_offset))))
				return TRUE;
	}

	return FALSE;
}

static void
buffer_delete_range(GtkTextBuffer *buffer, GtkTextIter *beg, GtkTextIter *end, GtkTextView *text)
{
	gboolean editable = TRUE;
	QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");
	gboolean has_editable_bits = get_has_editable_bits(buffer, beg, end, qti->marks);

	g_object_get(G_OBJECT(qti->uneditable_tag), "editable", &editable, NULL);

	if (editable) {
		g_signal_stop_emission_by_name(G_OBJECT(buffer), "delete-range");
		if (has_editable_bits) {
			g_object_set(G_OBJECT(qti->uneditable_tag), "editable", FALSE, NULL);
			gtk_text_buffer_delete_interactive(buffer, beg, end, TRUE);
		}
	}
}

static void
buffer_delete_range_after(GtkTextBuffer *buffer, GtkTextIter *beg, GtkTextIter *end, GtkTextView *text)
{
	g_object_set(G_OBJECT(((QueryTextInfo *)g_object_get_data(G_OBJECT(text), "query-text-info"))->uneditable_tag), "editable", TRUE, NULL);
}

static void
buffer_mark_set(GtkTextBuffer *buffer, GtkTextIter *itr, GtkTextMark *mark, GtkTextView *text)
{
	if (mark == gtk_text_buffer_get_insert(buffer)) {
		gboolean highlight_marks_valid = FALSE;
		QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");
		int Nix, min_idx = -1;
		int min_distance = G_MAXINT, distance;
		int insert_offset = gtk_text_iter_get_offset(itr);
		GtkTextIter mark_beg, mark_end;
		int mark_beg_offset, mark_end_offset;

		for (Nix = 0 ; Nix < qti->marks->len ; Nix += 2) {
			gtk_text_buffer_get_iter_at_mark(buffer, &mark_beg, g_array_index(qti->marks, GtkTextMark *, Nix));
			gtk_text_buffer_get_iter_at_mark(buffer, &mark_end, g_array_index(qti->marks, GtkTextMark *, Nix + 1));
			mark_beg_offset = gtk_text_iter_get_offset(&mark_beg);
			mark_end_offset = gtk_text_iter_get_offset(&mark_end);
			if (insert_offset >= mark_beg_offset && insert_offset <= mark_end_offset)
				return;
		}

		for (Nix = 0 ; Nix < qti->marks->len ; Nix++) {
			gtk_text_buffer_get_iter_at_mark(buffer, &mark_beg, g_array_index(qti->marks, GtkTextMark *, Nix));
			mark_beg_offset = gtk_text_iter_get_offset(&mark_beg);
			distance = ABS(insert_offset - mark_beg_offset);
			if (distance < min_distance) {
				min_distance = distance;
				min_idx = Nix;
			}
		}

		if (min_idx >= 0) {
			if (min_idx % 2) {
				if (min_idx - 1 >= 0) {
					highlight_marks_valid = TRUE;
					gtk_text_buffer_get_iter_at_mark(buffer, &mark_beg, g_array_index(qti->marks, GtkTextMark *, min_idx - 1));
					gtk_text_buffer_get_iter_at_mark(buffer, &mark_end, g_array_index(qti->marks, GtkTextMark *, min_idx));
				}
			}
			else {
				if (min_idx + 1 < qti->marks->len) {
					highlight_marks_valid = TRUE;
					gtk_text_buffer_get_iter_at_mark(buffer, &mark_beg, g_array_index(qti->marks, GtkTextMark *, min_idx));
					gtk_text_buffer_get_iter_at_mark(buffer, &mark_end, g_array_index(qti->marks, GtkTextMark *, min_idx + 1));
				}
			}
		}

		if (highlight_marks_valid) {
			g_print("buffer_mark_set: Hilighting %d --- %d\n", gtk_text_iter_get_offset(&mark_end), gtk_text_iter_get_offset(&mark_beg));
			gtk_text_buffer_select_range(buffer, &mark_end, &mark_beg);
		}
	}
}

static void
get_would_be_marks(GtkTextView *text, GtkMovementStep step, gint count, gboolean extend_selection, GtkTextIter *insert, GtkTextIter *sel_bound, GtkTextIter *would_be_insert, GtkTextIter *would_be_sel_bound)
{
	GtkTextBuffer *buffer = gtk_text_view_get_buffer(text);
	GtkTextMark *insert_mark    = gtk_text_buffer_get_insert(buffer),
	            *sel_bound_mark = gtk_text_buffer_get_selection_bound(buffer);

	gtk_text_buffer_get_iter_at_mark(buffer, insert, insert_mark);
	gtk_text_buffer_get_iter_at_mark(buffer, sel_bound, sel_bound_mark);

	g_signal_handlers_block_by_func(buffer, buffer_mark_set, text);
	GTK_TEXT_VIEW_GET_CLASS(text)->move_cursor(text, step, count, extend_selection);
	g_signal_handlers_unblock_by_func(buffer, buffer_mark_set, text);

	gtk_text_buffer_get_iter_at_mark(buffer, would_be_insert, insert_mark);
	gtk_text_buffer_get_iter_at_mark(buffer, would_be_sel_bound, sel_bound_mark);

	gtk_text_buffer_move_mark(buffer, insert_mark, insert);
	gtk_text_buffer_move_mark(buffer, sel_bound_mark, sel_bound);
}

static void
text_move_cursor(GtkTextView *text, GtkMovementStep step, gint count, gboolean extend_selection, gpointer null)
{
	GtkTextBuffer *buffer;
	int Nix, 
		insert_offset, insert_idx = -1, insert_interval_beg_offset, insert_interval_end_offset,
		selbnd_offset, selbnd_idx = -1, selbnd_interval_beg_offset, selbnd_interval_end_offset,
		insnxt_offset, insnxt_idx = -1, insnxt_interval_beg_offset, insnxt_interval_end_offset,
		selnxt_offset, selnxt_idx = -1, selnxt_interval_beg_offset, selnxt_interval_end_offset,
		beg_mark_offset, end_mark_offset;
	GtkTextIter insert, selbnd, insnxt, selnxt, beg_mark, end_mark;
	gboolean absorb_signal = FALSE;
	QueryTextInfo *qti = g_object_get_data(G_OBJECT(text), "query-text-info");

	g_object_get(G_OBJECT(text), "buffer", &buffer, NULL);

	get_would_be_marks(text, step, count, extend_selection, &insert, &selbnd, &insnxt, &selnxt);

	insert_offset = gtk_text_iter_get_offset(&insert);
	selbnd_offset = gtk_text_iter_get_offset(&selbnd);
	insnxt_offset = gtk_text_iter_get_offset(&insnxt);
	selnxt_offset = gtk_text_iter_get_offset(&selnxt);

//	g_print("text_move_cursor: insert: %d -> %d, selbound: %d -> %d\n",
//		insert_offset, insnxt_offset,
//		selbnd_offset, selnxt_offset);

	for (Nix = 0 ; Nix < qti->marks->len ; Nix += 2) {
		gtk_text_buffer_get_iter_at_mark(buffer, &beg_mark, g_array_index(qti->marks, GtkTextMark *, Nix));
		gtk_text_buffer_get_iter_at_mark(buffer, &end_mark, g_array_index(qti->marks, GtkTextMark *, Nix + 1));
		beg_mark_offset = gtk_text_iter_get_offset(&beg_mark);
		end_mark_offset = gtk_text_iter_get_offset(&end_mark);

//		g_print("text_move_cursor: Interval[%d]: %d --- %d\n", Nix, beg_mark_offset, end_mark_offset);

		if (insert_offset >= beg_mark_offset && insert_offset <= end_mark_offset) {
			insert_interval_beg_offset = beg_mark_offset;
			insert_interval_end_offset = end_mark_offset;
			insert_idx = Nix;
		}
		if (selbnd_offset >= beg_mark_offset && selbnd_offset <= end_mark_offset) {
			selbnd_interval_beg_offset = beg_mark_offset;
			selbnd_interval_end_offset = end_mark_offset;
			selbnd_idx = Nix;
		}
		if (insnxt_offset >= beg_mark_offset && insnxt_offset <= end_mark_offset) {
			insnxt_interval_beg_offset = beg_mark_offset;
			insnxt_interval_end_offset = end_mark_offset;
			insnxt_idx = Nix;
		}
		if (selnxt_offset >= beg_mark_offset && selnxt_offset <= end_mark_offset) {
			selnxt_interval_beg_offset = beg_mark_offset;
			selnxt_interval_end_offset = end_mark_offset;
			selnxt_idx = Nix;
		}
	}

	/* Go to the next interval when going one character at-a-time */
	if (insert_idx != -1 && insnxt_offset == insert_offset + 1 && insert_offset == insert_interval_end_offset) {
		if (insert_idx + 2 < qti->marks->len) {
			gtk_text_buffer_get_iter_at_mark(buffer, &beg_mark, g_array_index(qti->marks, GtkTextMark *, insert_idx + 2));
			gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_insert(buffer), &beg_mark);
			gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_selection_bound(buffer), &beg_mark);
		}
		absorb_signal = TRUE;
	}
	else
	if (insert_idx != -1 && insnxt_offset == insert_offset - 1 && insert_offset == insert_interval_beg_offset) {
		if (insert_idx - 1 >= 0) {
			gtk_text_buffer_get_iter_at_mark(buffer, &end_mark, g_array_index(qti->marks, GtkTextMark *, insert_idx - 1));
			gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_insert(buffer), &end_mark);
			gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_selection_bound(buffer), &end_mark);
		}
		absorb_signal = TRUE;
	}
	/* If about to hilight a range that includes portions outside the editable intervals, then stop at the edge of the starting
	 * interval */
	g_print("selnxt_idx = %d and insnxt_idx = %d\n", selnxt_idx, insnxt_idx);
	if (selnxt_idx != insnxt_idx) {
		if (selnxt_idx != -1) {
			if (insnxt_offset > selnxt_offset) {
				gtk_text_buffer_get_iter_at_mark(buffer, &end_mark, g_array_index(qti->marks, GtkTextMark *, selnxt_idx + 1));
				gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_insert(buffer), &end_mark);
			}
			else
			if (insnxt_offset < selnxt_offset) {
				gtk_text_buffer_get_iter_at_mark(buffer, &end_mark, g_array_index(qti->marks, GtkTextMark *, selnxt_idx));
				gtk_text_buffer_move_mark(buffer, gtk_text_buffer_get_insert(buffer), &end_mark);
			}
		}
		absorb_signal = TRUE;
	}

	if (absorb_signal)
		g_signal_stop_emission_by_name(text, "move-cursor");
}

GtkWidget *
query_text_new(const char *query)
{
	GtkWidget *text = NULL;
	if (g_utf8_validate(query, -1, NULL)) {
		GtkTextMark *mark;
		gboolean have_open = FALSE;
		GtkTextIter text_itr;
		char *itr;
		QueryTextInfo *qti = NULL;
		int Nix;
		GtkTextBuffer *buffer = g_object_new(GTK_TYPE_TEXT_BUFFER, "text", query, NULL);

		text = g_object_new(QUERY_TEXT_WIDGET_TYPE, "visible", TRUE, "buffer", buffer, NULL);
		qti = query_text_info_new();
		qti->highlight_tag = gtk_text_buffer_create_tag(buffer, NULL, NULL);
		qti->uneditable_tag = gtk_text_buffer_create_tag(buffer, NULL, NULL);
		g_object_set_data_full(G_OBJECT(text), "query-text-info", qti, (GDestroyNotify)query_text_info_free);

		g_signal_connect(text, "style-set", (GCallback)set_highlight_tag_clr, NULL);
		g_signal_connect(buffer, "changed", (GCallback)buffer_changed, text);
		g_signal_connect(buffer, "insert-text", (GCallback)buffer_insert_text, text);
		g_signal_connect(buffer, "delete-range", (GCallback)buffer_delete_range, text);
		g_signal_connect_after(buffer, "delete-range", (GCallback)buffer_delete_range_after, text);
		g_signal_connect(buffer, "mark-set", (GCallback)buffer_mark_set, text);
		g_signal_connect(text, "move-cursor", (GCallback)text_move_cursor, NULL);

		for (Nix = 0, itr = (char *)query ; *itr ; Nix++, itr = (char *)g_utf8_next_char(itr)) {
			if ((*itr) == '{') {
				if (have_open) {
					if (qti->marks->len > 0)
						gtk_text_buffer_delete_mark(buffer, g_array_index(qti->marks, GtkTextMark *, qti->marks->len - 1));
				}
				gtk_text_buffer_get_iter_at_offset(buffer, &text_itr, Nix);
				mark = gtk_text_buffer_create_mark(buffer, NULL, &text_itr, TRUE);
				g_array_append_val(qti->marks, mark);
				have_open = TRUE;
			}
			else
			if ((*itr) == '}') {
				if (have_open) {
					gtk_text_buffer_get_iter_at_offset(buffer, &text_itr, Nix + 1);
					mark = gtk_text_buffer_create_mark(buffer, NULL, &text_itr, FALSE);
					g_array_append_val(qti->marks, mark);
					have_open = FALSE;
				}
			}
		}

		buffer_changed(buffer, GTK_TEXT_VIEW(text));

		if (qti->marks->len > 1) {
			GtkTextIter text_itr_end;

			gtk_text_buffer_get_iter_at_mark(buffer, &text_itr, g_array_index(qti->marks, GtkTextMark *, 0));
			gtk_text_buffer_get_iter_at_mark(buffer, &text_itr_end, g_array_index(qti->marks, GtkTextMark *, 1));
			gtk_text_buffer_select_range(buffer, &text_itr_end, &text_itr);
		}
	}
	return text;
}

#ifdef RUN_STANDALONE_QUERY_TEXT
int
main(int argc, char **argv)
{
	GtkWidget *text = NULL;
	gtk_init(&argc, &argv);

	text = query_text_new("012{}345{}678");
	g_signal_connect(g_object_new(GTK_TYPE_WINDOW, "visible", TRUE, "child", text, NULL), "delete-event", (GCallback)gtk_main_quit, NULL);
	gtk_widget_grab_focus(text);

	gtk_main();

	return 0;
}
#endif /* RUN_STANDALONE_QUERY_TEXT */
