// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#ifndef PLANS_PLANT_TASKS_HPP
#define PLANS_PLANT_TASKS_HPP
#include <QMap>
#include <QList>
#include <QString>
#include <QDateTime>
#include <QXmlStreamReader>
#include <QXmlStreamWriter>
#if defined( PLANSPLANT_LIB_SHARED )
#define PLANSPLANT_CLASS Q_DECL_EXPORT
#elif defined( PLANSPLANT_STATIC )
#define PLANSPLANT_CLASS
#else // Use dynamic library by default.
#define PLANSPLANT_CLASS Q_DECL_IMPORT
#endif

namespace PlansPlant
{
  typedef int Time;
  class PLANSPLANT_CLASS Task //! \todo Add persistence from other libs.
  {
  public:
    typedef Task* Pointer;
    typedef QList<Pointer> List;
    typedef int ID;
    typedef QMap<ID,Pointer> Map;
    enum TimeUnits { Seconds = 1, Minutes = 60, Hours = Minutes*60, WorkDays = Hours*8, Days = Hours*24,
		     WorkWeeks = WorkDays*5, Weeks = Days*7, Months = Days*30, Quarters = Months*3, Years = int( Days*365.25 ) };
    static QString units_name( TimeUnits Units ); //!< \todo Save names in a dictionary.
    static QString units_short_name( TimeUnits Units ); //!< \todo Save names in a dictionary.
    static QDateTime add_time( const QDateTime& Start, Time Amount, TimeUnits Units );
    Task( const QString& Name0 = "NoName", Pointer SuperTask0 = 0, ID ID0 = 0 );
    virtual ~Task();
    ID id() const { return Ident; }
    const QString& name() const { return Name; }
    void name( const QString& NewName ) { Name = NewName; }
    Task* supertask() const { return SuperTask; }
    void supertask( Task* NewTask ); // No check for loops here - do it in the interface objects
    const List& subtasks() const { return SubTasks; }
    Task* subtask( int Index ) const { return Index < 0 ? 0 : ( Index < SubTasks.size() ? SubTasks[ Index ] : 0 ); }
    // Dependencies
    const List& needs() const { return Needs; }
    Task* needs( int Index ) const { return Index < 0 ? 0 : ( Index < Needs.size() ? Needs[ Index ] : 0 ); }
    const List& dependents() const { return Dependents; }
    Task* dependent( int Index ) const { return Index < 0 ? 0 : ( Index < Dependents.size() ? Dependents[ Index ] : 0 ); }
    // Add/remove dependecy: this task depends upon "Upon"
    void add_dependency( Task& Upon ); 
    void remove_dependency( Task& Upon );
    const QString& description() const { return Description; }
    void description( const QString& NewDescription ) { Description = NewDescription; }
    const QString& comment() const { return Comment; }
    void comment( const QString& NewComment ) { Comment = NewComment; }
    const QDateTime& plan_start() const { return PlanStart; } // When we plan to start this task
    void plan_start( const QDateTime& NewPlanStart ) { PlanStart = NewPlanStart; } // When we plan to start this task
    const QDateTime& plan_finish() const { return PlanFinish; } // When we plan to finish the task
    void plan_finish( const QDateTime& NewPlanFinish ) { PlanFinish = NewPlanFinish; } // When we plan to finish the task
    double completed() const { return Completed; } // 0 = not started 1 = completed
    void completed( double NewCompleted )
    {
      Completed = NewCompleted;
      if( Completed < 0 ) Completed = 0;
      else if( Completed > 1 ) Completed = 1;
    } // completed( double )
    bool blocked() const;
    Time estimation() const { return Estimation; } // Net time estimation in units (see below), <= 0 - no estimation set
    void estimation( Time NewEstimation ) { Estimation = NewEstimation; }
    TimeUnits estimation_units() const { return EstimationUnits; } // Units for net time estimation
    void estimation_units( TimeUnits NewUnits ) { EstimationUnits = NewUnits; }
    Time estimation_seconds() const { return Estimation * EstimationUnits; } // Net time estimation in seconds
    bool check_loop( Task& NewChild ) const;
    bool check_loop( const Task::List& Blockers, Task* NewChild = 0 ) const; // Check loop for child with different blockers
    bool move_subtask( int From, int To );
    bool move_dependency( int From, int To );
    void add_to_map( Map& TasksMap );
    void write( QXmlStreamWriter& Stream );
    void write_dependencies( QXmlStreamWriter& Stream );
    void load( QXmlStreamReader& Stream );
  protected:
    bool blockers_list_helper( Task::List& List, const Task::List& Source ) const;
    static ID new_id() { return ++LastID; }
    static ID LastID;
    ID Ident;
    Pointer SuperTask;
    List SubTasks;
    List Needs;
    List Dependents;
    QString Name;
    QString Description;
    QString Comment;
    QDateTime PlanStart; // When we plan to start this task
    QDateTime PlanFinish; // When we plan to finish the task
    double Completed; // 0 = not started 1 = completed
    Time Estimation; // Net time estimation in units
    TimeUnits EstimationUnits; // Units for net time estimation. Means seconds in the unit. 
  }; // Task
} // PlansPlant
#endif // PLANS_PLANT_TASKS_HPP
