#include <QTextCursor>
#include <QTextFrame>
#include "shared.h"
#include "epub.h"
#include "epubgenerator.h"
#include <QDebug>

EpubResourceReply::EpubResourceReply(QObject *parent, epub *nEpub, const QNetworkRequest &req) : QNetworkReply(parent)
{
    setRequest(req);
    setUrl(req.url());
    setOperation(QNetworkAccessManager::GetOperation);
    open(ReadOnly | Unbuffered);
    mEpub = nEpub;    
    requestUrl = req.url();
    QString name = requestUrl.toString().remove(dummyURL + "/"); 
    contentSize = epub_get_data(mEpub, name.toUtf8(), &buffer);
    offset = 0;
    setHeader(QNetworkRequest::ContentTypeHeader, QVariant("text/css; charset=UTF-8"));

    //For some reason emittin readeRead and finished was not enough..
    QMetaObject::invokeMethod(this, "metaDataChanged", Qt::QueuedConnection);
    QMetaObject::invokeMethod(this, "downloadProgress", Qt::QueuedConnection, Q_ARG(qint64, contentSize), Q_ARG(qint64, contentSize));
    QMetaObject::invokeMethod(this, "readyRead",Qt::QueuedConnection);
    QMetaObject::invokeMethod(this, "finished",Qt::QueuedConnection);
}

qint64 EpubResourceReply::readData(char *data, qint64 maxSize)
{
    if(offset < contentSize)
    {
        qint64 number = qMin(maxSize, contentSize - offset);
        memcpy(data, buffer + offset, number);
        offset += number;
        return number;
    }
    return -1;
}

qint64 EpubResourceReply::bytesAvailable() const
{
    return contentSize - offset;
}

void EpubResourceReply::abort()
{
    
}

EpubGenerator::EpubGenerator(const QString &filename, QObject *parent) : DocumentGenerator(filename, parent)
{
    mEpub = epub_open(qPrintable(filename), 0);
}

EpubGenerator::~EpubGenerator()
{
    if(mEpub)
        epub_close(mEpub);

    epub_cleanup();
}

QString EpubGenerator::getMetadata(enum epub_metadata type)
{
    int size;
    unsigned char **data = 0;

    data = epub_get_metadata(mEpub, type, &size);

    if(data){
        QString metadata = QString::fromUtf8((char*)data[0]);
        for(int i = 1; i < size; i++){
            metadata += ", ";
            metadata += QString::fromUtf8((char*)data[i]);
        }

        for(int i = 0; i < size; i++)
            free(data[i]);
        free(data);
        return metadata;
    }

    return QString();
}


QNetworkReply* EpubGenerator::createRequest(Operation op, const QNetworkRequest &req, QIODevice* outgoingData)
{ 
    if(req.url().scheme() != "ebook")
        return QNetworkAccessManager::createRequest(op, req, outgoingData);
    return new EpubResourceReply(this, mEpub, req);
}

void EpubGenerator::readMetaData()
{

    if(!mEpub){
        qDebug() << "Error while opening EPub file.";
        //TODO: emit error?
        return;
    }
    //Get document metadata
    emit setTitle(getMetadata(EPUB_TITLE));
    emit setAuthor(getMetadata(EPUB_CREATOR));
    emit setSubject(getMetadata(EPUB_SUBJECT));
    emit setPublisher(getMetadata(EPUB_PUBLISHER));
    emit setDescription(getMetadata(EPUB_DESCRIPTION));
    emit setPublishDate(getMetadata(EPUB_DATE));
    emit setCategory(getMetadata(EPUB_TYPE));
    emit setCopyright(getMetadata(EPUB_RIGHTS));


}

void EpubGenerator::readContents(){

    if(!mEpub){
        qDebug() << "Error while opening EPub file.";
        //TODO: emit error?
        return;
    }


    QHash<QString, QString> toc;
    struct titerator *tit;
    tit = epub_get_titerator(mEpub, TITERATOR_NAVMAP, 0);
    if(!tit)
        tit = epub_get_titerator(mEpub, TITERATOR_GUIDE, 0);

    if(tit){
        do{
            if(epub_tit_curr_valid(tit)){
                QString link = QString::fromUtf8(epub_tit_get_curr_link(tit));
                QString label = QString::fromUtf8(epub_tit_get_curr_label(tit));
                //TODO: Handle anchors. Removing anchors for now.
                if(link.contains("#"))
                    link = link.remove(QRegExp("\\#[a-z0-9]*"));
                toc.insert(link, label);

            }
        

        }while(epub_tit_next(tit));
    }

    epub_free_titerator(tit);

    struct eiterator *it;

    it = epub_get_iterator(mEpub, EITERATOR_SPINE, 0);
    do{
        if(epub_it_get_curr(it)){
            QString link = QString::fromUtf8(epub_it_get_curr_url(it));
            QString *content = new QString(QString::fromUtf8(epub_it_get_curr(it)));
            emit addSection(content, link);
            if(toc.contains(link)){
                emit addTOCItem(toc.value(link), link);
            }
        }
    } while(epub_it_get_next(it));

    epub_free_iterator(it);
}

char *EpubGenerator::readFile(const QString &filename, int &size){
    if(!mEpub)
        return 0;
    char *data;
    size = -1;
    size = epub_get_data(mEpub, filename.toUtf8(), &data);
    
    return data;
}
