#include "mainwindow.h"
#include "bookmark.h"
#include "addbookmarkdialog.h"
#include "addannotationdialog.h"
#include <QAction>
#include <QMenu>
#include <QMaemo5InformationBox>
#include "listviewdialog.h"

#include <QtGui/QX11Info>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

MainWindow::MainWindow(osso_context_t *context, QWidget *parent) : QMainWindow(parent)
{
    bookView = new BookView(this);
    setCentralWidget(bookView); 

    //Install eventFilter to bookView's viewport so we can intercept doubleclick event.
    bookView->viewport()->installEventFilter(this);

    
    connect(this, SIGNAL(volumeUpPressed()), bookView, SLOT(volumeUpPressed()));
    connect(this, SIGNAL(volumeDownPressed()), bookView, SLOT(volumeDownPressed()));

    toolbar = new QToolBar("toolbar", this);
    toolbar->setContextMenuPolicy(Qt::NoContextMenu);

    QAction *tocAction = toolbar->addAction("TOC");
    tocAction->setIcon(QIcon::fromTheme("notes_bullets"));
    connect(tocAction, SIGNAL(triggered()), this, SLOT(tocNavigation()));

    QAction *bookmarkAction = toolbar->addAction("bookmarkAction");
    bookmarkAction->setIcon(QIcon(":/bm_icon.png"));
    connect(bookmarkAction, SIGNAL(triggered()), this, SLOT(viewBookmarks()));

    QAction *annotationAction = toolbar->addAction("annotationAction");
    annotationAction->setIcon(QIcon(":/annotation_icon.png"));
    connect(annotationAction, SIGNAL(triggered()), this, SLOT(viewAnnotations()));

    setWindowFlags(windowFlags() | Qt::Window);
    setAttribute(Qt::WA_Maemo5StackedWindow);
    addToolBar(toolbar);
    grabZoomKeys(true);

    osso_context = context;
    brightnessTimer.setInterval(50000);
    connect(&brightnessTimer, SIGNAL(timeout()), this, SLOT(forceBacklightOn()));
    brightnessTimer.start();
    setWindowTitle("MeBook");
}

MainWindow::~MainWindow()
{
    grabZoomKeys(false);
}

void MainWindow::openBook(const QString &filename)
{
    Book *book = new Book(filename);
    openBook(book);
}

void MainWindow::openBook(Book *book)
{
    if(bookView)
        bookView->setBook(book);
    currentBook = book;
    setWindowTitle(book->getTitle());
    forceBacklightOn();
}

void MainWindow::setFont(const QString &family, int size)
{
    if(bookView)
        bookView->setFont(family, size);
}

void MainWindow::setColorMode(int mode)
{
    if(bookView)
        bookView->setNightMode(mode);
}

void MainWindow::enableBacklightForcing(bool toggle)
{
    if(toggle)
        brightnessTimer.start();
    else
        brightnessTimer.stop();
}

void MainWindow::setViewMode(BookViewMode mode, Qt::WidgetAttribute orientation)
{
    bookView->setViewMode(mode, orientation);
    if(mode == ScrollingMode){
        QAction *previousSectionAction = toolbar->addAction("previousSection");
        previousSectionAction->setIcon(QIcon::fromTheme("general_back"));
        connect(previousSectionAction, SIGNAL(triggered()), bookView, SLOT(previousSection()));

        QAction *nextSectionAction = toolbar->addAction("nextSection");
        nextSectionAction->setIcon(QIcon::fromTheme("general_forward"));
        connect(nextSectionAction, SIGNAL(triggered()), bookView, SLOT(nextSection()));
    }else{ 
        QAction *backAction = toolbar->addAction("back");
        backAction->setIcon(QIcon::fromTheme("browser_history"));
        connect(backAction, SIGNAL(triggered()), bookView, SLOT(backInHistory()));
    }
}

bool MainWindow::eventFilter(QObject *object, QEvent *event)
{
    if(event->type() == QEvent::MouseButtonDblClick)
    {
        if(isFullScreen()){
            showNormal();
            toolbar->show();
            return true;
        }

    } else if(event->type() == QEvent::MouseButtonPress){
        if(!isFullScreen()){
            showFullScreen();
            toolbar->hide();
            return true;
        }
    }

    return QObject::eventFilter(object, event);
}

void MainWindow::keyPressEvent(QKeyEvent *event)
{
    if(event->key() == Qt::Key_F7)
        emit volumeUpPressed();
    else if(event->key() == Qt::Key_F8)
        emit volumeDownPressed();
    QMainWindow::keyPressEvent(event);
}

void MainWindow::tocNavigation()
{
    QList<QString> toc = currentBook->getTOC();
    if(toc.isEmpty()){
        QMaemo5InformationBox::information(this, tr("This book doesn't have table of contents information."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    //TODO: Show notofication banner if no TOC
    QList<ListItem > list;
    for(QList<QString>::iterator iter = toc.begin(); iter != toc.end(); ++iter){
        ListItem item;
        item.text = *iter;
        list.push_back(item);
    }

    ListViewDialog *dialog = new ListViewDialog(this);
    ListModel *model = new ListModel(list);
    dialog->setModel(model);
    dialog->setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog->setWindowFlags(dialog->windowFlags() | Qt::Window);
    connect(dialog, SIGNAL(selectedIndex(int)), bookView, SLOT(goToTOCIndex(int)));
    dialog->show();

}

void MainWindow::viewBookmarks()
{
    QList<Bookmark> bm = currentBook->getBookmarks();
    if(bm.isEmpty()){
        QMaemo5InformationBox::information(this, tr("No bookmarks found. Press and hold to add a bookmark."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    QList<ListItem > list;
    for(QList<Bookmark>::iterator iter = bm.begin(); iter != bm.end(); ++iter){
        Bookmark bookmark = *iter;
        QPixmap ip(32, 32); 
        ip.fill(Qt::transparent);
        QPainter painter(&ip);
        painter.setRenderHint(QPainter::Antialiasing, true);
        painter.setPen(Qt::NoPen);
        painter.setBrush(bookmark.getColor());
        painter.drawRoundedRect(ip.rect(), 3, 3);
        ListItem item;
        item.text = bookmark.getName();
        QString sectionName = currentBook->getSectionName(bookmark.getSection());
        if(sectionName.isEmpty())
            sectionName = tr("Section") + " " + QString::number(bookmark.getSection());
        item.subText =  sectionName + ", " + bookmark.percentageAsString() + "%";
        item.icon = ip;

        list.push_back(item);
    }
    
    ListViewDialog *dialog = new ListViewDialog(this);
    ListModel *model = new ListModel(list);
    dialog->setModel(model);
    dialog->setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog->setWindowFlags(dialog->windowFlags() | Qt::Window);
    connect(dialog, SIGNAL(selectedIndex(int)), bookView, SLOT(goToBookmark(int)));
    dialog->show();
}


void MainWindow::viewAnnotations()
{
    QList<Annotation> a = currentBook->getAnnotations();
    if(a.isEmpty()){
        
        QMaemo5InformationBox::information(this, tr("No annotations found. Press and hold paragraph to add an annotation."), QMaemo5InformationBox::DefaultTimeout); 
        return;
    }
    QList<ListItem> list;
    for(QList<Annotation>::iterator iter = a.begin(); iter != a.end(); ++iter){
        Annotation annotation = *iter;
        QPixmap ip(32, 32); 
        ip.fill(Qt::transparent);
        QPainter painter(&ip);
        painter.setRenderHint(QPainter::Antialiasing, true);
        painter.setPen(Qt::NoPen);
        painter.setBrush(annotation.getColor());
        painter.drawRoundedRect(ip.rect(), 3, 3);

        ListItem item;
        item.icon = ip;
        item.text = annotation.getAnnotation();
        item.subText = currentBook->getSectionName(annotation.getSection());

        list.push_back(item);
    }
    ListViewDialog *dialog = new ListViewDialog(this);
    ListModel *model = new ListModel(list);
    dialog->setModel(model);
    connect(dialog, SIGNAL(selectedIndex(int)), bookView, SLOT(goToAnnotation(int)));
    dialog->setAttribute(Qt::WA_Maemo5StackedWindow);
    dialog->setWindowFlags(dialog->windowFlags() | Qt::Window);
    dialog->show();
}

void MainWindow::grabZoomKeys(bool grab)
{
    if(!winId()){
        qDebug("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if(!atom){
        qDebug("Unable to obtain _HILDON_ZOOM_KEY_ATOM.");
        return;
    }

    XChangeProperty(QX11Info::display(),
        winId(),
        atom,
        XA_INTEGER,
        32,
        PropModeReplace,
        reinterpret_cast<unsigned char*>(&val),
        1);

    grabKeyboard();
}

void MainWindow::closeEvent(QCloseEvent *event)
{
    bookView->saveProgress(); 
    hide();
    emit closing();
}


void MainWindow::forceBacklightOn()
{
    //Only force backligh if MainWindow is active.
    if(osso_context && isActiveWindow())
        osso_display_blanking_pause(osso_context);
}
