#ifndef BOOKVIEW_H
#define BOOKVIEW_H

#include <QGraphicsView>
#include <QThread>
#include <QTimer>
#include <QSlider>
#include <QGraphicsPixmapItem>
#include "libosso.h"
#include "book.h"
#include "shared.h"
#include "rendersettings.h"
#include "renderer.h"
#include "scrollablepage.h"
#include "bookprogressslider.h"

struct Section{
    int number;
    int contentHeight;
    int numPages;
    float pageMultiplier;
};

struct PositionInBook{
    Section section;
    int page;
    float multiplier;
};


//BookView is the base widget for the reading UI.
//It handles showing book content, navigating in book and adding bookmarks and annotations.
class BookView : public QGraphicsView
{
    Q_OBJECT
public:
    BookView(QWidget *parent = 0);
    ~BookView();
    void setBook(Book*);

    void setFont(const QString&, int);
    void setNightMode(bool);
    void setViewMode(BookViewMode, Qt::WidgetAttribute);

    //derprecated
    void addBookmark(QString, QColor);

protected:
    void mousePressEvent(QMouseEvent*);
    void mouseMoveEvent(QMouseEvent*);
    void mouseReleaseEvent(QMouseEvent*);
    void resizeEvent(QResizeEvent*);

private:
    Book *book;
    PositionInBook currentPosition;
    QList<PositionInBook> history;

    //Used when user swipes the screen in pagemode.
    QPoint swipeStart;

    BookViewMode viewMode;
    QSizeF pageSize;

    RenderSettings renderSettings;

    //Returns a pixmap that is displayed in pagemode when the renderer is rendering new content.
    QPixmap loadingPixmap();

    //TODO: replace PixmapWidget with QGraphicsPixmap?
    QGraphicsPixmapItem *pageWidgetCurrent;
    QGraphicsPixmapItem *pageWidgetNext;
    QGraphicsScene *scene;
    ScrollablePage *scrollablePage;
    BookProgressSlider *progressSlider;
    QGraphicsProxyWidget *sliderProxy;

    QHash<int, QHash<int, RenderTile*> > tiles;
    void deleteTiles();

    //Renderer takes html content and loads it to a QWebPage.
    //In pagemode it paginates it and renders it to QImages and emit's tileRendered(RenderTile*).
    //In scrollmode it emit's webPageLoaded(QWebPage*).
    Renderer *renderer;

    void nextPage();
    void previousPage();

    //Called in pagemode when we need to display current, next or previous page.
    QPixmap getCurrentPagePixmap();
    QPixmap getNextPagePixmap();
    QPixmap getPreviousPagePixmap();

    QTimer autoScrollTimer;
    const int autoScrollStep;
    int autoScrollBy; 

    QString annotationParagraph;

    bool isTap;
    int tapMove;



public slots:
    //Navigate to given section number.
    //multiplier is a number between 0 and 1. 0 is start of section and 1 end of section.
    //goToSection is used in all navigation excluding moving to the next or previous page.
    void goToSection(int, float multiplier = 0.0);
    void goToTOCIndex(int);
    void goToBookmark(int);
    void goToAnnotation(int);
    void backInHistory();
    void nextSection();
    void previousSection();
    void volumeUpPressed();
    void volumeDownPressed();
    void saveProgress();
    //Called by renderer when a new page has been rendered.
    void addTile(RenderTile*);
    //Called by renderer when section has been loaded and content's height is available.
    //Useful for calculating number of pages.
    void contentHeight(int);
    //Called by renderer when all pages have been rendered and the exact number of pages is known.
    void numberOfPages(int);
    //Called by renderer in scrollmode when the section html has been loaded to a QWebPage.
    void webPageLoaded(QWebPage*);
    void handleAutoScroll();
    void openContextMenu(const QPoint&);
    void addBookmark();
    void addAnnotation();
    //Connected to scrollablePage in scrollmode so we can calculate progress.
    void sectionScrolled(int);
    //Called when the user moves bookProgressSlider.
    void sliderMoved();

signals:
    //Connected to renderer.
    //Changes the current section, arguments are html content of section and section number.
    void requestRender(QString, int);

};

#endif
