#include "bookview.h"

#include <QDebug>
#include <QAction>
#include <QWebElement>
#include <QWebFrame>
#include <QMouseEvent>
#include <QWebSettings>
#include <QApplication>
#include <QDesktopWidget>
#include <QAbstractKineticScroller>
#include <QFontMetrics>
#include <QFontDatabase>
#include <math.h>
#include <QTime>
#include <QMenu>
#include <QMessageBox>
#include <QGraphicsProxyWidget>

#include "addbookmarkdialog.h"
#include "addannotationdialog.h"

#include "libosso.h"

//Q_DECLARE_METATYPE(RenderTile)

BookView::BookView(QWidget *parent) : QGraphicsView(parent), 
        autoScrollStep(1), 
        autoScrollBy(0), 
        isTap(false), 
        tapMove(0)
{
    setMouseTracking(false); 

    currentPosition.section.number = 0;
    currentPosition.page = 0;
    currentPosition.multiplier = 0.00;

    scene = new QGraphicsScene(this);
    setScene(scene);

    pageWidgetCurrent = NULL;
    pageWidgetNext = NULL;
    scrollablePage = NULL;
    

    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    
    renderer = new Renderer(this);

    //int type = qRegisterMetaType<RenderTile>("RenderTile");

    connect(renderer, SIGNAL(tileRendered(RenderTile*)), this, SLOT(addTile(RenderTile*)));
    connect(this, SIGNAL(requestRender(QString, int)), renderer, SLOT(addToQueue(QString, int)));

    renderer->setRenderSettings(&renderSettings);

  
    //Set PageMode as default.
    setViewMode(PageMode, Qt::WA_Maemo5PortraitOrientation);

    //Autoscroll every 200ms
    autoScrollTimer.setInterval(200);
    connect(&autoScrollTimer, SIGNAL(timeout()), this, SLOT(handleAutoScroll()));

    setContextMenuPolicy(Qt::CustomContextMenu);
    connect(this, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(openContextMenu(const QPoint&)));


    
}

BookView::~BookView()
{
    deleteTiles();
}

void BookView::setBook(Book *openBook)
{
    if(openBook){
        book = openBook;
        

        if(!book->isLoaded())
            book->load();

        renderer->setAnnotations(book->getAnnotations());
        renderer->setResourceManager(book->getResourceManager());

        progressSlider->setRange(0, book->getBookLength());

        if(book->getProgress().isValid()){
            currentPosition.multiplier = book->getProgress().getPercentage();
            currentPosition.section.number = book->getProgress().getSection();
        }else{ 
            currentPosition.multiplier = 0;
            currentPosition.section.number = 0;
        }  

        goToSection(currentPosition.section.number, currentPosition.multiplier);

    }
}

void BookView::deleteTiles()
{
    for(QHash< int, QHash<int, RenderTile*> >::iterator iter = tiles.begin(); iter != tiles.end(); ++iter)
    {
        qDeleteAll(*iter);
        (*iter).clear();
    }

    tiles.clear();
}

void BookView::goToSection(int section, float multiplier)
{ 
    history.push_back(currentPosition);
    if(viewMode == PageMode){
        if(tiles.contains(section)){
            currentPosition.section.number = section;
            currentPosition.multiplier = multiplier;
            currentPosition.page = ((float)currentPosition.section.numPages) * ((float)currentPosition.multiplier);
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else{
            QString content = book->getSection(section);
            currentPosition.section.number = section;
            currentPosition.multiplier = multiplier;
            currentPosition.page = -1;
            pageWidgetCurrent->setPixmap(loadingPixmap());
            deleteTiles();
            if(multiplier == 0)
                progressSlider->setValue(book->getSectionStart(currentPosition.section.number));
            emit requestRender(content, currentPosition.section.number);
        }
    }else{
        QString content = book->getSection(section);
        currentPosition.section.number = section;
        currentPosition.multiplier = multiplier;
        emit requestRender(content, currentPosition.section.number);
    }
}

void BookView::goToTOCIndex(int index)
{
    goToSection(book->getSectionNumberOfTOCIndex(index), 0);
}

void BookView::goToBookmark(int id)
{
    QList<Bookmark> bookmarks = book->getBookmarks();
    if(id < bookmarks.count() && id >= 0){
        Bookmark bm = bookmarks.at(id);
        goToSection(bm.getSection(), bm.getMultiplier());
    }else{
        qDebug() << "Can't find bookmark!";
    }
}

void BookView::goToAnnotation(int id)
{
    QList<Annotation> annotations = book->getAnnotations();
    if(id < annotations.count() && id >= 0){
        Annotation annotation = annotations[id];
        goToSection(annotation.getSection(), annotation.getPercentage());

    }
}

void BookView::backInHistory()
{
    PositionInBook last = history.last();
    goToSection(last.section.number, last.multiplier);
    history.removeLast();
}

void BookView::nextSection()
{
    if(currentPosition.section.number < (book->getNumberOfSections() - 1)){
        goToSection(currentPosition.section.number + 1, 0);
    } else{
        goToSection(0, 0);
    }
}

void BookView::previousSection()
{
    if(currentPosition.section.number > 0){
        goToSection(currentPosition.section.number - 1, 0);
    }
}

void BookView::setFont(const QString &family, int size)
{
    renderSettings.setFont(QFont(family, size));    
}

void BookView::setNightMode(bool toggle)
{
    if(toggle){
        renderSettings.setTextColor(QColor(Qt::white));
        renderSettings.setBackgroundColor(QColor(Qt::black));
        scene->setBackgroundBrush(QBrush(Qt::black));
    }
    else{
        renderSettings.setTextColor(QColor(Qt::black));
        renderSettings.setBackgroundColor(QColor(Qt::lightGray));
        scene->setBackgroundBrush(QBrush(Qt::lightGray));
    }
    
}

void BookView::setViewMode(BookViewMode mode, Qt::WidgetAttribute orientation)
{

    viewMode = mode;    

    int pageWidth = 0;
    int pageHeight = 0;
    int screenWidth = 0;
    int screenHeight = 0;
    if(orientation == Qt::WA_Maemo5PortraitOrientation){
        pageWidth = 480;
        pageHeight = 730;

        screenWidth = 480;
        screenHeight = 800;
    }
    else{
        pageWidth = 800;
        pageHeight = 410;

        screenWidth = 800;
        screenHeight = 480;
    }

    const int topMargin = 20;

    if(viewMode == PageMode){

        scene->clear();

        pageWidgetCurrent = new QGraphicsPixmapItem();
        pageWidgetNext = new QGraphicsPixmapItem();

        scene->addItem(pageWidgetCurrent);
        scene->addItem(pageWidgetNext);
        connect(renderer, SIGNAL(contentHeight(int)), this, SLOT(contentHeight(int)));
        connect(renderer, SIGNAL(numberOfPages(int)), this, SLOT(numberOfPages(int)));
        scene->setSceneRect(QRectF(0, 0, screenWidth, screenHeight));

        deleteTiles();

        pageWidgetCurrent->setPos(0, 0);
        pageWidgetNext->setPos(pageWidth, 0);

        pageWidgetCurrent->setOffset(0, topMargin);
        pageWidgetNext->setOffset(0, topMargin);

        pageWidgetCurrent->setPixmap(loadingPixmap());

        renderer->enablePagination(true);
        renderer->enablePixmapRendering(true);

        

    } else{
        scene->clear();

        scrollablePage = new ScrollablePage();
        scene->addItem(scrollablePage);
        connect(renderer, SIGNAL(webPageLoaded(QWebPage*)), this, SLOT(webPageLoaded(QWebPage*)));
        connect(scrollablePage, SIGNAL(positionChanged(int)), this, SLOT(sectionScrolled(int)));
        scrollablePage->setGeometry(0, 0, pageWidth, pageHeight);
        scene->setSceneRect(0, 0, screenWidth, screenHeight);
        renderer->enablePagination(false);
        renderer->enablePixmapRendering(false);

    }

    const int sliderHeight = 32;
    const int leftMargin = 5;

    progressSlider = new BookProgressSlider();
    progressSlider->setMinimumWidth(pageWidth - 10);
    progressSlider->setMaximum(100);
    progressSlider->setColors(renderSettings.getTextColor(), renderSettings.getBackgroundColor());

    connect(progressSlider, SIGNAL(sliderReleased()), this, SLOT(sliderMoved()));
    sliderProxy = scene->addWidget(progressSlider);
    sliderProxy->setPos(leftMargin, screenHeight - (sliderHeight + leftMargin));

    pageSize = QSizeF(pageWidth, pageHeight); 
    renderer->setPageSize(pageSize);
    renderSettings.setMaxImageSize(pageWidth-(2*leftMargin), pageHeight-(2*topMargin));

}

void BookView::addBookmark(QString name, QColor color)
{
    //TODO:Fix pagemode multiplier..
    float multiplier = currentPosition.multiplier;
    if(viewMode == PageMode) 
        multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));
    Bookmark bookmark(name, color, currentPosition.section.number, multiplier);
    book->addBookmark(bookmark);
}

void BookView::mousePressEvent(QMouseEvent *event)
{
    if(viewMode == PageMode && event->pos().y() < sliderProxy->pos().y())
        swipeStart = event->pos();    

    isTap = true;
    tapMove = 0;

    QGraphicsView::mousePressEvent(event);  
}

void BookView::mouseMoveEvent(QMouseEvent *event)
{
   if(viewMode == PageMode){
       if(!swipeStart.isNull() && event->pos().x() > 0){
            float moveX = event->pos().x() - swipeStart.x();
            swipeStart = event->pos();
            pageWidgetCurrent->moveBy(moveX, 0);

            //If moved enough user is not tapping
            tapMove += moveX;
            if((tapMove > 10 || tapMove < -10) && isTap){
                isTap = false;
                tapMove = 0;
            }

            int pageWidth = pageWidgetNext->boundingRect().width();
            //Calculate new position of pageWidgetNext.
            //It is either on the left or right side of pageWidgetCurrent
            float nextX = (pageWidgetCurrent->pos().x() >= 0) ? pageWidgetCurrent->pos().x() - pageWidth : pageWidgetCurrent->pos().x() + pageWidth;
            pageWidgetNext->setPos(nextX, pageWidgetNext->pos().y());

            if(nextX > 0){
                pageWidgetNext->setPixmap(getNextPagePixmap());
            }else{
                pageWidgetNext->setPixmap(getPreviousPagePixmap());
            }
       }
   }
    QGraphicsView::mouseMoveEvent(event);  
}

void BookView::mouseReleaseEvent(QMouseEvent *event)
{
    if(viewMode == PageMode){
        int pageWidth = pageSize.width();
        if(pageWidgetCurrent->x() < -(pageWidth/2)){
            pageWidgetCurrent->setPos(-pageWidth, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(0, pageWidgetNext->pos().y());
            //Switch pointers of current and next page widget
            QGraphicsPixmapItem *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            //Handle changing page logic.
            nextPage();
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }else if(pageWidgetCurrent->x() > (pageWidth/2) && (currentPosition.section.number != 0 || currentPosition.page != 0)){
            pageWidgetCurrent->setPos(pageWidth, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(0, pageWidgetNext->pos().y());
            //Switch pointers of current and next page widget
            QGraphicsPixmapItem *temp = pageWidgetCurrent;
            pageWidgetCurrent = pageWidgetNext;
            pageWidgetNext = temp;
            //Handle changing page logic.
            previousPage(); 
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
            
        }else{
            //If user didn't drag far enought snap current page back to position.
            pageWidgetCurrent->setPos(0, pageWidgetCurrent->pos().y());
            pageWidgetNext->setPos(pageWidth, pageWidgetNext->pos().y());
        }
        swipeStart = QPoint();
    }


    //Test if the user tapped a annotation
    if(isTap){
        QString annotationHit;
        bool hit = false;
        if(viewMode == PageMode){
            hit = renderer->hitAnnotation(event->pos() - QPoint(0, 20), annotationHit, tiles[currentPosition.section.number][currentPosition.page]);
        } else{
            QPoint t = QPoint(event->pos().x(), scrollablePage->getPosition() + event->pos().y());
            hit = renderer->hitAnnotation(t, annotationHit);
        }

        if(hit){
            Annotation *a = book->getAnnotation(annotationHit);
            if(a){
                QMessageBox annotationBox(this);
                annotationBox.setText(a->getAnnotation());
                annotationBox.setWindowTitle(tr("Annotation"));
                annotationBox.setStandardButtons(QMessageBox::Close);
                annotationBox.exec();
            }
        }
    }


    QGraphicsView::mouseReleaseEvent(event);
}

void BookView::resizeEvent(QResizeEvent *event)
{
    //Make sure pageWidget stays fully visible when returning to application from app switcher
/*    QPointF viewPos = mapToScene(pos());
    if(viewPos != QPointF(0, 0)){
        QPointF move = QPointF(0, 0) - viewPos;
        translate(viewPos.x(), viewPos.y());
    }
    ensureVisible(QRectF(0, 0, pageSize.width(), pageSize.height()));*/
    QGraphicsView::resizeEvent(event);
}


QPixmap BookView::loadingPixmap()
{
    //TODO: Replace with a more useful loading picture.
/*    QPixmap tile(20, 20);
    QPainter painter;
    painter.begin(&tile);
    painter.setPen(Qt::NoPen);
    painter.setBrush(Qt::lightGray);
    painter.drawRect(0, 0, 20, 20);
    painter.setBrush(Qt::white);
    painter.drawRect(0, 0, 10, 10);
    painter.drawRect(10, 10, 10, 10);
    painter.end();*/
    QPixmap pixmap(pageSize.toSize());
    pixmap.fill(renderSettings.getBackgroundColor());
/*    painter.begin(&pixmap);
    painter.drawTiledPixmap(0, 0, pageSize.width(), pageSize.height(), tile);
    painter.end();*/
    return pixmap;
}

void BookView::nextPage()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        history.push_back(currentPosition);
        if((currentPosition.page + 1) < currentPosition.section.numPages){
            currentPosition.page++;
            currentPosition.multiplier = (((float)currentPosition.page) / ((float)currentPosition.section.numPages));
            int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);
            progressSlider->setValue(value);
        }else{
            goToSection(currentPosition.section.number + 1, 0);
        }
    }


}

void BookView::previousPage()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        history.push_back(currentPosition);
        if(0 < currentPosition.page){
            currentPosition.page--;
            currentPosition.multiplier = (((float)currentPosition.page) / ((float)currentPosition.section.numPages));
            int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);

            progressSlider->setValue(value);

        }else{
            goToSection(currentPosition.section.number - 1, 1.0);
        }
    }

    if(currentPosition.section.number < 0)
        currentPosition.section.number = 0;

}

QPixmap BookView::getCurrentPagePixmap()
{
    if(tiles.contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page))
            return QPixmap::fromImage(pages[currentPosition.page]->getPixmap());
    }    
    
    return loadingPixmap();
}

QPixmap BookView::getNextPagePixmap()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page+1)){
            return QPixmap::fromImage(pages[currentPosition.page+1]->getPixmap());
        }else if(tiles.keys().contains(currentPosition.section.number+1)){
            pages = tiles[currentPosition.section.number+1];
            return QPixmap::fromImage(pages[0]->getPixmap());
        }
    }
    return loadingPixmap();
}

QPixmap BookView::getPreviousPagePixmap()
{
    if(tiles.keys().contains(currentPosition.section.number)){
        QHash<int, RenderTile*> pages = tiles[currentPosition.section.number];
        if(pages.contains(currentPosition.page-1)){
            return QPixmap::fromImage(pages[currentPosition.page-1]->getPixmap());
        }
    }
    return loadingPixmap();
}

void BookView::volumeUpPressed(){
   if(viewMode == PageMode){
        nextPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy += autoScrollStep; 
        autoScrollTimer.start();
   }
}

void BookView::volumeDownPressed()
{
   if(viewMode == PageMode){
        previousPage();
        pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
   }else{
        autoScrollBy -= autoScrollStep; 
        autoScrollTimer.start();
   }
}


void BookView::saveProgress()
{
    if(viewMode == PageMode){
        int section = currentPosition.section.number;
        float multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));

        BookProgress progress;
        progress.setProgress(section, multiplier);
        book->setProgress(progress);
    }else{
        int scrollPosition = scrollablePage->getPosition();
        int section = currentPosition.section.number; 
        float multiplier = ((float)scrollPosition / (float)currentPosition.section.contentHeight);

        BookProgress progress;
        progress.setProgress(section, multiplier);
        book->setProgress(progress);
    }

}

void BookView::addTile(RenderTile *tile)
{
    if(tiles.keys().contains(tile->getSection()))
        tiles[tile->getSection()].insert(tile->getID(), tile);
    else{
        QHash<int, RenderTile*> pages;
        pages.insert(tile->getID(), tile);
        tiles.insert(tile->getSection(), pages);
    }

    if(tile->getID() == currentPosition.page && tile->getSection() == currentPosition.section.number){
        pageWidgetCurrent->setPixmap(QPixmap::fromImage(tile->getPixmap()));
    }
    
    //Set first page as cover if no cover image found.
    if(tile->getSection() == 0 && tile->getID() == 0 && book->getCoverImage().isNull()){
        book->addCoverImage(QPixmap::fromImage(tile->getPixmap()));
    }
}

void BookView::contentHeight(int height)
{
    if(viewMode == PageMode){
        if(currentPosition.page == -1){
            currentPosition.page = ((float)height / pageSize.height()) * ((float)currentPosition.multiplier);
            currentPosition.section.contentHeight = height;
            currentPosition.section.numPages = (float)height / (float)pageSize.height();
        }
        
    }else{
        currentPosition.section.contentHeight = height;
        int scrollPosition = (float)height * ((float)currentPosition.multiplier);
        scrollablePage->setPosition(scrollPosition);
    }
        
}

void BookView::numberOfPages(int n)
{
    currentPosition.section.numPages = n;
    currentPosition.page = ((float)currentPosition.section.numPages) * ((float)currentPosition.multiplier);
    if(currentPosition.page >= currentPosition.section.numPages)
        currentPosition.page = currentPosition.section.numPages - 1;
    pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
}


void BookView::webPageLoaded(QWebPage *page){
    scrollablePage->setPage(page);
}

void BookView::handleAutoScroll()
{
    //Only scroll if MainWindow is active.
    if(parentWidget()->isActiveWindow()){
        if(autoScrollBy != 0)
            scrollablePage->scrollBy(autoScrollBy);
        else
            autoScrollTimer.stop();
    }
}

void BookView::openContextMenu(const QPoint &pos)
{
    QMenu *menu = new QMenu();
    menu->addAction(tr("Add bookmark"), this, SLOT(addBookmark()));
    menu->addAction(tr("Add annotation"), this, SLOT(addAnnotation()));

    QString temp;
    if(viewMode == PageMode){
        temp = renderer->getParagraphAt(pos - QPoint(0, 20), tiles[currentPosition.section.number][currentPosition.page]);
    } else{
        QPoint t = QPoint(pos.x(), scrollablePage->getPosition() + pos.y());
        temp = renderer->getParagraphAt(t);
    }

    if(temp.isEmpty()) temp = "Couldn't get paragraph!";

    annotationParagraph = temp;

    menu->exec(mapToGlobal(pos));
}

void BookView::addBookmark()
{
    AddBookmarkDialog dialog;
    if(dialog.exec() == QDialog::Accepted){
        addBookmark(dialog.getName(), dialog.getColor());        
    }
}

void BookView::addAnnotation()
{
    AddAnnotationDialog dialog;
    dialog.setParagraph(annotationParagraph);
    if(dialog.exec() == QDialog::Accepted){
        float multiplier = currentPosition.multiplier;
        if(viewMode == PageMode) 
            multiplier = ((float)currentPosition.page / ((float)currentPosition.section.numPages));
        Annotation annotation(currentPosition.section.number, multiplier, annotationParagraph, dialog.getSelectedText(), dialog.getAnnotation(), dialog.getColor());

        book->addAnnotation(annotation);
        renderer->setAnnotations(book->getAnnotations());
        if(viewMode == PageMode){
            renderer->renderTile(tiles[currentPosition.section.number][currentPosition.page]);
            pageWidgetCurrent->setPixmap(getCurrentPagePixmap());
        }
    }
}

void BookView::sectionScrolled(int pos)
{
    currentPosition.multiplier = ((float)pos / (float)currentPosition.section.contentHeight);
    int value = ((float)book->getSectionLength(currentPosition.section.number) * ((float)currentPosition.multiplier)) + book->getSectionStart(currentPosition.section.number);
    progressSlider->setValue(value);
}

void BookView::sliderMoved()
{
    int sliderPosition = progressSlider->sliderPosition();
    int section = book->getSectionAtPosition(sliderPosition);

    if(section != -1){
        currentPosition.multiplier = (((float)sliderPosition - (float)book->getSectionStart(section)) / (float)book->getSectionLength(section));
        goToSection(section, currentPosition.multiplier);
    }

}
