/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "application.h"
#include "mainwindow.h"

#ifdef Q_WS_MAEMO_5
    #include <QtGui/QX11Info>
    #include <X11/Xlib.h>
    #include <X11/Xatom.h>
    #include <QMaemo5InformationBox>

    #include "pickselectoraction.h"
#endif

#include <QDebug>
#include <QApplication>
#include <QDesktopWidget>
#include <QMenu>
#include <QMenuBar>
#include <QMessageBox>
#include <QPushButton>

#include "fullscreenexitbutton.h"
#include "waitingbutton.h"

#include "gotodlg.h"
#include "bookmark.h"
#include "bookmarksdlg.h"
#include "bookmarkdlg.h"

MainWindow::MainWindow(MainWindow *pParent) :
    QMainWindow(),
    m_webPage(this), m_pParent(pParent),
#ifdef Q_WS_MAEMO_5
    m_pOrientationSelector(NULL), m_pUserAgentSelector(NULL),
#endif
    m_pBookmarks(NULL), m_pFullscreenAction(NULL), m_pWaitingButton(NULL)
{
    setWindowTitle("Macuco");

#ifdef Q_WS_MAEMO_5
    FullScreenExitButton *pButt = new FullScreenExitButton(this);
    connect(pButt, SIGNAL(clicked()), this, SLOT(showNormal()));

    grabZoomKeys(true);

    setAttribute(Qt::WA_Maemo5StackedWindow);
#endif

#ifdef Q_WS_MAEMO_5
    QString storagePath = "/home/user/.macuco";
#else
    QString storagePath = QDesktopServices::storageLocation(QDesktopServices::CacheLocation) +
            QDir::separator() + "macuco";
            //QApplication::applicationDirPath() + "/cache";
#endif

    qDebug() << "cache location: " << storagePath;
    // QWebSettings::enablePersistentStorage(storagePath);
    QWebSettings::setIconDatabasePath(storagePath);

    // disk cache
    // QString storagePath = QWebSettings::offlineWebApplicationCachePath();
    qDebug() << "disk cache: " << storagePath;
    QNetworkDiskCache *pCache = new QNetworkDiskCache(this);
    pCache->setCacheDirectory(storagePath);
    m_webPage.networkAccessManager()->setCache(pCache);

    // m_webPage.setLinkDelegationPolicy(QWebPage::DelegateAllLinks);
    // m_webPage.settings()->setAttribute(QWebSettings::PluginsEnabled, true);
    // m_webPage.setForwardUnsupportedContent(true);
    m_webPage.settings()->setAttribute(QWebSettings::DeveloperExtrasEnabled, true);

    connect(m_webPage.networkAccessManager(), SIGNAL(finished(QNetworkReply*)),
        this, SLOT(finishedRequest(QNetworkReply*)));

    // connect(&m_webPage, SIGNAL(linkClicked(QUrl)), this, SLOT(linkClicked(QUrl)));
    connect(&m_webPage, SIGNAL(unsupportedContent(QNetworkReply*)), this, SLOT(unsupportedContent(QNetworkReply*)));
    connect(&m_webPage, SIGNAL(downloadRequested(QNetworkRequest)), this, SLOT(downloadRequested(QNetworkRequest)));

#ifdef Q_WS_MAEMO_5
    m_pWaitingButton = new WaitingButton(this);
#endif
}

void MainWindow::startUp() {
    Application &app(Application::Instance());

    m_pBookmarks = &app.m_bookmarks;
    m_zoomFactor = 1;
    // setFullScreen(app.m_fullScreen);
    setUserAgent( m_pParent ? m_pParent->m_webPage.isIphoneUserAgent() : app.m_fakeUserAgent );

    if ( m_pParent && m_pParent->width()<m_pParent->height() ) {
#ifdef Q_WS_MAEMO_5
        // portrait
        setAutoRotate(false);
        setLandscape(false);
#endif
    } else {
        // default orientation
        setOrientation(app.m_orientation);
    }

    /*
    if ( m_pParent ) {
        // copy settings from mainwindow
        m_zoomFactor = m_pParent->m_zoomFactor;
        //m_pBookmarks = m_pParent->m_pBookmarks;
        setFullScreen(m_pParent->isFullScreen());
        setUserAgent( m_pParent->m_webPage.isIphoneUserAgent() );

#ifdef Q_WS_MAEMO_5
        qDebug() << "New Window: " << m_pParent->isLandscape() << " - " << m_pParent->isAutoRotate();
        setAutoRotate(false);
        setLandscape(m_pParent->isLandscape());
#endif

    } else {

        m_pBookmarks = new QList<Bookmark>();

        // load settings
        loadSettings();
    }
    */

#ifdef Q_WS_X11
#ifndef Q_WS_MAEMO_5
    // meego?
    setGeometry(QRect(QPoint(0,0),QApplication::desktop()->size()));
#endif
#endif

    createMenu();
    updateTitle();
}

MainWindow::~MainWindow() {
    qDebug() << "~MainWindow, topMost: " << ( m_pParent==NULL );
}

void MainWindow::createMenu() {
    QMenu *pMenu = menuBar()->addMenu("&File");
    pMenu->addAction("Home", this, SLOT(goHome()));
    pMenu->addAction("Go To...", this, SLOT(on_actionGo_To_triggered()));
    pMenu->addAction("New Window", &Application::Instance(), SLOT(start()));

#ifndef Q_WS_MAEMO_5
    pMenu->addSeparator();
    pMenu->addAction("&Exit", &Application::Instance(), SLOT(quit()));
#endif

    pMenu = menuBar()->addMenu("&Bookmarks");
    pMenu->addAction("Bookmarks", this, SLOT(on_actionBookmarks_triggered()));
    pMenu->addAction("Add Bookmark", this, SLOT(on_actionAdd_Bookmark_triggered()));

#ifdef Q_WS_MAEMO_5
    menuBar()->addAction("Fullscreen", this, SLOT(fullscreen_triggered()));

    m_pOrientationSelector = new PickSelectorAction(this);
    m_pOrientationSelector->setText("Orientation");
    m_pOrientationSelector->items() << "Auto" << "Landscape" << "Portrait";
    m_pOrientationSelector->setSelectedIndex(
        isAutoRotate() ? 0 :
        (testAttribute(Qt::WA_Maemo5LandscapeOrientation) || !testAttribute(Qt::WA_Maemo5PortraitOrientation) ? 1 : 2)
    );
    connect(m_pOrientationSelector, SIGNAL(selectionChanged(int)), this, SLOT(orientation_selectionChanged(int)));
    menuBar()->addAction(m_pOrientationSelector);

    m_pUserAgentSelector = new PickSelectorAction(this);
    m_pUserAgentSelector->setText("User Agent");
    m_pUserAgentSelector->items() << "iPhone" << "Default";
    m_pUserAgentSelector->setSelectedIndex(m_webPage.isIphoneUserAgent()?0:1);
    connect(m_pUserAgentSelector, SIGNAL(selectionChanged(int)), this, SLOT(userAgent_selectionChanged(int)));
    menuBar()->addAction(m_pUserAgentSelector);

    if ( !QApplication::quitOnLastWindowClosed() ) {
        pMenu->addAction("Shutdown", &Application::Instance(), SLOT(quit()));
    }
#else
    pMenu = menuBar()->addMenu("&View");
    m_pFullscreenAction = pMenu->addAction("Fullscreen");
    m_pFullscreenAction->setCheckable(true);
    m_pFullscreenAction->setChecked(isFullScreen());
    connect(m_pFullscreenAction, SIGNAL(toggled(bool)), this, SLOT(setFullScreen(bool)));

    pMenu->addSeparator();

    QActionGroup *pActionGroup = new QActionGroup(pMenu);
    QAction *pAct1 = pActionGroup->addAction("iPhone User Agent");
    QAction *pAct2 = pActionGroup->addAction("Default User Agent");
    connect(pActionGroup, SIGNAL(selected(QAction*)), SLOT(onUserAgentChanged(QAction*)));
    pMenu->addActions(pActionGroup->actions());
    pAct1->setCheckable(true);
    pAct2->setCheckable(true);
    if ( m_webPage.isIphoneUserAgent()) pAct1->setChecked(true); else pAct2->setChecked(true);
#endif

#ifdef QT_DEBUG
    pMenu = menuBar()->addMenu("&About");
    pMenu->addAction("About Qt..", this, SLOT(aboutQt()));
#endif

    pMenu->addAction("About/FAQ/Donate", &Application::Instance(), SLOT(aboutMacuco()));
}

void MainWindow::aboutQt() {
    QMessageBox::aboutQt(this, "Qt");
}

void MainWindow::onUserAgentChanged(QAction *pAction) {
    qDebug() << pAction->text();

    bool fakeUserAgent = pAction->text().startsWith("iPhone");
    m_webPage.setIPhoneUserAgent(fakeUserAgent);
    updateTitle();

    // change default settings
    Application::Instance().m_fakeUserAgent = fakeUserAgent;
}

void MainWindow::userAgent_selectionChanged(int index) {
    m_webPage.setIPhoneUserAgent(index==0);

    // change default settings
    Application::Instance().m_fakeUserAgent = index==0;
}

void MainWindow::setOrientation(int index) {
#ifdef Q_WS_MAEMO_5
    switch ( index ) {
        case 0: // auto
            setAutoRotate(true);
            break;

        case 1: // landscape
            setAutoRotate(false);
            setLandscape(true);
            break;

        case 2: // portrait
            setAutoRotate(false);
            setLandscape(false);
            break;
    }
#endif
}

void MainWindow::orientation_selectionChanged(int index) {
    // update default orientation
    Application::Instance().m_orientation = index;

    setOrientation(index);
}

MainWindow* MainWindow::topMost() {
    if ( m_pParent ) return m_pParent->topMost();
    return this;
}

void MainWindow::showLoading() {
    m_errors.clear();

#ifdef Q_WS_MAEMO_5
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-windowstates.html
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
#else
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
        /*
    m_pErrorBox->hide();
    hideOverlayOnPage();

    m_pLoadingLabel->show();
    */
#endif

    if ( m_pWaitingButton ) m_pWaitingButton->showAndStart();
}

void MainWindow::hideLoading(bool success) {
    if ( m_pWaitingButton ) m_pWaitingButton->stopAndHide();

    setViewport();

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);

    // use notification
    // ...

    if ( !success ) {
        QString msg("<b>Loading Error</b><br/>");
        if ( m_errors.length()>0 ) msg+=m_errors.join("<br/>"); else msg+=m_currentUrl.toString();
        QMaemo5InformationBox::information(this,
            msg,
            QMaemo5InformationBox::DefaultTimeout
        );
        return;
    }
#else
    QApplication::restoreOverrideCursor();

    if ( !success ) QMessageBox::warning(this, "Loading Error", m_errors.join("\n"));
#endif
}

void MainWindow::changeUrl(QUrl url) {
    /*
    QIcon icon = m_webPage.settings()->iconForUrl(url);
    qDebug() << icon.isNull();
    */

    m_currentUrl = url;

    /*
#ifndef Q_WS_MAEMO_5
    m_pToolbar->getInputAddress()->setText( url.toString() );
#endif
    */
}

void MainWindow::visitUrl(QString url) {
    if ( url.isNull() || url.isEmpty() ) return;
    visitUrl(QUrl::fromUserInput(url));
}

void MainWindow::reload() {
    visitUrl(m_currentUrl);
}

void MainWindow::adjustSizes() {
    if ( !m_currentUrl.isEmpty() && m_webPage.isIphoneUserAgent() ) zoomFit();
}

void MainWindow::zoomFit() {
    if ( width() > height() ) {
        setZoom(qreal(width())/480.0);
    } else {
        setZoom(qreal(height())/480.0);
    }
}

void MainWindow::changeEvent(QEvent *pEvent) {
    QMainWindow::changeEvent(pEvent);

    Q_ASSERT(pEvent);

    if ( pEvent->type()== QEvent::WindowStateChange ) {
        if ( !m_pFullscreenAction ) return;
        m_pFullscreenAction->setChecked(isFullScreen());
    }


}

void MainWindow::resizeEvent(QResizeEvent* pEvent) {
    QWidget::resizeEvent(pEvent);

    setViewport();

    if ( m_pWaitingButton ) {
        // center button
        m_pWaitingButton->move(0,height()-m_pWaitingButton->height());
        // m_pWaitingButton->move((width()-m_pWaitingButton->width())/2,(height()-m_pWaitingButton->height())/2);
    }
}

void MainWindow::closeEvent(QCloseEvent *pEvent)
{
#ifdef Q_WS_MAEMO_5
    grabZoomKeys(false);
#endif

    if ( !m_pParent ) {
        // top most
        // saveSettings();
        m_webPage.saveCookies();
    }

    pEvent->accept();
}

void MainWindow::zoomIn() {
    setZoom(m_zoomFactor*1.15);
}

void MainWindow::zoomOut() {
    setZoom(m_zoomFactor/1.15);
}

void MainWindow::setZoom(qreal zoomFactor) {
    enableWebkitAutoSize();
    m_zoomFactor = zoomFactor;
    if ( m_zoomFactor>.90 && m_zoomFactor<1.1) m_zoomFactor =1;

    QSizeF sz = centralWidget()->contentsRect().size();
    sz /= m_zoomFactor;

    // qDebug() << "Preferred Content Size: " << sz;
    m_webPage.setPreferredContentsSize(sz.toSize());

    updateTitle();
}

void MainWindow::keyPressEvent(QKeyEvent *event)
{
    switch ( event->key() ) {
        case Qt::Key_F7:
            zoomIn();
            break;

        case Qt::Key_F8:
            zoomOut();
            break;

        case Qt::Key_B:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionBookmarks_triggered();
            break;

        case Qt::Key_G:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            on_actionGo_To_triggered();
            break;

        case Qt::Key_H:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            goHome();
            break;

        case Qt::Key_F:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setFullScreen(true);
            break;

        case Qt::Key_O:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            setZoom(1);
            break;

        case Qt::Key_I:
            if ( (event->modifiers()&Qt::ControlModifier) != Qt::ControlModifier  ) break;
            zoomFit();
            break;
    }
}

void MainWindow::enableWebkitAutoSize() {
    m_webPage.mainFrame()->evaluateJavaScript("document.body.style.WebkitTextSizeAdjust = 'auto'");
}

void MainWindow::goHome() {
    visitUrl("http://www.crochik.com/macuco");
}

void MainWindow::on_actionBookmarks_triggered()
{
    Q_ASSERT(m_pBookmarks);
    BookmarksDlg dlg((*m_pBookmarks), this);

#ifdef Q_WS_MAEMO_5
    if ( !isLandscape() ) dlg.setAttribute(Qt::WA_Maemo5PortraitOrientation);
    dlg.setAttribute(Qt::WA_Maemo5StackedWindow);
    dlg.setWindowFlags(Qt::Window);
#endif

    if ( dlg.exec()!= QDialog::Accepted || dlg.m_bookmark.isNull()) return;

    visitUrl(dlg.m_bookmark.url());
}

void MainWindow::setFullScreen(bool full/*=true*/) {
#ifndef Q_WS_MAEMO_5
    /*
    if ( m_pUI->actionFullscreen->isChecked()!=full ) {
        // change status of menu item and wait for event to be triggered
        m_pUI->actionFullscreen->setChecked(full);
        return;
    }
    */
#endif

    if (!full) {
        showNormal();
    } else {
        showFullScreen();
    }
}

bool MainWindow::isFullScreen() {
    return windowState() & Qt::WindowFullScreen;
}


void MainWindow::on_actionGo_To_triggered()
{
    GoToDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    if ( dlg.exec()!=QDialog::Accepted ) return;
    visitUrl(dlg.url());
}


void MainWindow::on_actionAdd_Bookmark_triggered()
{
    BookmarkDlg dlg(this);
    dlg.setUrl(m_currentUrl.toString());
    dlg.setTitle( windowTitle() );
    dlg.removeBtt()->setVisible(false);

    if ( dlg.exec()!= QDialog::Accepted ) return;

    Bookmark mark;
    mark.setTitle(dlg.title());
    mark.setUrl(dlg.url());

    Q_ASSERT(m_pBookmarks);
    m_pBookmarks->append(mark);
}

void MainWindow::on_actionLandscape_triggered() {
#ifdef Q_WS_MAEMO_5
    setLandscape(!isLandscape());
#endif
}

void MainWindow::on_actionAuto_Rotate_triggered() {
#ifdef Q_WS_MAEMO_5
    setAutoRotate(!isAutoRotate());
#endif
}

#ifdef Q_WS_MAEMO_5
void MainWindow::fullscreen_triggered() {
    setFullScreen(!isFullScreen());
}

void MainWindow::orientationChanged() {

    QRect screenGeometry = QApplication::desktop()->screenGeometry();

    qDebug() << "Orientation has changed: " << screenGeometry.width() <<", " << screenGeometry.height();

    // setLandscape( screenGeometry.width() > screenGeometry.height(), true );
}

void MainWindow::setLandscape(bool landscape/*=true*/) {
    if ( m_pOrientationSelector ) m_pOrientationSelector->setSelectedIndex(landscape?1:2);

    if ( landscape ) {
        qDebug() << ">> set Landscape";
        //setAttribute(Qt::WA_Maemo5PortraitOrientation, false);
        setAttribute(Qt::WA_Maemo5LandscapeOrientation, true); //

    } else {
        qDebug() << ">> set Portrait";
        //setAttribute(Qt::WA_Maemo5LandscapeOrientation, false);
        setAttribute(Qt::WA_Maemo5PortraitOrientation, true);

    }
}

void MainWindow::setAutoRotate(bool autoRotate /*=true*/) {
    // http://qt.nokia.com/doc/qt-maemo-4.6/maemo5-rotation.html
    if ( autoRotate ) {
        qDebug() << "set autorotate: on";
        setAttribute(Qt::WA_Maemo5AutoOrientation, true);
        if ( m_pOrientationSelector ) m_pOrientationSelector->setSelectedIndex(0);
        return;
    }

    qDebug() << "set autorotate: off";
    setAttribute(Qt::WA_Maemo5AutoOrientation, false);
}

bool MainWindow::isLandscape(){
    return width()>height();
    // return testAttribute(Qt::WA_Maemo5LandscapeOrientation) || !testAttribute(Qt::WA_Maemo5PortraitOrientation);
}

bool MainWindow::isAutoRotate(){
    return testAttribute(Qt::WA_Maemo5AutoOrientation);
}

void MainWindow::grabZoomKeys(bool grab)
{
    if (!winId()) {
        qWarning("Can't grab keys unless we have a window id");
        return;
    }

    unsigned long val = (grab) ? 1 : 0;
    Atom atom = XInternAtom(QX11Info::display(), "_HILDON_ZOOM_KEY_ATOM", False);
    if (!atom) {
        qWarning("Unable to obtain _HILDON_ZOOM_KEY_ATOM. This example will only work "
                 "on a Maemo 5 device!");
        return;
    }

    XChangeProperty (QX11Info::display(),
            winId(),
            atom,
            XA_INTEGER,
            32,
            PropModeReplace,
            reinterpret_cast<unsigned char *>(&val),
            1);

}
#endif

void MainWindow::setUserAgent(bool iphone/*=true*/) {
    m_webPage.setIPhoneUserAgent(iphone);

#ifdef Q_WS_MAEMO_5
    if ( m_pUserAgentSelector ) m_pUserAgentSelector->setSelectedIndex(iphone?0:1);
#endif
}

void MainWindow::linkClicked( const QUrl & url ) {
    qDebug() << "Link clicked: " << url.toString();
}

void MainWindow::unsupportedContent(QNetworkReply* pReply) {
    Q_ASSERT(pReply);

    qDebug() << "unsupportedContent: " << pReply->request().url(); // << pReply;
    QList<QByteArray> headerList = pReply->rawHeaderList();
    QString mimeType = pReply->header(QNetworkRequest::ContentTypeHeader).toString();
    qDebug() << ">> mime type: " << mimeType;
    foreach ( QByteArray array, headerList ) {
        QString header(array);
        QString value(pReply->rawHeader(array));
        qDebug() << ">> Header: " << header << value;
    }

}

void MainWindow::downloadRequested (const QNetworkRequest& request) {
}

void MainWindow::finishedRequest(QNetworkReply* reply)
{
    if (reply->error() == QNetworkReply::NoError) return;

    m_errors << reply->errorString();
    qDebug() << "Error(s): " << m_errors;
}

bool MainWindow::setViewport() {
    QMultiMap<QString, QString> meta = m_webPage.mainFrame()->metaData();
    QStringList viewport(meta.values("viewport"));
    if ( viewport.isEmpty() ) {
        // keep same zoom?
        setZoom(m_zoomFactor);
        return false;
    }

    qreal zoom = m_zoomFactor;
    QSize sz = centralWidget()->contentsRect().size();

    qDebug() << "ViewPort: " << viewport;
    QStringList list = viewport.join(", ").split(QRegExp("[;,]\\s*"), QString::SkipEmptyParts);
    foreach (QString str, list) {
        int index = str.indexOf('=');
        if ( index<0 ) continue;
        QString param = str.mid(0, index).trimmed();
        QString value = str.mid(index+1).trimmed();

        if ( !param.compare("width") ) {
            // qDebug()<<"Viewport.Width = " << value;
            if ( !value.compare("device-width") ) {
                // qDebug() << ">> zoom 1:1";
#ifdef Q_WS_MAEMO_5
                if ( m_webPage.isIphoneUserAgent() ) {
                    setZoom(sz.width()<sz.height() ? sz.width()/320.0 : sz.width()/640.0);
                } else {
                    setZoom(1.5);
                }
#else
                setZoom(1);
#endif
                return true;
            }

            bool ok =false;
            int w = value.toInt(&ok);
            if ( ok ) {
                zoom = (qreal)sz.width() / w;
                // qDebug() << ">> zoom: " << zoom;
                setZoom(zoom);
                return true;
            }
        }

        if ( !param.compare("initial-scale") ) {
            bool ok = false;
            double z = value.toDouble(&ok);
            if ( ok ) {
                qDebug() << ">> Initial Scale: " << z;
#ifdef Q_WS_MAEMO_5
                zoom = z * 1.5;
#else
                zoom = z;
#endif
            }
        }
    }

    setZoom(zoom);
    return false;
}


