/*
 * driver.c -  driver vkontakte of MSA program
 * This file is part of MSA program.
 *
 * Copyright (C) 2009 - Sergey Zakharov
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <libxml2/libxml/tree.h>

#include "driver.h"
#include "vkontakte.h"
#include "parser.h"
#include "profile.h"
#define XSL_URL DATADIR"/msa/drivers/vkontakte/make_url.xsl"

#ifdef TEST
#include "../test/CTF/src/ctf.h"
CTF_TEST_DATA_FILE_NAME_DECL;
#endif

int counter = 0;
int max_count = 0;
static int global_temp = 0;
int attempts = 0;
int captch = 0;

int init_event_manager(int time)
{
#ifdef TEST
    g_debug("TEST = %d",TEST );
#endif
g_debug("init_event_maneger start");
/** request **/
xmlChar* event_message = xmlCharStrdup("\
<TransitData id=\"0\" type=\"data\">\
<SourceID>db</SourceID>\
<TargetID>vkontakte</TargetID>\
<Content>\
<Request class=\"messages\" function = \"getNewInboxMessages\">\
<Params/>\
</Request>\
</Content>\
</TransitData>");

    char* action;
    if (time == 0) {
        action = g_strdup("remove");
    } else
        action = g_strdup("set");
    gchar* str_time = g_strdup_printf("%d", time);

    gchar* event_struct_str = g_strconcat("\
<TransitData type=\"signal\">\
<SourceID>"KERNEL_ID"</SourceID>\
<TargetID>", HANDLER_ID, "</TargetID>\
<Content>\
<Signal id=\"event_message_vk\">\
<StartTime>",str_time,"</StartTime>\
<Action>",action,"</Action>\
<RepeatTime>",str_time,"</RepeatTime>\
<Content/>\
</Signal>\
</Content>\
</TransitData>", NULL);
    xmlChar * event_struct = xmlCharStrdup(event_struct_str);
    g_free(event_struct_str);
    free(action);
    free(str_time);

    xmlNodePtr node;
    xmlNodePtr node_root;
    xmlDocPtr doc;
    xmlDocPtr doc_root;

    xmlChar* doc_char;
    int size;
    doc = xmlParseDoc(event_message);

    doc_root = xmlParseDoc(event_struct);
    node = xmlDocGetRootElement(doc);
    xmlUnlinkNode(node);
    xmlNodePtr oldroot = xmlDocSetRootElement(doc, NULL);
    if (oldroot != NULL)
        xmlFreeNode(oldroot);
    
    xmlXPathObject* Obj = xpath("//TransitData/Content/Signal/Content", doc_root);
    node_root = Obj->nodesetval->nodeTab[0];

    xmlAddChild(node_root, node);
    xmlDocDump(stdout, doc_root);

    
    xmlDocDumpMemory(doc_root, &doc_char, &size);
    g_debug("set_event");

    //set_events(doc_char);

    xmlXPathFreeObject(Obj);
    xmlFreeDoc(doc);
    xmlFreeDoc(doc_root);
    xmlFree(event_struct);
    xmlFree(event_message);
    return 0;
}


/**
 * Allocate memory
 *
 * @param ptr - pointer
 * @param size - size of data
* @return 0 on success or 1 otherwise.
 */
void* myrealloc(void* ptr, size_t size)
{
/* There might be a realloc() out there that doesn't like reallocing
NULL pointers, so we take care of it here */
    if(ptr)
        return realloc(ptr, size);
    else
        return malloc(size);
}

/**
 * Callbak for libcurl for receiving data
 *
 * @param ptr - pointer
 * @param size - size of chunk of data
 * @param nmemb - number of chunks of data
 * @param data - pointer to data
* @return 0 on success or 1 otherwise.
 */
 static size_t
WriteMemoryCallback(void* ptr, size_t size, size_t nmemb, void* data)
{
    size_t realsize = size * nmemb;
    struct MemoryStruct *mem = (struct MemoryStruct *)data;
    mem->memory = myrealloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }
    return realsize;
}

/**
 * Get captcha image from server
 *
 * @param captcha_sid - random number for generating captcha
 * @return image.
 */

static char* get_captcha(guint32 captcha_sid)
{
    g_debug("get_captcha START");
    gchar* url;
    char* img;

    url = g_strdup_printf("http://userapi.com/data?act=captcha&csid=%d", captcha_sid);

    if (http_get(url, info_proxy, info_port, NULL, &img) == 200){
        g_free(url);
        return img;
    }
    g_free(url);
    g_debug("get_captcha END");
    return NULL;
}

/**
 * Perform request
 *
 * @param reqXml - request
 * @param doc - response
 * @return 0 on success or 1 otherwise.
 */
int vk_req(xmlDocPtr reqXml, xmlDocPtr doc, const struct msa_module* info)
{
    captch = 0;
    int init_status = 0;
    xmlChar* class_val = NULL;
    xmlChar* function = NULL;

    struct ProfileData *cp = (struct ProfileData *)profile_get_by_id(info->id);
    g_debug("%s:%d:%s: id=%s, cp->login=%s", __FILE__, __LINE__, __FUNCTION__, info->id, cp->login);
    info_proxy = info->proxy;
    info_port = info->port;

    global_temp++;
    
    g_print("IN VK_REQ!!! %d\n", global_temp);
    counter = 0;
    xmlDocDump(stdout, reqXml);

//Prepare response
    xmlNodePtr rootNode = NULL, node = NULL, childNode = NULL;
    xmlDocPtr reqXml2 = NULL;
    rootNode = xmlNewDocNode(doc, NULL, BAD_CAST "Response", NULL);
    xmlNodePtr oldRoot = xmlDocSetRootElement(doc, rootNode);
    if (oldRoot != NULL)
        xmlFreeNode(oldRoot);

    class_val = get_req_class(reqXml);
    function = get_req_function(reqXml);
    xmlSetProp(rootNode, BAD_CAST CLASS, class_val);
    xmlFree(class_val);
    class_val = NULL;
    xmlSetProp(rootNode, BAD_CAST FUNCTION, function);
    
    node = xmlNewDocNode(doc, NULL, BAD_CAST PARAMS,  NULL);
    childNode = xmlAddChild(rootNode, node);
 
    g_debug("function = %s", function);

    xmlChar* temp = NULL;
    if (g_strcmp0((gchar *)function, GET_LIST_FRIENDS) == 0 || 
        g_strcmp0((gchar *)function, UPDATE_WALL_MESSAGES) == 0 ||
        g_strcmp0((gchar *)function, UPDATE_INBOX_MESSAGES) == 0 ||
        g_strcmp0((gchar *)function, UPDATE_OUTBOX_MESSAGES) == 0 ||
        g_strcmp0((gchar *)function, GET_PHOTOS) == 0) {

        node = xpath_get_node("//Request/Params/number[@name='from']", reqXml);
        if(node != NULL){
            g_debug("node not NULL");
            temp = xmlNodeGetContent(node);
            counter = atoi((gchar *)temp);
            g_free(temp); 
        } else 
            counter = 0;

        node = xpath_get_node("//Request/Params/number[@name='to']", reqXml);
        if(node != NULL){
            g_debug("node not NULL");
            temp = xmlNodeGetContent(node);
            max_count = atoi((gchar *)temp);
            g_free(temp);
        } else 
            max_count = MAX_COUNTER;
    }

    if (g_strcmp0((gchar *)function, GET_SETTINGS) == 0) {
        get_settings(doc, cp);
        xmlFree(function);
       return 0;
    }
    
    if(g_strcmp0((gchar *)function, SET_SETTINGS) == 0) {
        if (set_settings(reqXml, cp)) {
            info_message("404", doc, DRV_SETT_ERROR, 1);
        } else {
            info_message("410", doc, DRV_SETT_SAVED, 0);
        }
       xmlFree(function);
       return 0;
    }
    if (g_strcmp0((gchar *)function, CAPTCHA_MESSAGE) == 0) {
        xmlFree(function);
        captch = 1;

        xmlChar* captcha_code = NULL;
        node = xpath_get_node("//Request/Params/string[@name='text']", reqXml);
        if(node == NULL)
            return 1;
        captcha_code = xmlNodeGetContent(node);
        g_debug("captcha code %s", captcha_code);

        node = xpath_get_node("//Request/Params/string[@name='oldRequest']/Request/Params/string[@name='fccode']", reqXml);
        if(node == NULL)
            return 1;
        xmlNodeSetContent(node, captcha_code);

        node = xpath_get_node("//Request/Params/string[@name='oldRequest']/Request", reqXml);
	    if(node != NULL){
            childNode = xmlCopyNode(node,1);
            xmlUnlinkNode(childNode);
            reqXml2 = xmlNewDoc(BAD_CAST "1.0");
            xmlNodePtr oldChild = xmlDocSetRootElement(reqXml2, childNode);
            if (oldChild != NULL)
                xmlFreeNode(oldChild);
        } else {
            return 1;
          }
    }

    if(captch) {
        class_val = get_req_class(reqXml2);
        function = get_req_function(reqXml2);
        xmlSetProp(rootNode, BAD_CAST CLASS, class_val);
        xmlSetProp(rootNode, BAD_CAST FUNCTION, function);
    }
    if(class_val)
        xmlFree(class_val);
    if(function)
        xmlFree(function);
  
    init_status = initialization(cp);
    attempts = 0;
    if (init_status != 0) {
        if(init_status == -2) {
            //info_message("402", doc, DRV_AUTH_ERROR, 1);
                if(captch)
                    captcha_message(doc, reqXml2);
                else
                    captcha_message(doc, reqXml);
                return 0;
        }
        if (init_status == 2){
            info_message(DRV_NETW_ERROR_CODE, doc, DRV_NETW_ERROR, 1);
            return 0;
        }
        if(init_status == 1) {
            info_message(DRV_AUTH_ERROR_CODE, doc, DRV_AUTH_ERROR, 1);
            return 0;
        }
                 
    }

    if(captch)
        make_request(reqXml2, doc, cp);
    else
        make_request(reqXml, doc, cp);

    return 0;
}

/**
 * Perform request
 *
 * @param reqXml - request
 * @param doc - response
 * @param cp - current profile
* @return 0 on success or 1 otherwise.
 */
int make_request(xmlDocPtr reqXml, xmlDocPtr doc, struct ProfileData *cp)
{
    gchar* converted_friends = NULL;
    gchar* tmp = NULL;
    char* url;
    char* file;
    char* file_name;

    //gchar* class = get_req_class(reqXml);
    xmlChar* function = get_req_function(reqXml);
    int num = 0;
    int init_status = 0;
    xmlNodePtr node = NULL;
    xmlNodePtr node2 = NULL;

    url = make_url(reqXml, cp->remixmid, cp->remixsid, cp);
    
    if (url == NULL || strlen(url) < 1) {
        if (g_strcmp0((gchar *)function, GET_BINARY_DATA) == 0 ) {
            counter = -1;
            g_free(function);
            return 0;
        } else {
            info_message(DRV_REQ_ERROR_CODE, doc, DRV_REQ_ERROR, 1);
            g_free(function);
            return 0;
        }
    }
    int code = 0;
    if (g_strcmp0((gchar *)function, UPLOAD_PHOTO) == 0 ) {
        node = xpath_get_node("//Request/Params/string[@name='file']", reqXml);
        node2 = xpath_get_node("//Request/Params/string[@name='fileName']", reqXml);
        if (node != NULL && node2 != NULL){
            file = xmlNodeGetContent(node);
            file_name = xmlNodeGetContent(node2);
            code = http_upload_photo(url, info_proxy, info_port, NULL, &tmp, cp, file, file_name);
        } else {
            info_message(DRV_REQ_ERROR_CODE, doc, DRV_REQ_ERROR, 1);
            g_free(function);
            return 0;
        }
    }
    else
        code = http_get(url, info_proxy, info_port, NULL, &tmp);
    if (code == 200) {
        if (g_strcmp0((gchar *)function, GET_BINARY_DATA) == 0 ) {
            g_free(function);
            if (counter == -1) {
        	g_free(url);
                return 0;
            }
            parser_image(doc, tmp, url);
            g_free(url);
            g_free(tmp);
            counter++;
            make_request(reqXml, doc, cp);
            return 0;
        }
        g_free(url);
        switch (check(tmp)) {
            case 0:
                if (g_strcmp0((gchar *)function, SEND_MESSAGE) == 0 ||
                    g_strcmp0((gchar *)function, SEND_WALL_MESSAGE) == 0) {
                    info_message(DRV_MESS_ERROR_CODE, doc, DRV_MESS_ERROR, 1);
		    g_free(tmp);
                    xmlFree(function);
                    return 0;
                }
                if (g_strcmp0((gchar *)function, DELETE_MESSAGE) == 0) {
                    info_message(DRV_MESS_DEL_ERROR_CODE, doc, DRV_MESS_DEL_ERROR, 1);
                    g_free(tmp);
                    xmlFree(function);
                    return 0;
                }
            case 1:

                if (g_strcmp0((gchar *)function, GET_MY_PROFILE) == 0 || 
                    g_strcmp0((gchar *)function, UPDATE_PROFILE) == 0) {
                    if(parser_profile(doc, tmp, cp->remixmid, TRUE, cp) == 1){
                        info_message(DRV_RESP_ERROR_CODE, doc, DRV_RESP_ERROR, 1);
                    }
                    g_free(tmp);      	    
                }
                if (g_strcmp0((gchar *)function, GET_PROFILE_NO_IMAGE) == 0 ) {
                    if(parser_profile(doc, tmp, cp->remixmid, FALSE, cp) == 1){
                        info_message(DRV_RESP_ERROR_CODE, doc, DRV_RESP_ERROR, 1);
                    }
                    g_free(tmp);      	    
                }


                if (g_strcmp0((gchar *)function, SEND_MY_PROFILE) == 0) {
                    info_message(DRV_PROF_SAVED_CODE, doc, DRV_PROF_SAVED, 0);
                    g_free(tmp);
                    xmlFree(function);
                    return 0;
                }
                if (g_strcmp0((gchar *)function, GET_ALBUMS) == 0 ) {
                    if(parser_profile_albums(doc, tmp, cp->remixmid, cp) == 1){
                        info_message(DRV_RESP_ERROR_CODE, doc, DRV_RESP_ERROR, 1);
                    }
                    g_free(tmp);      	    
                }
                if (g_strcmp0((gchar *)function, GET_PHOTOS) == 0) {
                    num = parser_photos(tmp, doc, counter);
                    g_free(tmp);
                    if (num == PAGE_SIZE && (counter + PAGE_SIZE) < max_count ) {
                        counter += PAGE_SIZE;
                        make_request( reqXml, doc, cp);
                    } else {
                        xmlFree(function);
                        return 0;
                    }
                }


                if (g_strcmp0((gchar *)function, GET_LIST_FRIENDS) == 0) {
                    g_debug("GET_LIST_FRIENDS");
                    converted_friends = g_convert(tmp, -1, UTF8, CP1251, NULL, NULL, NULL);
                    g_free(tmp);
                    num = parser_friends(converted_friends, doc, counter);
                    g_free(converted_friends);
                    if (num == PAGE_SIZE && (counter + PAGE_SIZE) < max_count ) {
                        counter += PAGE_SIZE;
                        make_request( reqXml, doc, cp);
                    } else {
                        xmlFree(function);
                        return 0;
                    }
                }
                if (g_strcmp0((gchar *)function, UPDATE_INBOX_MESSAGES) == 0 || 
                    g_strcmp0((gchar *)function, UPDATE_WALL_MESSAGES) == 0 ||
                    g_strcmp0((gchar *)function, UPDATE_OUTBOX_MESSAGES) == 0) {
                    xmlChar* message_id = NULL;
                    xmlNodePtr node;
                    node = xpath_get_node("//Request/Params", reqXml);
                    if(node != NULL)
                        message_id = xmlGetProp(node, BAD_CAST "messageId");

                    num = parser_messages(tmp, doc, (gchar *)message_id, counter, max_count);
                    g_free(tmp);
                    if (num == PAGE_SIZE && (counter + PAGE_SIZE) < max_count) {
                        g_debug("num_messages = %d", num);
                        counter += PAGE_SIZE;
                        make_request( reqXml, doc, cp);
                    } else {
                        xmlFree(function);
                        return 0;
                    }
                }
                if (g_strcmp0((gchar *)function, GET_NEW_MESSAGES) == 0) {
                    g_debug("GET_NEW_MESSAGES");
                    num = parser_new_messages(tmp);
                    g_free(tmp);
                    g_debug("num = %d",num);
                    if (num > 0) {
                        url = g_strdup_printf("http://userapi.com/data?act=inbox&from=0&to=%d&id=%s&sid=%s",
                                num, cp->remixmid, cp->remixsid);
                        if(http_get(url, info_proxy, info_port, NULL, &tmp) == 200){
                            g_free(url);
                            num = parser_messages(tmp, doc, NULL, 0, 0);
                            g_free(tmp);
			    //read_message(doc, remixmid, remixsid);
                        } else {
                            info_message(DRV_NETW_ERROR_CODE, doc, DRV_NETW_ERROR, 1);
                        }

                    }
                }

                if (g_strcmp0((gchar *)function, SEND_MESSAGE) == 0) {
                    info_message(DRV_MESS_SEND_CODE, doc, DRV_MESS_SEND, 0);
                    g_free(tmp);
                    g_free(function);
                    return 0;
                }
                if (g_strcmp0((gchar *)function, DELETE_MESSAGE) == 0) {
                    info_message(DRV_MESS_DEL_CODE, doc, DRV_MESS_DEL, 0);
                    g_free(tmp);
                    g_free(function);
                    return 0;
                }
                if (g_strcmp0((gchar *)function, READ_MESSAGE) == 0) {
                    info_message(DRV_MESS_READ_CODE, doc, DRV_MESS_READ, 0);
                    g_free(tmp);
                    g_free(function);
                    return 0;
                }
                g_free(function);
                break;
            case 5:
                if (g_strcmp0((gchar *)function, DELETE_FRIEND) == 0) {
                    info_message(DRV_FRIEND_DEL_CODE, doc, DRV_FRIEND_DEL, 0);
                    g_free(tmp);
                    g_free(function);
                    return 0;
                }
                break;
            case -1:
                g_free(cp->remixsid);
                cp->remixsid = NULL;

                if (attempts == 0) {
                    attempts = 1;
                    init_status = initialization(cp);
                    if (init_status == 0)
                        make_request(reqXml, doc, cp);
                }
                else 		
                    info_message(DRV_AUTH_ERROR_CODE, doc, DRV_AUTH_ERROR, 1);
                g_free(function);
                g_free(tmp);
		break;
            case -2:
                captcha_message(doc, reqXml);
                g_free(tmp);
                g_free(function);
		break;
            case -3:
                g_debug("%s", tmp);
                info_message(DRV_ACCESS_ERROR_CODE, doc, DRV_ACCESS_ERROR, 1);
                g_free(function);
                g_free(tmp);
		break;
        }

    } else if (code == 302){
           if (g_strcmp0((gchar *)function, UPLOAD_PHOTO) == 0 ) {
               if(tmp == NULL)
                    info_message(DRV_PHOTO_UPLOAD_CODE, doc, DRV_PHOTO_UPLOAD, 0);
           }
        g_free(url);
        g_free(function);
        return 0;
    } else if (code >= 500){
        info_message(DRV_SERVER_ERROR_CODE, doc, DRV_SERVER_ERROR, 1);
        g_free(url);
        g_free(function);
        return 0;
    } else if (code == 400){
        info_message(DRV_NETW_ERROR_CODE, doc, DRV_NETW_ERROR, 1);
        g_free(url);
        g_free(function);
        return 0;
    }
    return 0;
}

/**
 * Perform authorization
 *
 * @param cp - current profile settings
 * @return 0 on success or 1 otherwise.
 */
int initialization(struct ProfileData *cp)
{
    g_debug("init START");

    int auth_status = 0;

    if(cp->remixmid == NULL || cp->remixsid == NULL ) {

        g_print("dr_vk: authorization\n");
        auth_status = authorize(cp);
        if (auth_status != 0 ) {
            g_debug("dr_vk: Could not get remixmid, remixsid\n"); 
            //if (auth_status == -2 ) {
            //    return -2;
            //}
            return auth_status;
        }
    }
    g_debug("init END");
    return 0;
}

/**
 * Get settings from current profile
 *
 * @param doc - response
 * @return 0 on success or 1 otherwise.
 */
int get_settings(xmlDocPtr doc, const struct ProfileData *cp)
{
    g_debug("get_settings START");

    xmlNodePtr rootNode, childNode;
    
    rootNode = xmlDocGetRootElement(doc);
    rootNode = rootNode->children;
    
    childNode = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST cp->login);
    xmlSetProp(childNode, BAD_CAST NAME, BAD_CAST LOGIN);
    childNode = xmlAddChild(rootNode, childNode);

    childNode = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST cp->password);
    xmlSetProp(childNode, BAD_CAST NAME, BAD_CAST PASSWORD);
    childNode = xmlAddChild(rootNode, childNode);

    g_debug("get_settings END");
   return 0;
}

/**
 * Set settings in to current profile
 *
 * @param reqXml - request
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
int set_settings (xmlDocPtr reqXml, struct ProfileData *cp)
{
    g_debug("set_settings START");

    gboolean checked = 1;
    int status = 0;
    xmlChar* name;
    xmlNodePtr rootNode;

    rootNode = xmlDocGetRootElement(reqXml);
    rootNode = rootNode->children;
    rootNode = rootNode->children;

    while(rootNode != NULL) {
        g_debug("%s", rootNode->name);
        name = xmlGetProp(rootNode, BAD_CAST NAME);
        if (g_strcmp0((gchar *)name, LOGIN) == 0) {
            if (cp->login != NULL)
            {
                g_free(cp->login);
                cp->login = NULL;
            }
            cp->login = (gchar *)xmlNodeGetContent(rootNode);
        }
        xmlFree(name);
        
        name = xmlGetProp(rootNode, BAD_CAST NAME);
        if (g_strcmp0((gchar *)name, PASSWORD) == 0) {
            if (cp->password != NULL)
            {
                g_free(cp->password);
                cp->password = NULL;
            }
	    cp->password = (gchar *)xmlNodeGetContent(rootNode);
        }
        xmlFree(name);
        
        rootNode = rootNode->next;
        if(!checked)
            status = 1;
    }

    if (cp->remixmid != NULL)
    {
        g_free(cp->remixmid);
        cp->remixmid = NULL;
    }

    if (cp->remixsid != NULL)
    {
        g_free(cp->remixsid);
        cp->remixsid = NULL;
    }
    
    g_debug("%s:%d:%s: login=%s password=%s", __FILE__, __LINE__, __FUNCTION__, cp->login, cp->password);

    return status;
}

/**
 * Authorization (get remixsid and remixmid for future requests)
 *
 * @param cp - current profile
 * @return 0 on success or 1 otherwise.
 */
int authorize(struct ProfileData *cp)
{
    char *body = NULL;
    char* mid = NULL, sid[MID_SIZE];
    char *tmp;
    gchar* post_data;
    int code = 0;
    struct curl_slist *cookies = NULL;


    if ( cp->login == NULL || cp->password == NULL) {
        g_debug("login or password = NULL");
        return 1;
    }

#ifdef TEST
    info_proxy = CTF_TEST_DATA("PROXY");
    info_port = atoi(CTF_TEST_DATA("PORT"));
#endif

    g_debug("%s:%d:%s(): Get mid", __FILE__, __LINE__, __FUNCTION__);
    post_data = g_strdup_printf("email=%s&pass=%s&expire=&vk=1&try_to_login=1", cp->login, cp->password);

    code = http_post("http://login.vk.com/?act=login", 
		     info_proxy, info_port, post_data, NULL, &body, &cookies);
    g_free(post_data);
    if (code == 200) {

        int i = 0;
	gchar** cookie;

        struct curl_slist *cookies_orig = cookies;
	while (cookies) {
		g_debug("%s:%d:%s: cookie=%s", __FILE__, __LINE__, __FUNCTION__, cookies->data);
	    cookie = g_strsplit(cookies->data,"\t", 20);
	    i = 0;
	    while(cookie[i] != NULL){
	        if(strcmp(cookie[i], "l") == 0) {
		    mid = g_strdup(cookie[i+1]);
                    g_debug("mid = %s", mid);
                }
	        i++;
            }
            g_strfreev(cookie);
            cookies = cookies->next;
        }

        curl_slist_free_all(cookies_orig);
	if ( mid != NULL ) {
            cp->remixmid = mid;
	} else {
            g_debug("mid not found");
	    //g_free(mid);
	    return 1;
	}


    //Get remixsid
        g_debug("BODY %s", body);
        if(body != NULL){
    	    tmp = g_strrstr(body, "name='s' value='");
    	    if (tmp != NULL) {
        	tmp += strlen("name='s' value='");
            }
            
            if(tmp == NULL){
                if (cp->remixmid != NULL) {
                    g_free(cp->remixmid);
                    cp->remixmid = NULL;
                }
                g_free(body);
	        g_debug("no sid");
	        return 1;
            } else {
	        g_debug("sid found");
	        sscanf(tmp, "%s", sid);
	        sid[strlen(sid)-1]='\0';
                if (cp->remixsid != NULL)
                    g_free(cp->remixsid);
                cp->remixsid = g_strdup(sid);
                g_free(body);
                return 0;
            }
        }
        if (cp->remixmid != NULL) {
            g_free(cp->remixmid);
            cp->remixmid = NULL;
        }
        return 1;
    } else {
        curl_slist_free_all(cookies);
        if (code == 400)
            return 2;
        else    
            return 1;
    }
}

/**
 * Get class of request
 *
 * @param reqXml - request
 * @return class of request.
 */
xmlChar* get_req_class(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if(reqXml != NULL)
        rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
        return xmlGetProp(rootNode, BAD_CAST CLASS);
    return NULL;
}

/**
 * Get function of request
 *
 * @param reqXml - request
 * @return function of request.
 */
xmlChar* get_req_function(xmlDocPtr reqXml)
{
    xmlNodePtr rootNode = NULL;
    if (reqXml != NULL)
	rootNode = xmlDocGetRootElement(reqXml);
    if (rootNode != NULL)
	return xmlGetProp(rootNode, BAD_CAST FUNCTION);
    return NULL;
}

/**
 * Set message status = read
 *
 * @param reqXml - request
 * @param remixmid - user id 
 * @param remixsid - session remixmid
 * @return 0 on success or 1 other.
 */
int read_message(xmlDocPtr reqXml, char* remixmid, char* remixsid)
{
    g_debug("read_message START");
    xmlXPathObject *Obj;
    xmlNodePtr node, node2;
    xmlChar *id;
    char *url;
    char *tmp;
    Obj = xpath("//Response/Params/array/struct", reqXml);
    if (Obj != NULL) {
	node = Obj->nodesetval->nodeTab[0];
	for(node2 = node; node2 != NULL; node2 = node2->next) {
            id = xmlGetProp(node2, BAD_CAST "id");
            url = g_strdup_printf("http://userapi.com/data?act=history&id=%s&read=%s_&sid=%s", remixmid, (gchar *)id + 2, remixsid);
            http_get(url, info_proxy, info_port, NULL, &tmp);
            g_free(url);
	    g_debug("%s", tmp);
	    xmlFree(id);
	}
	return 0;
    }
    return 1;
}
/**
 * Buil url for request
 *
 * @param reqXml - request
 * @param remixmid - authorization parameter
 * @param remixmid - authorization parameter
 * @return function of request.
 */
gchar* make_url(xmlDocPtr reqXml, char* remixmid, char* remixsid, struct ProfileData *cp)
{
    g_debug("make_url START");
    xmlChar* url;
    const char* params[13];
    xmlDocPtr urlXml;
    xmlNodePtr root;
    //xmlDocDump(stdout, reqXml);

    params[0] = g_strdup("MID");
    params[1] = g_strdup_printf("\"%s\"", remixmid);
    params[2] = g_strdup("SID");
    params[3] = g_strdup_printf("\"%s\"", remixsid);
    params[4] = g_strdup("COUNTER");
    params[5] = g_strdup_printf("%d", counter);
    params[6] = g_strdup("MAXCOUNTER");
    params[7] = g_strdup_printf("%d", max_count);
    params[8] = g_strdup("UPLOAD_URL");

    if (cp->upload_url != NULL)
        params[9] = g_strdup_printf("\"%s\"", cp->upload_url);
    else
        params[9] = g_strdup("url");

    if(captch == 1) {
        params[10] = g_strdup("CAPTCHA");
        params[11] = g_strdup("1");
        params[12] = NULL;
    } else 
        params[10] = NULL;

    xsltStylesheetPtr cur = NULL;

    cur = xsltParseStylesheetFile(BAD_CAST XSL_URL);

    if(cur != NULL && reqXml != NULL) {
        urlXml = xsltApplyStylesheet(cur, reqXml, params);
        //xmlDocDump(stdout, urlXml);
    } else 
	return NULL;

    if (urlXml == NULL)
        return NULL;

    root = xmlDocGetRootElement(urlXml);
    if (root == NULL)
        return NULL;

    url = xmlNodeGetContent(root);

    xsltFreeStylesheet(cur);
    xmlFreeDoc(urlXml);
    g_free((gpointer*)params[0]);
    g_free((gpointer*)params[1]);
    g_free((gpointer*)params[2]);
    g_free((gpointer*)params[3]);
    g_free((gpointer*)params[4]);
    g_free((gpointer*)params[5]);
    g_free((gpointer*)params[6]);
    g_free((gpointer*)params[7]);
    g_free((gpointer*)params[8]);
    g_free((gpointer*)params[9]);

    if(captch == 1) {
        g_free((gpointer*)params[10]);
        g_free((gpointer*)params[11]);
        captch = 0;
    }
    g_printf("%s", url);
    g_debug("make_url END");

    return (gchar *)url;
}


/**
 * Form INFO or ERROR message
 *
 * @param code - error code
 * @param doc - response (buffer for message)
 * @param message - text of message
 * @param type - 0 for info message or 1 for error message
* @return 0 on success or 1 otherwise.
 */
int info_message(char* code, xmlDocPtr doc, char* message, int type)
{
    xmlNodePtr rootNode, node, childNode;
    rootNode = xmlNewDocNode(doc, NULL, BAD_CAST "Response", NULL);
    xmlNodePtr oldroot = xmlDocSetRootElement(doc, rootNode);
    if (oldroot != NULL)
        xmlFreeNode(oldroot);

    xmlSetProp(rootNode, BAD_CAST CLASS, BAD_CAST "systemMessages");

    if (type == 1)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "errorMessage");
    else if (type == 0)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "infoMessage");
    else if (type == 2)
        xmlSetProp(rootNode, BAD_CAST FUNCTION, BAD_CAST "captchaMessage");
    else
        return 1;

    node = xmlNewDocNode(doc, NULL, BAD_CAST PARAMS, NULL);
    childNode = xmlAddChild(rootNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST DRIVER_NAME);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "moduleName");
    node = xmlAddChild(childNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST code);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "code");
    node = xmlAddChild(childNode, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST message);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "text");
    node = xmlAddChild(childNode, node);
    return 0;
}

int captcha_message(xmlDocPtr doc, xmlDocPtr reqXml)
{
    guint32 cpatcha_sid = g_random_int_range(1, 1000);
    char* captcha_image;
    captcha_image = get_captcha(cpatcha_sid);
    info_message("403", doc, DRV_CAPT_ERROR, 2);

    xmlNodePtr node_param, node, rootNode;

    node_param = xpath_get_node("//Response/Params", doc);

    node = xmlNewDocNode(doc, NULL, BAD_CAST IMG, BAD_CAST captcha_image);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "Img");
    node = xmlAddChild(node_param, node);

    node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, NULL);
    xmlSetProp(node, BAD_CAST NAME, BAD_CAST "oldRequest");
    node = xmlAddChild(node_param, node);

    rootNode = xmlDocGetRootElement(reqXml);
    xmlUnlinkNode(rootNode);

    xmlAddChild(node, rootNode);

    node_param = xpath_get_node("//Response/Params/string[@name='oldRequest']/Request/Params", doc);

    if (node_param == NULL)
        g_debug("node_param == NULL");
    else {
	gchar *str_sid = g_strdup_printf("%d", cpatcha_sid);

    node = xpath_get_node("//Response/Params/string[@name='oldRequest']/Request/Params/string[@name='fcsid']", doc);
        if (node != NULL) {
            xmlNodeSetContent(node, BAD_CAST str_sid);
        } else {
            node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, BAD_CAST str_sid);
            xmlSetProp(node, BAD_CAST NAME, BAD_CAST "fcsid");
            node = xmlAddChild(node_param, node);
        
            node = xmlNewDocNode(doc, NULL, BAD_CAST STRING, NULL);
            xmlSetProp(node, BAD_CAST NAME, BAD_CAST "fccode");
            node = xmlAddChild(node_param, node);
        }

        g_free(str_sid);
    }
    
    return 0;
}

/**
 * Perform HTTP GET request
 *
 * @param url - request
 * @param proxy - proxy address 
 * @param port - proxy port
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @return http server response code 
 */
int http_get(const char* url, const char* proxy, int port, char** header, gchar** body)
{
    CURL* c;
    long code;
    char* type2;
    struct MemoryStruct chunk, chunk2;
    
    chunk.memory = NULL; 
    chunk.size = 0;

    chunk2.memory = NULL; 
    chunk2.size = 0;

    c = curl_easy_init();
    g_debug("%s:%d:%s:url in http_get %s, proxy=%s, port=%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    curl_easy_setopt(c, CURLOPT_PROXY, proxy);
    curl_easy_setopt(c, CURLOPT_PROXYPORT, port);
    curl_easy_setopt(c, CURLOPT_TIMEOUT, 120);

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    g_debug("send request");

    if (curl_easy_perform(c) != 0) {
        g_warning("Can't perform the request\n");
        curl_easy_cleanup(c);
        return 400;
    }

    CURLcode ret = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    g_debug("CODE %ld", code);

    //type2 = g_malloc(sizeof(char)*50);
    ret = curl_easy_getinfo(c, CURLINFO_CONTENT_TYPE, &type2 );
    g_debug("TYPE %s", type2);

    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }

    g_print("HEADER  %s\n", chunk2.memory);
    if(chunk2.memory)
        free(chunk2.memory);

    if (body != NULL) {
        if (g_strcmp0(type2, "image/jpeg") == 0) {
            *body = g_base64_encode((guchar *)chunk.memory, chunk.size);
        } else {
    	    *body = g_strdup(chunk.memory);
        }
    }
    if(chunk.memory)
        free(chunk.memory);

    //g_print("CONTENT  %s\n", *body);

    //g_free(type2);
    curl_easy_cleanup(c);

    return code;
}

/**
 * Perform HTTP POST request
 *
 * @param url - request
 * @param proxy - proxy address
 * @param port - proxy port
 * @param post_data - data to send
 * @param header - buffer for header of response
 * @param body - buffer for body of response
 * @param cookies - list of cookies from server
 * @return http server response code
 */

int http_post(const char* url, const char* proxy, int port, char* post_data, char** header, char** body, struct curl_slist** cookies)
{
    CURL* c;
    long code;
    CURLcode res;
    struct MemoryStruct chunk,chunk2;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;


    c = curl_easy_init();
    g_debug("%s:%d:%s:url in http_post %s, proxy=%s:%d", __FILE__, __LINE__, __FUNCTION__, url, proxy, port);

    curl_easy_setopt(c, CURLOPT_URL, url);

    curl_easy_setopt(c, CURLOPT_PROXY, proxy);
    curl_easy_setopt(c, CURLOPT_PROXYPORT, port);
    curl_easy_setopt(c, CURLOPT_TIMEOUT, 120);


    curl_easy_setopt(c, CURLOPT_POST, 1);
    curl_easy_setopt(c, CURLOPT_POSTFIELDS, post_data);
    curl_easy_setopt(c, CURLOPT_POSTFIELDSIZE, strlen(post_data));
    curl_easy_setopt(c, CURLOPT_FOLLOWLOCATION, 1);

    curl_easy_setopt(c, CURLOPT_COOKIEFILE, "");

    curl_easy_setopt(c, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
    curl_easy_setopt(c, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

    curl_easy_setopt(c, CURLOPT_WRITEDATA, (void *)&chunk);
    curl_easy_setopt(c, CURLOPT_WRITEHEADER, (void *)&chunk2);
    g_debug("%s:%d:%s():send request", __FILE__, __LINE__, __FUNCTION__);
    
    CURLcode errornum = curl_easy_perform(c);
    if (errornum != 0) {
        g_warning("%s:%d:%s: Can't perform the request: %s", __FILE__, __LINE__, __FUNCTION__, curl_easy_strerror(errornum));
        curl_easy_cleanup(c);
        return 400;
    }
    
    res = curl_easy_getinfo(c, CURLINFO_COOKIELIST, &*cookies);
    if (res != CURLE_OK) {
        curl_easy_cleanup(c);
        curl_slist_free_all(*cookies);
        *cookies = NULL;
        return 400;
    }
    
    res = curl_easy_getinfo(c, CURLINFO_RESPONSE_CODE, &code );
    g_debug("CODE %ld", code);
    g_debug("HEADER\n  %s", chunk2.memory);
    //g_debug("BODY====\n  %s\n=====BODY_END======", chunk.memory);
    
    
    if (header != NULL) {
        *header = g_strdup(chunk2.memory);
    }
    
    if(chunk2.memory)
        g_free(chunk2.memory);
    
    if (body != NULL) {
        *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	    g_free(chunk.memory);
    }
    
    curl_easy_cleanup(c);
    return code;
}

int http_upload_photo(const char* url, const char* proxy, const int port, char** header, gchar** body, struct ProfileData *cp, char* file, char* file_name)
{
    CURL *curl;
    CURLcode res;
    long code;
    struct MemoryStruct chunk,chunk2;

    chunk.memory = NULL;
    chunk.size = 0;

    chunk2.memory = NULL;
    chunk2.size = 0;

    struct curl_httppost *formpost=NULL;
    struct curl_httppost *lastptr=NULL;
    struct curl_slist *headerlist=NULL;
    static const char buf[] = "Expect:";

    curl_global_init(CURL_GLOBAL_ALL);

    curl_formadd(&formpost,
                  &lastptr,
         CURLFORM_COPYNAME, "aid",
     CURLFORM_COPYCONTENTS, cp->aid,
              CURLFORM_END);

    curl_formadd(&formpost,
                  &lastptr,
         CURLFORM_COPYNAME, "hash",
     CURLFORM_COPYCONTENTS, cp->hash,
              CURLFORM_END);

     curl_formadd(&formpost,
                   &lastptr,
          CURLFORM_COPYNAME, "rhash",
      CURLFORM_COPYCONTENTS, cp->rhash,
              CURLFORM_END);

    /* Fill in the file upload field */
    curl_formadd(&formpost,
                  &lastptr,
         CURLFORM_COPYNAME, "file1",
             CURLFORM_FILE, file,
         CURLFORM_FILENAME, file_name,
      CURLFORM_CONTENTTYPE, "image/jpeg",
              CURLFORM_END);

    curl = curl_easy_init();
    /* initalize custom header list (stating that Expect: 100-continue is not
     *      wanted */
    headerlist = curl_slist_append(headerlist, buf);

    if(curl) {
     /* what URL that receives this POST */
        curl_easy_setopt(curl, CURLOPT_URL, url);

     /* only disable 100-continue header if explicitly requested */
        curl_easy_setopt(curl, CURLOPT_HTTPHEADER, headerlist);
        curl_easy_setopt(curl, CURLOPT_HTTPPOST, formpost);
        curl_easy_setopt(curl, CURLOPT_PROXY, proxy);
        curl_easy_setopt(curl, CURLOPT_PROXYPORT, port);
        curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);

        curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, WriteMemoryCallback);
        curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, WriteMemoryCallback);

        curl_easy_setopt(curl, CURLOPT_WRITEDATA, (void *)&chunk);
        curl_easy_setopt(curl, CURLOPT_WRITEHEADER, (void *)&chunk2);

        res = curl_easy_perform(curl);

        res = curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &code );
        g_debug("CODE %ld", code);
        g_debug("HEADER\n  %s", chunk2.memory);        
        
        if (header != NULL) {
            *header = g_strdup(chunk2.memory);
        }
        
        if(chunk2.memory)
            g_free(chunk2.memory);
        
        if (body != NULL) {
            *body = g_strdup(chunk.memory);
        if(chunk.memory) 
	        g_free(chunk.memory);
        }
        /* always cleanup */
        curl_easy_cleanup(curl);
        /* then cleanup the formpost chain */
        curl_formfree(formpost);
        /* free slist */
        curl_slist_free_all (headerlist);
    }
   return code;
}

xmlNodePtr xpath_get_node(char* path, xmlDocPtr doc)
{
    xmlXPathObject *Obj = NULL;
    xmlNodePtr node = NULL;
    Obj = xpath(path, doc);
    if(Obj == NULL)
        return NULL;
    if(!xmlXPathNodeSetIsEmpty(Obj->nodesetval))
        node = xmlXPathNodeSetItem(Obj->nodesetval, 0);
    xmlXPathFreeObject(Obj);
    return node;
}

/**
 * @param req - xpath req
 * @param doc - xmlDocPtr  
 * @return xmlXPathObject*  
**/
xmlXPathObject* xpath(gchar* req, xmlDocPtr doc)
{
	xmlXPathObject *Obj;
	xmlXPathContextPtr Ctx;
	
	g_debug("vk_xpath: %s", req);

	Ctx = xmlXPathNewContext(doc);
	
	if (Ctx == NULL) {
		g_debug("vk_xpath: error wrong request format!");
		return NULL;	
	}

	Obj = xmlXPathEvalExpression(BAD_CAST req, Ctx);

	g_debug("vk_xpath: end");

        xmlXPathFreeContext(Ctx);

	return Obj;

}

