/* 
 * Shared library add-on to iptables to add IDLETIMER support for IPv6.
 *
 * Copyright (C) 2005, 2006, 2007 Nokia Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License 
 * version 2 as published by the Free Software Foundation. 
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <stdio.h>
#include <netdb.h>
#include <string.h>
#include <stdlib.h>
#include <syslog.h>
#include <getopt.h>
#include <ip6tables.h>
#include <linux/netfilter_ipv6/ip6_tables.h>
#include <linux/netfilter_ipv6/ip6t_IDLETIMER.h>

#define IP6T_IDLETIMER_OPT_TIMEOUT		0x01

void _init(void);

static struct option opts[] = {
	{ .name = "timeout",	.has_arg = 1, .flag = 0, .val = 't' },
	{ .name = 0 }
};

/* Function which prints out usage message. */
static void help(void)
{
    printf("IDLETIMER v%s options:\n"
	   " --%s idle timeout\t\tTimeout until idle kevent is sent (in seconds)\n",
	   IPTABLES_VERSION, opts[0].name);
}

/* Initialize the target. */
static void init(struct ip6t_entry_target *t, unsigned int *nfcache)
{
	struct ip6t_idletimer_info *info =
		(struct ip6t_idletimer_info *) t->data;
	
	memset(info, 0, sizeof(*info));
	*nfcache |= NFC_UNKNOWN;
}

/* Function which parses command options; returns true if it
 * ate an option */
static int parse(int c, char **argv, int invert, unsigned int *flags,
		 const struct ip6t_entry *entry,
		 struct ip6t_entry_target **target)
{
	struct ip6t_idletimer_info *info = 
		(struct ip6t_idletimer_info *)(*target)->data;

	switch (c) {
	case 't':
		if (*flags & IP6T_IDLETIMER_OPT_TIMEOUT)
			exit_error(PARAMETER_PROBLEM, "Can't specify --%s twice",
				   opts[0].name);

		if (check_inverse(optarg, &invert, NULL, 0))
			exit_error(PARAMETER_PROBLEM, "Unexpected `!' after --%s",
				   opts[0].name);

                info->timeout = atoi(optarg);
		
		*flags |= IP6T_IDLETIMER_OPT_TIMEOUT;
		break;
		
	default:
		return 0;
	}
	return 1;
}

/* Final check; nothing. */
static void final_check(unsigned int flags)
{
	if (!(flags & IP6T_IDLETIMER_OPT_TIMEOUT))
		exit_error(PARAMETER_PROBLEM, "IDLETIMER: needs --%s",
			   opts[0].name);
}

static void print_it(const struct ip6t_ip6 *ip,
		     const struct ip6t_entry_target *target, char do_prefix)
{
	struct ip6t_idletimer_info *info = 
		(struct ip6t_idletimer_info *) target->data;

	if (do_prefix)
		printf("--");
	printf("%s %d ", opts[0].name, info->timeout);
}

/* Prints out the targinfo. */
static void print(const struct ip6t_ip6 *ip,
		  const struct ip6t_entry_target *target,
		  int numeric)
{
	print_it(ip, target, 0);
}

/* Saves the union ip6t_targinfo in parsable form to stdout. */
static void save(const struct ip6t_ip6 *ip, 
		 const struct ip6t_entry_target *target)
{
	print_it(ip, target, 1);
}

static struct ip6tables_target account = {
	.next          = NULL,
	.name          = "IDLETIMER",
	.version       = IPTABLES_VERSION,
	.size          = IP6T_ALIGN(sizeof(struct ip6t_idletimer_info)),
	.userspacesize = IP6T_ALIGN(sizeof(struct ip6t_idletimer_info)),
	.help          = &help,
	.init          = &init,
	.parse         = &parse,
	.final_check   = &final_check,
	.print         = &print,
	.save          = &save,
	.extra_opts    = opts
};

void _init(void)
{
	register_target6(&account);
}
