// -*- c++ -*-
// Generated by assa-genesis
//------------------------------------------------------------------------------
// $Id: inifile_test.cpp,v 1.11 2006/07/20 02:30:56 vlg Exp $
//------------------------------------------------------------------------------
//                            IniFile_Test.cpp
//------------------------------------------------------------------------------
//  Copyright (c) 2003,2005 by Vladislav Grinchenko 
//
//  This program is free software; you can redistribute it and/or 
//  modify it under the terms of the GNU General Public License   
//  as published by the Free Software Foundation; either version  
//  2 of the License, or (at your option) any later version.      
//------------------------------------------------------------------------------
//
// Date   : Sat Sep  6 22:10:05 2003
//
//------------------------------------------------------------------------------

static const char help_msg[]=
"                                                                            \n"
" NAME:                                                                      \n"
"                                                                            \n"
"   inifile_test                                                             \n"
"                                                                            \n"
" DESCRIPTION:                                                               \n"
"                                                                            \n"
"   Driver for IniFile class. Tests IniFile class interface.                 \n"
"                                                                            \n"
" USAGE:                                                                     \n"
"                                                                            \n"
"   shell>  inifile_test [OPTIONS]                                           \n"
"                                                                            \n"
" OPTIONS:                                                                   \n"
"                                                                            \n"
" -D, --log-file NAME     - Write debug to NAME file                         \n"
" -d, --log-stdout        - Write debug to standard output                   \n"
" -z, --log-size NUM      - Maximum size debug file can reach (dfl: is 10Mb) \n"
"                                                                            \n"
" -c, --log-level NUM     - Log verbosity                                    \n"
" -s, --with-log-server   - Redirect log messages to the log server          \n"
" -S, --log-server NAME   - Define assa-logd server address                  \n"
" -m, --mask MASK         - Mask (default: ALL = 0x7fffffff)                 \n"
"                                                                            \n"
" -h, --help              - Print this messag                                \n"
" -v, --version           - Print version number                            \n";
//------------------------------------------------------------------------------

#include <fstream>
#include <iostream>

#ifdef HAVE_CONFIG_H
#    include "config.h"
#endif
#include <string>
using std::string;

#include <assa/Assure.h>
#include <assa/GenServer.h>
#include <assa/Singleton.h>
#include <assa/IniFile.h>
using ASSA::IniFile;

enum { TEST = ASSA::USR1 };

/* Perform double-logging: to the file and STDOUT
 */
void log_status (const std::string& m_)
{
    std::cout << m_ << std::flush;
    DL((TEST," %s", m_.c_str ()));
}

//------------------------------------------------------------------------------
class IniFile_Test :
    public ASSA::GenServer,
    public ASSA::Singleton<IniFile_Test>
{
public:
    IniFile_Test ();
    ~IniFile_Test ();

    virtual void init_service ();
    virtual void process_events ();

private:
	int run_test_one   ();
	int run_test_two   ();
	int run_test_three ();
	int run_test_four  ();
	int run_test_five  ();

private:
	string m_test_fname;
	IniFile m_ini_file;
};
//------------------------------------------------------------------------------
/* Useful definitions */

#define INIFILE_TEST  IniFile_Test::get_instance()
#define REACTOR INIFILE_TEST->get_reactor()

// Static declarations mandated by Singleton class
ASSA_DECL_SINGLETON(IniFile_Test);

//------------------------------------------------------------------------------
IniFile_Test::
IniFile_Test () : 
	m_test_fname ("test-inifile.conf"),
	m_ini_file (m_test_fname)
{
    // ---Configuration---
    rm_opt ('f', "config-file"  );
    rm_opt ('n', "instance"     );
    rm_opt ('p', "port"         );

    // ---Process bookkeeping---
    rm_opt ('b', "daemon"       );
    rm_opt ('l', "pidfile"      );
    rm_opt ('L', "ommit-pidfile");

    /*---
     * By defauil disable all debugging
     *---*/
    // m_debug_mask = ASSA::APP | ASSA::ASSAERR;
    m_log_file = "IniFile_Test.log";

	// Enable if you want to see internal messages
    m_mask = TEST | ASSA::INIFILE | ASSA::ASSAERR;
}

IniFile_Test::
~IniFile_Test () 
{
	if (get_exit_value () == 0) {
		::unlink (m_test_fname.c_str ());
	}
}

void
IniFile_Test::
init_service ()
{
    trace("IniFile_Test::init_service");

	std::ofstream test_file;

	test_file.open (m_test_fname.c_str (), std::ios::out);

	if (!test_file) {
		set_exit_value (1);
		stop_service ();
		return;
	}

	test_file << "# This is the input test file for \n"
			  << "# inifile_test driver\n"
			  << "#\n"
			  << "\n"
			  << "[options]  \n"
			  << "gtk_options = --g-fatal-warnings --disable-crash-dialog  \n"
			  << "wavjoin_options=\n"
			  << "log_file=gwm.log\n"
			  << "  log_size  = 0\n"
			  << "\tmask =  \t 0x0\n"
			  << "log_stdout=false\n"
			  << "\n"
			  << "[History] \n"
			  << "0=/Segment25/Segment25.gwm\n"
			  << "1=/Segment26/Segment26.gwm\n"
			  << "2=/Segment27/Segment27.gwm\n"
			  << "\n"
			  << "[Default]\n"
			  << "jump=true\n"
			  << "\n";

	test_file.close ();
}

void
IniFile_Test::
process_events ()
{
	if (!service_is_active ()) {
		goto failed;
    }

	log_status ("= Running inifile_test Test =\n");

	if (run_test_one () < 0) {
		goto failed;
	}

	if (run_test_two () < 0) {
		goto failed;
	}

	if (run_test_three () < 0) {
		goto failed;
	}

	if (run_test_four () < 0) {
		goto failed;
	}

	if (run_test_five () < 0) {
		goto failed;
	}

	log_status ("inifile_test Passed\n");
	m_reactor.stopReactor ();
	return;

 failed:
	log_status ("inifile_test Failed!\n");
	set_exit_value (1);
    m_reactor.stopReactor ();
}

/*
  Load configuration file that was generated by init_service().
  Save configuration in cache to inifile01.conf.
  Reload it into another IniFile object, test01, and compare.
*/
int 
IniFile_Test::
run_test_one ()
{
	std::string alt_fname = "inifile01.conf";
	std::string err;

	log_status ("Testing load() ... ");

	if (m_ini_file.load () < 0) {
		err = " failed to load(\"" + m_test_fname + "\" file ...!\n";
		log_status (err);
		return -1;

	}
	m_ini_file.dump ();
	log_status ("ok\n");

	log_status ("Testing sync(file) ... ");
	if (m_ini_file.sync (alt_fname) < 0) {
		err = " failed to sync(\"" + alt_fname + "\") file ...!\n";
		log_status (err);
		return -1;
	}

	IniFile test01 (alt_fname);
	if (test01.load () < 0) {
		err = " failed to re-load(\"" + alt_fname + "\") !\n";
		log_status (err);
		return -1;
	}

	if (test01 != m_ini_file) {
		err = " failed to sync(\"" + alt_fname + "\") !\n";
		log_status (err);
		return -1;
	}

	log_status ("ok\n");
	return 0;
}

/*
  Check for expected values.
  Check for non-existing value.
*/
int 
IniFile_Test::
run_test_two ()
{
	log_status ("Testing get_value() ... ");

	if (m_ini_file.get_value ("options", "log_file") != "gwm.log") {
		log_status (" failed to get_value(\"options\", \"log_file\")\n");
		return -1;
	}

	if (m_ini_file.get_value ("History", "2") != "/Segment27/Segment27.gwm") {
		log_status (" failed to get_value(\"options\", \"log_file\")\n");
		return -1;
	}

	if (m_ini_file.get_value ("Bogus", "Not an option") == "failure") {
		log_status (" get_value(\"Bogus\", \"Not an option\") succeeded\n");
		return -1;
	}

	log_status ("ok\n");
	return 0;
}

/**
   Add new section and test to see if it was added.
   Fill up newly created section with values and test
   to see if they were added.
*/

int 
IniFile_Test::
run_test_three ()
{
	log_status ("Testing add_section() / set_pair() ... ");

	m_ini_file.add_section ("Interface");

	if (m_ini_file.find_section ("Interface") == m_ini_file.sect_end ()) {
		log_status (" failed to find_section (\"Interface\") ...!\n");
		return -1;

	}

	if (m_ini_file.set_pair ("Interface", 
							 IniFile::tuple_type ("zoom", "true")) < 0) 
	{
		log_status (" failed to set_pair(\"Interface\","
					"(\"zoom\",\"true\"))!\n");
		return -1;
	}
		
	if (m_ini_file.set_pair ("Interface", 
							 IniFile::tuple_type ("profile", "default")) < 0) 
	{
		log_status (" failed to set_pair(\"Interface\", "
					"(\"profile\",\"default\"))!\n");
		return -1;
	}
	
	if (m_ini_file.get_value ("Interface", "zoom") != "true") {
		log_status (" failed to get_value (\"Interface\", \"zoom\") ...!\n");
		return -1;
	}
	
	if (m_ini_file.get_value ("Interface", "profile") != "default") {
		log_status (" failed to get_value (\"Interface\", \"profile\")!\n");
		return -1;
	}
	
	log_status ("ok\n");
	return 0;
}

/**
   Traverse through the list of sections and print out section names.
   Traverse through a section and print out its name/value pairs.
*/
int 
IniFile_Test::
run_test_four ()
{
	log_status ("Testing iteration through list of sections  ...\n");

	int count = 0;
	std::ostringstream msg;

	IniFile::const_config_iterator i = m_ini_file.sect_begin ();

	while (i != m_ini_file.sect_end ()) {
		msg << "    " << (*i).first.c_str () << "\n";
		log_status (msg.str ().c_str ());
		i++, count++;
		msg.str (string ()), msg.clear ();
	}
	msg.str (string ()), msg.clear ();

	if (count != m_ini_file.size ()) {
		msg << " failed! # of sections " << count << " != " 
			<< m_ini_file.size () << "\n" << std::ends;
		log_status (msg.str ().c_str ());
		return -1;
	}
	msg.str (string ()), msg.clear ();

	log_status ("ok\n");

	log_status ("Testing iteration through a section  ...\n");
	count = 0;
	IniFile::const_config_iterator j = m_ini_file.find_section ("History");

	if (j != m_ini_file.sect_end ()) {
		IniFile::const_tuple_iterator k = (*j).second.begin ();
		while (k != (*j).second.end ()) {
			msg << "    " << (*k).first.c_str () << "=" << (*k).second.c_str ()
				<< "\n";
			log_status (msg.str ());
			k++, count++;
			msg.str (string ()), msg.clear ();
		}
	}
	msg.str (string ()), msg.clear ();

	if (count != 3) {
		msg << " failed! # of name/value pairs " << count << " != 3\n";
		log_status (msg.str ());
		return -1;
	}
	log_status ("ok\n");

	return 0;
}

/** 
	Remove section and test that it has been deleted.
	Clean up the entire configuration cache.
*/
int 
IniFile_Test::
run_test_five ()
{

	log_status ("Testing drop_pair() ... ");

	if (m_ini_file.drop_pair ("Interface", "profile") < 0) {
		log_status (" failed to drop_pair (\"Interface\", \"profile\")!\n");
		return -1;
	}
	if (m_ini_file.get_value ("Interface", "profile") != "") {
		log_status (" failed to get_value (\"Interface\",\"profile\")!\n");
		return -1;
	}
	log_status ("ok\n");

	log_status ("Testing drop_section() ... ");
	if (m_ini_file.drop_section ("Interface") < 0) {
		log_status (" failed to drop_section (\"Interface\")!\n");
		return -1;
	}
	if (m_ini_file.find_section ("Interface") != m_ini_file.sect_end ()) {
		log_status (" failed to remove \"Interface\" section!\n");
		return -1;
	}
	log_status ("ok\n");

	log_status ("Testing drop_all() ... ");
	m_ini_file.drop_all ();
	if (m_ini_file.size () != 0) {
		log_status (" failed to drop_all() !\n");
		return -1;
	}
	log_status ("ok\n");

	return 0;
}

int
main (int argc, char* argv[])
{
    static const char release[] = "1.0";
    int patch_level = 0;

    INIFILE_TEST->set_version (release, patch_level);
    INIFILE_TEST->set_author  ("Vladislav Grinchenko");
    INIFILE_TEST->set_flags   (ASSA::GenServer::RMLOG);

    INIFILE_TEST->init (&argc, argv, help_msg);
 
    INIFILE_TEST->init_service ();
    INIFILE_TEST->process_events ();

    return INIFILE_TEST->get_exit_value ();
}


