/*************************************************************************************                                                          
 *  Copyright (C) 2009 by Aleix Pol <aleixpol@kde.org>                               *                                                          
 *                                                                                   *                                                          
 *  This program is free software; you can redistribute it and/or                    *                                                          
 *  modify it under the terms of the GNU General Public License                      *                                                          
 *  as published by the Free Software Foundation; either version 2                   *                                                          
 *  of the License, or (at your option) any later version.                           *                                                          
 *                                                                                   *                                                          
 *  This program is distributed in the hope that it will be useful,                  *                                                          
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of                   *                                                          
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                    *                                                          
 *  GNU General Public License for more details.                                     *                                                          
 *                                                                                   *                                                          
 *  You should have received a copy of the GNU General Public License                *                                                          
 *  along with this program; if not, write to the Free Software                      *                                                          
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA   *                                                          
 *************************************************************************************/

#include "tasksview.h"

#include <QHeaderView>
#include <QKeyEvent>
#include <QDebug>
#include <QMenu>
#include "tasksmodel.h"

TasksView::TasksView(QWidget* parent) : QTreeView(parent)
{
	header()->hide();
	header()->setDefaultSectionSize(10);
	header()->setSortIndicatorShown(false);
	setRootIsDecorated(false);
	setSortingEnabled(true);
	
	connect(itemDelegate(), SIGNAL(closeEditor(QWidget*,QAbstractItemDelegate::EndEditHint)),
							SLOT(editorClosed(QWidget*,QAbstractItemDelegate::EndEditHint)));
	connect(this, SIGNAL(customContextMenuRequested(QPoint)), SLOT(customContextMenuRequested(QPoint)));
	
	setContextMenuPolicy(Qt::CustomContextMenu);
}

void TasksView::editorClosed(QWidget* editor, QAbstractItemDelegate::EndEditHint hint)
{
	QModelIndexList idxs=selectedIndexes();
	QModelIndex idx=idxs.last();
	if(model()->data(idx, Qt::DisplayRole).toString().isEmpty())
	{
		model()->removeRow(idx.row());		
		idx=idx.sibling(idx.row()-1, 0);
		selectionModel()->clear();
		selectionModel()->setCurrentIndex(idx, QItemSelectionModel::Rows | QItemSelectionModel::Select);
	}
}

void TasksView::mousePressEvent(QMouseEvent* event) 
{
	QTreeView::mousePressEvent(event);
	QModelIndex idx= indexAt(event->pos());
	if (!idx.isValid())
		emit addTask();
}

void TasksView::keyPressEvent(QKeyEvent* e)
{
	QModelIndexList idxs=selectedIndexes();
	if(idxs.isEmpty())
	{
		QTreeView::keyPressEvent(e);
		return;
	}
	
	switch(e->key())
	{
		case Qt::Key_Backspace:
			edit(idxs.last());
			break;
		case Qt::Key_Down:
			if(state()!=QAbstractItemView::EditingState && idxs.last().row()+1>=model()->rowCount())
				emit addTask();
			else
				QTreeView::keyPressEvent(e);
			break;
		case Qt::Key_Return:
			break;
		case Qt::Key_Tab:
		case Qt::Key_T:
			foreach(const QModelIndex& idx, idxs)
				model()->setData(idx, !idx.data(TasksModel::ImportantRole).toBool(), TasksModel::ImportantRole);
			break;
		default:
			QTreeView::keyPressEvent(e);
			break;
	}
}

void TasksView::sortView()
{
	header()->setSortIndicator(0, Qt::DescendingOrder);
}

void TasksView::aboutToAdd()
{
	setState(QAbstractItemView::NoState);
}

void TasksView::customContextMenuRequested(const QPoint& point )
{
	QModelIndex idx=indexAt(point);
	if(idx.isValid()) {
		QScopedPointer<QMenu> menu(new QMenu);
		
		bool isImportant=idx.data(TasksModel::ImportantRole).toBool();
		QString text;
		if(isImportant)
			text=tr("Not important");
		else
			text=tr("Mark as important");
		
		QAction* ratingAction=menu->addAction(TasksModel::ratingIcon(), text);
		QAction* removeAction=menu->addAction(QIcon::fromTheme( "list-remove" ), tr("Remove task"));
		removeAction->setEnabled(state()!=EditingState);
		
		QAction* res=menu->exec(mapToGlobal(point));
		if(res==ratingAction)
			model()->setData(idx, !isImportant, TasksModel::ImportantRole);
		else if(res==removeAction)
			model()->removeRow(idx.row());
		
	}
}

void TasksView::setModel(QAbstractItemModel* model)
{
	QTreeView::setModel(model);
	connect(model, SIGNAL(dataChanged(QModelIndex,QModelIndex)), SLOT(sortView()));
}

#include "tasksview.moc"
