#ifndef LUM_OS_WINDOW_H
#define LUM_OS_WINDOW_H

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Private/ImportExport.h>

#include <Lum/Base/Object.h>

#include <Lum/Model/Action.h>

#include <Lum/OS/DrawInfo.h>
#include <Lum/OS/Event.h>

namespace Lum {
  namespace OS {

    /**
      Representation of on OS desktop window.
    */
    class LUMAPI Window : public Lum::Base::MsgObject
    {
    public:
      enum Type {
        typeToolbar,
        typeTooltip,
        typeMenu,
        typeMenuDropdown,
        typeMenuPopup,
        typePopup,
        typeUtility,
        typeSplash,
        typeDialog,
        typeMain,
        typeAuto,
        typeCustom    //! Framework will leave the window allown and set any type hints
      };

      enum Position {
        positionManual,
        positionCenterOnParent,
        positionCenterOnScreen,
        positionOS
      };

      class LUMAPI RedrawMsg : public ::Lum::Base::ResyncMsg
      {
      public:
        int    x;
        int    y;
        size_t width;
        size_t height;
      };

      class LUMAPI ResizeMsg : public ::Lum::Base::ResyncMsg
      {
      public:
        size_t width;
        size_t height;
      };

      class LUMAPI EventMsg : public ::Lum::Base::ResyncMsg
      {
      public:
        Event *event;
      };

    public:
      virtual void SetMaster(Lum::Base::MsgObject* master) = 0;
      virtual Lum::Base::MsgObject* GetMaster() const = 0;

      /**
        @name Window type
      */
      //@{
      virtual void SetType(Window::Type type) = 0;
      virtual Window::Type GetType() const = 0;
      //@}

      /**
        @name Parent window
      */
      //@{
      virtual void SetParent(Window* parent) = 0;
      virtual Window* GetParent() const = 0;
      //@}

      /**
        @name Window title
      */
      //@{
      virtual void SetTitle(const std::wstring& name) = 0;
      virtual std::wstring GetTitle() const = 0;
      //@}

      /**
        @name Window opacity
      */
      //@{
      virtual bool SetOpacity(double opacity) = 0;
      virtual double GetOpacity() const = 0;
      //@}

      /**
        @name Position and size
        Method to set get and position and size of the window.
      */
      //@{
      virtual void SetSize(size_t width, size_t height) = 0;
      virtual void SetMinWidth(size_t width) = 0;
      virtual void SetMinHeight(size_t height) = 0;
      virtual void SetMaxWidth(size_t width) = 0;
      virtual void SetMaxHeight(size_t height) = 0;
      virtual void SetPos(int x, int y) = 0;
      virtual void SetPosition(Window::Position horiz, Window::Position vert) = 0;

      virtual int GetX() const = 0;
      virtual int GetY() const = 0;
      virtual size_t GetWidth() const = 0;
      virtual size_t GetHeight() const = 0;
      virtual size_t GetMinWidth() const = 0;
      virtual size_t GetMinHeight() const = 0;
      virtual size_t GetMaxWidth() const = 0;
      virtual size_t GetMaxHeight() const = 0;
      //@}

      /**
        @name Actions
        The Window offers a number of action that you can listen to get notified about
        special events
      */
      //@{
      virtual void SetExitAction(::Lum::Model::Action* action) = 0;

      virtual Model::Action* GetExitAction() const = 0;
      virtual Model::Action* GetClosedAction() const = 0;
      virtual Model::Action* GetMapedAction() const = 0;
      virtual Model::Action* GetUnmapedAction() const = 0;
      virtual Model::Action* GetPreOpenedAction() const = 0;
      virtual Model::Action* GetOpenedAction() const = 0;
      virtual Model::Action* GetHiddenAction() const = 0;
      virtual Model::Action* GetFocusInAction() const = 0;
      virtual Model::Action* GetFocusOutAction() const = 0;
      virtual Model::Action* GetMouseInAction() const = 0;
      virtual Model::Action* GetMouseOutAction() const = 0;
      virtual Model::Action* GetResizeAction() const = 0;
      virtual Model::Action* GetRedrawAction() const = 0;
      virtual Model::Action* GetPreInitAction() const = 0;
      virtual Model::Action* GetEventAction() const = 0;
      virtual Model::Action* GetGrabCancelAction() const =0;
      virtual Model::Action* GetMenuAction() const =0;
      //@}

      /**
        @name Mouse
        Methods reading mouse position and chekcing for double and tripple clicks.
      */
      //@{
      virtual bool GetMousePos(int& x, int& y) const = 0;
      virtual bool CursorIsIn() const = 0;
      virtual bool HasFocus() const = 0;
      virtual bool IsDoubleClicked() const = 0;
      virtual const MouseEvent& GetLastButtonClickEvent() const = 0;
      //@}

      /**
        @name Window visibility
      */
      //@{
      virtual bool IsInited() const = 0;
      virtual bool Open(bool activate=true) = 0;
      virtual void Close() = 0;
      virtual bool IsOpen() const = 0;

      virtual bool Show(bool activate=true) = 0;
      virtual bool Hide() = 0;
      virtual bool IsShown() const = 0;
      //@}

      /**
        @name Event loop and event handling
      */
      //@{
      virtual bool HandleEvent(Event* event) = 0;
      virtual void Exit() = 0;
      virtual void EventLoop() = 0;
      virtual bool IsInEventLoop() const = 0;
      virtual void Enable() = 0;
      virtual void Disable() = 0;
      virtual bool IsEnabled() const = 0;
      virtual void EnableParents() = 0;
      virtual void DisableParents() = 0;
      //@}

      /**
        @name Full screen methods
      */
      //@{
      virtual bool SetFullScreen() = 0;
      virtual bool LeaveFullScreen() = 0;
      virtual bool IsFullScreen() const = 0;
      //@}

      /**
        @name Double buffering methods
      */
      //@{
      /**
        On the first call of a pair: Exchange the current draw info for the visible
        window contents with an alternative DrawInfo structure for an off-screen
        background buffer. You can draw into this buffer without these drawing
        getting visible. Calling SwitchBackgroundDrawInfo() again
        will then make this changes visible.
      */
      virtual bool SwitchBackgroundDrawInfo(int x, int y, size_t width, size_t height) = 0;
      //@}

      virtual Window* GetTopWindow() = 0;

      virtual DrawInfo* GetDrawInfo() const = 0;

      virtual void Resize(size_t width, size_t height) = 0;

      virtual void GrabMouse(bool grab) = 0;
      virtual void GrabKeyboard(bool grab) = 0;

      virtual bool SendBalloonMessage(const std::wstring& text,
                                      unsigned long timeout) = 0;

      virtual void TriggerResize(size_t width, size_t height) = 0;
      virtual void TriggerRedraw(int x, int y, size_t width, size_t height) = 0;
    };
  }
}

#endif
