/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "profiling.h"

#include <QDebug>
#include <QDateTime>
#include <QNetworkReply>


Profiling::Profiling() :
    QObject(0),
    m_cacheHits(0),
    m_cacheMisses(0),
    m_isEnabled(false) {
}

Profiling::~Profiling() {
}

QVariantList Profiling::getMarks() const {

    return m_marks;
}

QVariantMap Profiling::getCacheStats() const {

    QVariantMap cacheStats;
    cacheStats["hits"] = m_cacheHits;
    cacheStats["misses"] = m_cacheMisses;
    return cacheStats;
}

void Profiling::enable() {

    qDebug() << "Enabling profiling";
    m_isEnabled = true;
}

void Profiling::disable() {

    qDebug() << "Disable profiling";
    m_isEnabled = false;
}

void Profiling::clear() {

    m_timerMap.clear();
    m_marks.clear();
    m_cacheHits = 0;
    m_cacheMisses = 0;
}

void Profiling::start(const QString &key) {

    if (isEnabled()) {
        QTime timer;
        timer.start();
        m_timerMap.insert(key, timer);
    }
}

void Profiling::stop(const QString &key) {

    if (!isEnabled()) {
        return;
    }

    if (!m_timerMap.contains(key)) {
        qDebug() << "ERROR key not found:" << key;
        return;
    }

    QTime timer = m_timerMap[key];
    int elapsed = timer.elapsed();

    QVariantMap mark;
    mark["key"] = key;
    // convert to unix timestamp in ms precision
    uint timestampSecs = QDateTime(QDateTime::currentDateTime().date(), timer).toTime_t();
    mark["timestamp"]= QString().sprintf("%d%d", timestampSecs, timer.msec()).toLongLong();
    mark["elapsed"] = elapsed;
    m_marks.append(mark);
}

void Profiling::startRequest(const QNetworkReply *reply) {

    if (!isEnabled() || !isApiCall(reply)) {
        return;
    }

    start(reply->url().toString());
}

void Profiling::stopRequest(const QNetworkReply *reply) {

    if (!isEnabled()) {
        return;
    }

    if (isApiCall(reply)) {
        stop(reply->url().toString());
    } else {
        // register whether cache was hit
        QVariant fromCache = reply->attribute(QNetworkRequest::SourceIsFromCacheAttribute);
        cacheHit(fromCache.toBool());
    }
}

bool Profiling::isEnabled() {

    return m_isEnabled;
}

bool Profiling::isApiCall(const QNetworkReply *reply) {

    return reply->url().scheme() != "file" && reply->url().toString().contains("ha_method");
}

void Profiling::cacheHit(bool cacheHit) {

    if (cacheHit) {
        m_cacheHits++;
    } else {
        m_cacheMisses++;
    }
}
