#include "debugconsole.h"

#include <QDebug>
#include <QDateTime>


static const int DEFAULT_MAX_LENGTH = 500;

static void globalMsgHandler(QtMsgType type, const char *msg) {

    QtMsgHandler handler = qInstallMsgHandler(0);

    switch (type) {
        case QtDebugMsg:
            DebugConsole::instance()->log(QString("qDebug: ") + msg);
            break;
        case QtWarningMsg:
            DebugConsole::instance()->log(QString("qWarning: ") + msg);
            break;
        case QtCriticalMsg:
            DebugConsole::instance()->log(QString("qCritical: ") + msg);
            break;
        case QtFatalMsg:
            qFatal(msg);
            abort();
    }

    qInstallMsgHandler(handler);
}

DebugConsole::DebugConsole() :
    QObject(0),
    m_maxLength(DEFAULT_MAX_LENGTH) {

    qInstallMsgHandler(globalMsgHandler);
}

DebugConsole::~DebugConsole() {

    qInstallMsgHandler(0); // unregister global message handler
}

DebugConsole *DebugConsole::instance() {

    static DebugConsole console;
    return &console;
}

void DebugConsole::log(const QString &message) {

    QtMsgHandler handler = qInstallMsgHandler(0);
    qDebug() << message;
    qInstallMsgHandler(handler);

    m_logMessages.enqueue(QDateTime::currentDateTime().toString("hh:mm:ss ") + message);

    if (m_logMessages.count() > m_maxLength) {
        m_logMessages.dequeue();
    }
}

QStringList DebugConsole::getLogMessages() const {

    return QStringList(m_logMessages);
}

int DebugConsole::maxLength() const {

    return m_maxLength;
}

void DebugConsole::setMaxLength(int max) {

    m_maxLength = max;
    for (int i = 0; i < (m_logMessages.count() - max); i++) {
        m_logMessages.dequeue();
    }
}
