from org.maemo.hermes.engine.facebook.service import Service
from org.maemo.hermes.engine.facebook.api import FacebookApi
from org.maemo.hermes.engine.names import canonical
from org.maemo.hermes.engine.friend import Friend
import unittest


class FakeContact():
    id_counter = 0
    def __init__(self, name, addr, id=None):
        self.name = name;
        self.urls = addr
        self.id = id or FakeContact.id_counter
        FakeContact.id_counter = FakeContact.id_counter + 1;
    def get_urls(self):
        return self.urls
    def get_name(self):
        return self.name
    def get_identifiers(self):
        return [canonical(self.name)]
    
class FakeFacebookApi():
    friends = []
    
    def get_friends(self):
        return FakeFacebookApi.friends
    
class FakeOAuth():
    def authorise(self, authorise_url, access_token_url, args = None):
        pass
    
    def request(self, url, args = None):
        if url == 'https://graph.facebook.com/me':
            return '{"id":"1234567","name":"Maemo Hermes"}'
        
        elif url == 'https://graph.facebook.com/me/friends':
            return '{"data":[{"id":"1","name":"J Smith","website":"http:\/\/www.example.org"},{"id":"2","name":"P Barnum"}]}'
        
        else:
            raise Exception("Unknown URL: %s" % (url))
        

class TestFacebookService(unittest.TestCase):
    
    def setUp(self):
        self.testee = Service("facebook", FakeFacebookApi())
        
    
    def test_that_get_friends_to_create_contacts_for_works(self):
        def run_test(expected_length):
            self._fake_server_response([{'id':'123456','name':'Facebook Junkie', 'birthday':'now'}])
            self._run_service([])
            friends = self.testee.get_friends_to_create_contacts_for()
            assert len(friends) == expected_length
            
        # default is to NOT create contacts
        self.testee = Service("facebook", FakeFacebookApi())
        run_test(0)
        
        # passing False explicitly
        self.testee = Service("facebook", FakeFacebookApi(), False)
        run_test(0)
        
        # passing True to constructor
        self.testee = Service("facebook", FakeFacebookApi(), True)
        run_test(1)
        
    
    def test_that_gftccf_returns_friends_with_birth_date(self):
        self.testee = Service("facebook", FakeFacebookApi(), True)
        bday = '1980-10-15'
        props_with_bday = {'id':'123456','name':'Facebook Junkie', 'birthday':bday}
        props_without = {'id':'123457','name':'Johnny Secret'}
        self._fake_server_response([props_with_bday, props_without])
        self._run_service([])
        
        assert len(self.testee.get_friends()) == 2
        friends = self.testee.get_friends_to_create_contacts_for()
        assert len(friends) == 1
        assert friends[0].has_birthday_date()
        assert friends[0].get_birthday_date() == bday 
        
        
    def test_that_process_contact_returns_friend_object_for_known_contact(self):
        known_url = 'http://www.facebook.com/profile.php?id=123456'
        known_contact = FakeContact('Facebook Junkie', [known_url])
        self._fake_server_response([{'id':'123456','name':'Facebook Junkie'}])
        
        self.testee.process_friends()
        result = self.testee.process_contact(known_contact)
        assert isinstance(result, Friend)


    def test_that_process_contact_returns_None_for_unknown_contact(self):
        known_contact = FakeContact('Facebook Junkie', [])
        self._fake_server_response([])
        
        self.testee.process_friends()
        result = self.testee.process_contact(known_contact)
        assert result is None


    def test_main_flow_one_match_by_url_one_by_name(self):
        # arrange
        self.existing_address = 'http://www.facebook.com/profile.php?id=123456'
        self.existing_contact = FakeContact("Facebook Person", [self.existing_address])
        existing_fake = {'id':'123456','name':'Name Doesnt Match but URL Does'}
        
        self.other_address = 'http://twitter.com/not/correct/site'
        self.other_contact = FakeContact("Twitter Person", [self.other_address])
        other_fake = {'id':'123','name':self.other_contact.get_name()}
        
        self.none_contact = FakeContact("No URLson", [])
        
        fake_data = [existing_fake, other_fake]
        self._fake_server_response(fake_data)

        # act        
        self._run_service([self.existing_contact, self.other_contact, self.none_contact])
        
        # assert
        friends = self.testee.get_friends()
        contacts = self.testee.get_contacts_with_match()
        assert len(friends) == 2
        assert len(contacts) == 2
        assert self.other_contact in contacts
        assert self.other_contact == contacts[self.other_contact].get_contact()
        assert self.existing_contact in contacts
        assert self.existing_contact == contacts[self.existing_contact].get_contact()
        assert self.none_contact not in contacts
        
        
    def test_name_collision_avoided_by_previous_matching(self):
        name = "Same Name"
        contact_do_match = FakeContact(name, ["http://www.facebook.com/profile.php?id=123"], 1);
        contact_no_match = FakeContact(name, [None], 2)
        
        data = [{'id':'123','name':name}]
        self._fake_server_response(data)
        
        self._run_service([contact_no_match, contact_do_match])
        
        assert len(self.testee.get_unmatched_friends()) == 0
        matchers = self.testee.get_contacts_with_match().keys()
        assert len(matchers) == 1
        assert matchers[0].id == 1

      
    def test_name_collision_avoided_only_one_person_matched(self):
        name = "Same Name"
        contact_do_match = FakeContact(name, ["Contact 1"]);
        contact_no_match = FakeContact(name, ["Contact 2"])
        
        data = [{'id':'123','name':name}]
        self._fake_server_response(data)
        
        self._run_service([contact_no_match, contact_do_match])
        
        matchers = self.testee.get_contacts_with_match().keys()
        assert len(matchers) == 1
        assert len(self.testee.get_unmatched_friends()) == 0
        
        
    def _run_service(self, contacts):
        for contact in contacts:
            self.testee.pre_process_contact(contact)
        self.testee.process_friends()
        for contact in contacts:
            self.testee.process_contact(contact)
        
    def _fake_server_response(self, data):
        FakeFacebookApi.friends = data


class TestFacebookAPI(unittest.TestCase):

    def setUp(self):
        self.oauth = FakeOAuth()
        self.api = FacebookApi(self.oauth)

    def test_authenticate(self):
        pass
    
    
    def test_get_user(self):
        assert self.api.get_user() == 'Maemo Hermes'
       
        
    def test_get_friends(self):
        friends = self.api.get_friends()
        assert len(friends) == 2
        assert friends[0]['name'] == 'J Smith'
        assert friends[1]['id'] == '2'

    
if __name__ == '__main__':
    unittest.main()
