/*
  This source is part of the FindMine program.
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include "Control.h"

#include <cstdlib>

#include <Lum/Base/String.h>
#include <Lum/Base/Util.h>

#include <Lum/OS/Driver.h>

Sweeper::Sweeper()
: sWidth(0),sHeight(0),
  emptyColor(0.6,0.6,0.6,Lum::OS::Display::blackColor),
  cellColor(0.7,0.7,0.7,Lum::OS::Display::blackColor),
  lightColor(0.95,0.95,0.95,Lum::OS::Display::blackColor),
  darkColor(0.2,0.2,0.2,Lum::OS::Display::blackColor),
  textColor(1,1,1,Lum::OS::Display::whiteColor),
  bombColor(0,0,0,Lum::OS::Display::whiteColor),
  markColor(1,0,0,Lum::OS::Display::whiteColor),
  status(new Lum::Model::ULong()),
  marks(new Lum::Model::Long()),
  bombs(0),
  oneButtonMode(false),
  game(NULL),
  themeIndex(0)
{
  sMaxWidth=areaWidth;
  sMaxHeight=areaHeight;

  status->Set(waiting);
  marks->Set(0);

  font=Lum::OS::display->GetFont(Lum::OS::Display::fontTypeFixed,
                                 Lum::OS::Display::fontScaleFootnote);
}

Sweeper::~Sweeper()
{
  delete game;
}

bool Sweeper::RequiresKnobs() const
{
  return false;
}


void Sweeper::EvaluateThemeName()
{
  for (size_t i=0; i<themes.size(); i++) {
    if (themes[i].name==themeName) {
      themeIndex=i;

      if (game!=NULL) {
        delete game;
        game=Lum::OS::driver->CreateBitmap(GetAreaWidth()*GetBoxSize(),
                                           GetAreaHeight()*GetBoxSize());
        RedrawArea();

        Redraw();
      }
      return;
    }
  }

  assert(false);
}

void Sweeper::CalcSize()
{
  width=16*24;
  height=16*24;

  minWidth=10*24;
  minHeight=10*24;

  Lum::Scrollable::CalcSize();
}

void Sweeper::UpdateDimensions()
{
  if (game!=NULL) {
    hAdjustment->SetDimension(width,game->GetWidth());
    vAdjustment->SetDimension(height,game->GetHeight());
  }
  else {
    hAdjustment->SetInvalid();
    vAdjustment->SetInvalid();
  }
}

void Sweeper::DrawEmptyCellBackground(Lum::OS::DrawInfo* draw,
                                       int x, int y,
                                       size_t w, size_t h)
{
  draw->PushForeground(emptyColor);
  draw->FillRectangle(x,y,w,h);
  draw->PopForeground();
}

void Sweeper::DrawHiddenCellBackground(Lum::OS::DrawInfo* draw,
                                       int x, int y,
                                       size_t w, size_t h)
{
  draw->PushForeground(cellColor);
  draw->FillRectangle(x,y,w,h);
  draw->PopForeground();
}

void Sweeper::DrawHiddenCellInternal(Lum::OS::DrawInfo* draw,
                                     int x, int y,
                                     size_t w, size_t h)
{
  draw->PushForeground(lightColor);
  draw->DrawLine(x,y+h-1,x,y);
  draw->DrawLine(x,y,x+w-1,y);
  draw->PopForeground();

  draw->PushForeground(lightColor);
  draw->DrawLine(x+1,y+h-2,x+1,y);
  draw->DrawLine(x,y+1,x+w-2,y+1);
  draw->PopForeground();

  draw->PushForeground(darkColor);
  draw->DrawLine(x+1,y+h-1,x+w-1,y+h-1);
  draw->DrawLine(x+w-1,y+h-1,x+w-1,y);
  draw->PopForeground();

  draw->PushForeground(darkColor);
  draw->DrawLine(x+2,y+h-2,x+w-2,y+h-2);
  draw->DrawLine(x+w-2,y+h-2,x+w-2,y+1);
  draw->PopForeground();
}

void Sweeper::DrawCell(size_t row, size_t column)
{
  size_t            x,y,frameA,frameB,frameSize;
  Lum::OS::DrawInfo *draw=game->GetDrawInfo();

  x=row*GetBoxSize();
  y=column*GetBoxSize();

  frameA=4;
  frameB=4;
  frameSize=frameA+frameB;

  DrawEmptyCellBackground(draw,x,y,GetBoxSize(),GetBoxSize());

  // Check if we have a filled gaming area
  if (area[row][column].active) {
    // if the given cell is already unrevealed...
    if (area[row][column].open) {
      // if there is a bomb...
      if (area[row][column].bomb) {
        if (GetImage(bombImage)!=NULL) {
          GetImage(bombImage)->Draw(draw,x,y);
        }
        else {
          draw->PushForeground(bombColor);
          draw->FillArc(x+frameA,
                        y+frameA,
                        GetBoxSize()-frameSize,
                        GetBoxSize()-frameSize,
                        0,360*64);
          draw->PopForeground();
        }
      }
      // no bomb...
      else {
        // paint the number of hidden neighbours...
        if (area[row][column].count!=0) {
          std::wstring string=Lum::Base::NumberToWString((unsigned long)area[row][column].count);

          draw->PushFont(font);
          draw->PushForeground(textColor);
          draw->DrawString(x+(GetBoxSize()-font->StringWidth(string,Lum::OS::Font::normal)) / 2,
                           y+(GetBoxSize()-font->height) / 2+font->ascent,
                           string);
          draw->PopForeground();
          draw->PopFont();
        }
      }
    }
    // Cell is still hidden
    else {
      // if we are finished playing, draw hidden bomb cells as unhidden and having a bomb...
      if ((status->Get()!=playing) && (area[row][column].bomb)) {
        if (GetImage(bombImage)!=NULL) {
          GetImage(bombImage)->Draw(draw,x,y);
        }
        else {
          draw->PushForeground(bombColor);
          draw->FillArc(x+frameA,
                        y+frameA,
                        GetBoxSize()-frameSize,
                        GetBoxSize()-frameSize,
                        0,360*64);
          draw->PopForeground();
        }
      }
      // if the cell is marked as having a bomb, do special drawing...
      else if (area[row][column].marked) {
        DrawHiddenCellBackground(draw,x,y,GetBoxSize(),GetBoxSize());

        if (GetImage(markedImage)!=NULL) {
          GetImage(markedImage)->Draw(draw,x,y);
        }
        else {
          DrawHiddenCellInternal(draw,x,y,GetBoxSize(),GetBoxSize());

          draw->PushForeground(markColor);
          draw->FillRectangle(x+frameA,
                              y+frameA,
                              GetBoxSize()-frameSize,
                              GetBoxSize()-frameSize);
          draw->PopForeground();
        }
      }
      // draw the cell as normal hidden cell...
      else {
        DrawHiddenCellBackground(draw,x,y,GetBoxSize(),GetBoxSize());

        if (GetImage(hiddenImage)!=NULL) {
          GetImage(hiddenImage)->Draw(draw,x,y);
        }
        else {
          DrawHiddenCellInternal(draw,x,y,GetBoxSize(),GetBoxSize());
        }
      }
    }
  }
}

bool Sweeper::GetNeighbour(size_t n,
                           size_t column, size_t row,
                           size_t& x, size_t& y)
{
  /*
    012
    345
    678
   */

  switch (n) {
  case 0:
    x=column-1;
    y=row-1;
    break;
  case 1:
    x=column;
    y=row-1;
    break;
  case 2:
    x=column+1;
    y=row-1;
    break;
  case 3:
    x=column-1;
    y=row;
    break;
  case 4:
    x=column;
    y=row;
    break;
  case 5:
    x=column+1;
    y=row;
    break;
  case 6:
    x=column-1;
    y=row+1;
    break;
  case 7:
    x=column;
    y=row+1;
    break;
  case 8:
    x=column+1;
    y=row+1;
    break;
  }

  return x<sWidth && y<sHeight;
}

void Sweeper::RecalcArea()
{
  bool changed,notFound;

  size_t x,y;

  /* automatically open cells which have 0 neighbours */

  do {
    long marks=bombs;

    changed=false;
    notFound=false;

    /* calculate neighbourhood */
    for (size_t column=0; column<sWidth; column++) {
      for (size_t row=0; row<sHeight; row++) {
        size_t oldBombs,bombs;

        oldBombs=area[column][row].count;
        bombs=0;

        for (size_t n=0; n<9; n++) {
          if (GetNeighbour(n,column,row,x,y)) {
            if (area[x][y].bomb) {
              bombs++;
            }
          }
        }

        if (bombs!=oldBombs) {
          area[column][row].count=bombs;

          if (area[column][row].open) {
            DrawCell(column,row);
          }
        }

        // check if you have opened a bomb
        if (area[column][row].open && area[column][row].bomb && (status->Get()!=lost)) {
          timer.Pause();
          status->Set(lost);
          RedrawArea(); // We redraw the complete area to show all bombs
          Redraw();
        }

        // still unopened areas without bombs
        if (!area[column][row].bomb && !area[column][row].open) {
          notFound=true;
        }
      }
    }

    /* open automatically when neighbour has count 0 and is opened */
    for (size_t column=0; column<sWidth; column++) {
      for (size_t row=0; row<sHeight; row++) {

        if (area[column][row].open && (area[column][row].count==0)) {
          for (size_t n=0; n<9; n++) {
            if (GetNeighbour(n,column,row,x,y)) {
              if (!area[x][y].open) {
                area[x][y].open=true;
                DrawCell(x,y);
                changed=true;
              }
            }
          }
        }
      }
    }

    for (size_t column=0; column<sWidth; column++) {
      for (size_t row=0; row<sHeight; row++) {
        if (area[column][row].marked) {
          marks--;
        }
      }
    }

    this->marks->Set(marks);

    if ((status->Get()==playing) && !notFound) {
      timer.Pause();
      status->Set(won);
    }
  } while (changed);
}

size_t Sweeper::GetBoxSize() const
{
  return themes[themeIndex].boxSize;
}

Lum::Images::Image* Sweeper::GetImage(Image image) const
{
  return themes[themeIndex].images[image];
}

void Sweeper::RedrawArea()
{
  for (size_t x=0; x<sWidth; x++) {
    for (size_t y=0; y<sHeight; y++) {
      DrawCell(x,y);
    }
  }
}

void Sweeper::OpenField(size_t x, size_t y)
{
  if (area[x][y].active) {
    if (!area[x][y].open) {
      if (status->Get()==playing && (timer.IsStoped() || timer.IsPausing())) {
        timer.Start();
      }

      area[x][y].open=true;
      area[x][y].marked=false;
      DrawCell(x,y);

      RecalcArea();

      Redraw();
    }
  }
}

void Sweeper::ToggleMark(size_t x, size_t y)
{
  if (area[x][y].active) {
    if (!area[x][y].open) {
      if (status->Get()==playing && (timer.IsStoped() || timer.IsPausing())) {
        timer.Start();
      }

      area[x][y].marked=!area[x][y].marked;
      DrawCell(x,y);

      RecalcArea();

      Redraw();
    }
  }
}

void Sweeper::Draw(int x, int y, size_t w, size_t h)
{
  Scrollable::Draw(x,y,w,h);

  if (!OIntersect(x,y,w,h)) {
    return;
  }

  /* --- */

  if (game==NULL) {
    return;
  }

  Lum::OS::DrawInfo *draw=GetDrawInfo();

  draw->CopyFromBitmap(game,
                       hAdjustment->GetTop()-1,vAdjustment->GetTop()-1,
                       this->width,this->height,
                       this->x,this->y);
}

bool Sweeper::HandleMouseEvent(const Lum::OS::MouseEvent& event)
{
  if (!visible || status->Get()!=playing) {
    return false;
  }

  if (event.type==Lum::OS::MouseEvent::down && PointIsIn(event)) {
    mbdx=event.x;
    mbdy=event.y;

    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) &&
           event.button==Lum::OS::MouseEvent::button1 && event.qualifier==Lum::OS::qualifierButton1) {

    size_t x,y;

    x=(event.x-this->x+hAdjustment->GetTop()-1)/GetBoxSize();
    y=(event.y-this->y+vAdjustment->GetTop()-1)/GetBoxSize();

    if (oneButtonMode) {
      if (GetWindow()->IsDoubleClicked()) {
        OpenField(x,y);
      }
      else {
        ToggleMark(x,y);
      }
    }
    else {
      OpenField(x,y);
    }

    return true;
  }
  else if (event.type==Lum::OS::MouseEvent::up && PointIsIn(event) &&
           event.button==Lum::OS::MouseEvent::button3 &&
           event.qualifier==Lum::OS::qualifierButton3) {

    if (oneButtonMode) {
      return true;
    }

    size_t x,y;

    x=(event.x-this->x+hAdjustment->GetTop()-1)/GetBoxSize();
    y=(event.y-this->y+vAdjustment->GetTop()-1)/GetBoxSize();

    ToggleMark(x,y);

    return true;
  }

  return false;
}

bool Sweeper::SetSize(size_t width, size_t height, size_t bombs)
{
  if (width>sMaxWidth || height>sMaxHeight || bombs>width*height) {
    return false;
  }

  sWidth=width;
  sHeight=height;
  this->bombs=bombs;

  delete game;
  game=Lum::OS::driver->CreateBitmap(GetAreaWidth()*GetBoxSize(),
                                     GetAreaHeight()*GetBoxSize());

  Run();
  RedrawArea();
  UpdateDimensions();
  Redraw();

  return true;
}

void Sweeper::SetOneButtonMode(bool set)
{
  oneButtonMode=set;
}

size_t Sweeper::GetAreaWidth() const
{
  return sWidth;
}

size_t Sweeper::GetAreaHeight() const
{
  return sHeight;
}

size_t Sweeper::GetMines() const
{
  return bombs;
}

size_t Sweeper::GetMinesPercent() const
{
  return Lum::Base::RoundDiv(bombs*100,sWidth*sHeight);
}

Lum::Model::ULong* Sweeper::GetStatusModel() const
{
  return status.Get();
}

Lum::Model::Long* Sweeper::GetMarksModel() const
{
  return marks.Get();
}

void Sweeper::Run()
{
  size_t cells,bombs;

  /* Set a more random seed */
  srand(time(NULL));

  /* initialize */
  for (size_t x=0; x<sMaxWidth; x++) {
    for (size_t y=0; y<sMaxHeight; y++) {
      area[x][y].bomb=false;
      area[x][y].open=false;
      area[x][y].marked=false;
      area[x][y].count=0;
      area[x][y].active=false;
    }
  }

  cells=sWidth*sHeight;
  bombs=this->bombs;

  for (size_t x=0; x<sWidth; x++) {
    for (size_t y=0; y<sHeight; y++) {
      area[x][y].active=true;

      area[x][y].bomb=((float)cells*rand()/RAND_MAX)<bombs;

      if (area[x][y].bomb) {
        bombs--;
      }

      cells--;
    }
  }

  status->Set(playing);
  marks->Set(this->bombs);

  RedrawArea();

  Redraw();

  timer.Reset();
}

Sweeper::Status Sweeper::GetStatus() const
{
  return (Status)status->Get();
}

time_t Sweeper::GetElapsedTime() const
{
  return timer.GetTime();
}

void Sweeper::SetStatusToWaiting()
{
  status->Set(waiting);
  if (visible) {
    Redraw();
  }
}

void Sweeper::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
{
  if (model==hAdjustment->GetTopModel() || model==vAdjustment->GetTopModel()) {
    Redraw();
  }

  Lum::Scrollable::Resync(model,msg);
}

