#include <QMutex>

#include <FCam/processing/Demosaic.h>
#include <FCam/processing/DNG.h>
#include <FCam/processing/Dump.h>
#include <FCam/processing/JPEG.h>
#include "ImageItem.h"
#include "UserDefaults.h"

#include <iostream>
#include <QDir>

ImageItem::ImageItem() :
    fname(""), saved(false), error(false), lock(QMutex::Recursive) {
    loading = FALSE;
    saving = FALSE;
    loadingThumb = FALSE;
    loadedThumb = FALSE;
    loaded = FALSE;
}

ImageItem::ImageItem(const FCam::Frame &f) : 
    src(f), saved(false), error(false), lock(QMutex::Recursive) {
    loading = FALSE;
    saving = FALSE;
    loadingThumb = FALSE;
    loaded = TRUE;
    loadedThumb = FALSE;
    

    UserDefaults &userDefaults = UserDefaults::instance();
    fpath = userDefaults["rawPath"].asString().c_str();
    QString prefix = userDefaults["filenamePrefix"].asString().c_str();
    QString suffix;
    if (userDefaults["filenameSuffix"].asString() == "timestamp") {
        suffix =  src.exposureStartTime().toString().c_str();
    } else {
        // Calculate lowest index number that's greater than all files with the same prefix
        // in the same directory.
        int biggestIndex = 0;
        QDir dir(fpath);
        QStringList files = dir.entryList();
        for (int i = 0; i < files.size(); i++) {
            QString f = files.at(i);
            if (f.startsWith(prefix) && f.endsWith(".dng")) {
                QString indexSubstring = f.mid(prefix.count());
                indexSubstring = indexSubstring.left(indexSubstring.count() - 4);
                bool isNumber;
                int index = indexSubstring.toInt(&isNumber);
                if (isNumber) {
                    biggestIndex = qMax(index, biggestIndex);
                }
            }
        }
        suffix = QString().sprintf("%03d", biggestIndex+1);
    }    
    fname = QString("%1%2").arg(prefix).arg(suffix);    

    if (!src.valid() || !src.image().valid()) error = true;
}

ImageItem::ImageItem(const QString &filename) :
    saved(true), error(false), lock(QMutex::Recursive) {
    int lastSlashIndex = filename.lastIndexOf("/");
    
    fpath = filename.left(lastSlashIndex+1);
    fname = filename.right(filename.count() - lastSlashIndex);
    fname = fname.left(fname.count() - 4); // 4 because we want to trim ".dng"
    loading = FALSE;
    saving = FALSE;
}

ImageItem::~ImageItem() {

}

FCam::Frame ImageItem::frame() {
    return src;
}

void ImageItem::discardFrame() {
    if (!lock.tryLock()) return;
    if (saved) {
        src = FCam::Frame();
        loaded = FALSE;
    }
    // Clear cached data
    demosaicImage = FCam::Image();
    demosaicMap = FCam::Image();
    demosaicPixmap.reset();

    
    // Keep the thumbnail around
    //thumb = FCam::Image();
    //pix.reset();
    lock.unlock();
}

void ImageItem::discardThumbnail() {
    if (!lock.tryLock()) return;
    thumb = FCam::Image();
    loadedThumb = FALSE;
    
    pix.reset();
    lock.unlock();
}

QString ImageItem::fullPath() {
    return QString("%1%2.dng").arg(fpath).arg(fname);
}

void ImageItem::load() {
    if (error) return;
    lock.lock();
    if (src.valid() || fname.size() == 0) {
        lock.unlock();
        return;
    }
    src = FCam::loadDNG(this->fullPath().toStdString());
    thumb = FCam::loadDump(this->fullPath().toStdString() + ".thumb");
    if (src.valid() && !thumb.valid()) thumbnail();
    if (!src.valid()) error = true;
    saved = true;
    loaded = TRUE;
    loadedThumb = TRUE;
    loading = FALSE;
    loadingThumb = FALSE;
    lock.unlock();
}

void ImageItem::loadThumbnail() {
    if (error) return;
    lock.lock();
    if (thumb.valid()) {
        loadingThumb = FALSE;
        loadedThumb = TRUE;
        lock.unlock();
        return;
    }
    thumb = FCam::loadDump(this->fullPath().toStdString() + ".thumb");
    loadingThumb = FALSE;
    loadedThumb = TRUE;
    if (!thumb.valid()) error = true;
    lock.unlock();
    
}

void ImageItem::loadThumbnailAsync() {
    if (error) return;
    if (thumb.valid()) return;
    if (fname.size() == 0) return;
    loadingThumb = TRUE;
    IOThread::instance().loadThumbnail(this);
}

void ImageItem::loadAsync() {
    if (error) return;
    if (src.valid()) return;
    if (fname.size() == 0) return;
    loading = TRUE;
    loadingThumb = TRUE;
    IOThread::instance().load(this);
}

void ImageItem::save() {
    if (error) return;
    UserDefaults &userDefaults = UserDefaults::instance();
    lock.lock();
    if (!src.valid() || saved) {
        lock.unlock();
        return;
    }
    
    QDir dir;
    dir.mkpath(userDefaults["rawPath"].asString().c_str());
    
    FCam::saveDNG(src, this->fullPath().toStdString());
    FCam::saveDump(thumbnail(), this->fullPath().toStdString() + ".thumb");
    
    
    if (userDefaults["autosaveJPGs"].asInt()) {
        QString path = "/home/user/MyDocs/DCIM/";
        path.append(fname).append(".jpg");
        FCam::saveJPEG(src, path.toStdString());
    }
    saving = FALSE;
    saved = TRUE;
    
    lock.unlock();
}

QString ImageItem::tempJPEGPath(){
    QString path = QDir::tempPath();
    path.append("/").append(fname).append(".jpg");
    if (!src.valid() || !src.image().valid())
        this->load();        
    FCam::saveJPEG(src, path.toStdString());
    return path;
}

bool ImageItem::placeholder() {
    return (fname.count() + fpath.count() == 0);
}
void ImageItem::saveAsync() {
    if (!src.valid() || saved) return;
    saving = TRUE;
    IOThread::instance().save(this);
}

void ImageItem::demosaic() {
    if (!src.valid() || !src.image().valid())
        return;
    FCam::Image demosaicFImage = FCam::demosaic(src);
    QImage demosaicQImage(demosaicFImage(0,0), 
                          demosaicFImage.width(), demosaicFImage.height(), QImage::Format_RGB888);

    QPixmap * tempPixmap = new QPixmap(QPixmap::fromImage(demosaicQImage));
    
    lock.lock();
    demosaicPixmap.reset(tempPixmap);
    lock.unlock();
}

void ImageItem::demosaicAsync() {
    if (!src.valid()) return;
    IOThread::instance().demosaic(this);
}


FCam::Image ImageItem::thumbnail() {
    if (!lock.tryLock()) {
        return FCam::Image();
    }
    if (!thumb.valid() && src.valid()) {
        thumb = FCam::makeThumbnail(src, FCam::Size(640, 480));
    }
    lock.unlock();
    return thumb;
}

const QPixmap ImageItem::pixmap() {
    if (!lock.tryLock()) return QPixmap();
    thumbnail();
    if (!thumb.valid()) {
        lock.unlock();
        return QPixmap();
    } else if (!pix) {
        QImage thumbQ(thumb(0,0), 640, 480, QImage::Format_RGB888);
        pix.reset(new QPixmap(QPixmap::fromImage(thumbQ)));
    } 
    lock.unlock();
    return *pix;
}

const QPixmap ImageItem::fullResPixmap() {    
    lock.lock();
    
    if (!demosaicPixmap || demosaicPixmap->isNull()) {
        if (!src.valid()) {
            this->load();
        } 
        demosaicPixmap.reset(
            new QPixmap(this->pixmap().scaled(src.image().width(), src.image().height())));
        this->demosaicAsync();
    }
    lock.unlock();
    return *demosaicPixmap;
}

const QString &ImageItem::filename() {
    return fname;
}

bool ImageItem::safeToDelete() {
    //printf("saving %d (ed) %d loading %d, loadingThumb %d\n", saving, saved, loading, loadingThumb);
    return saved && !saving && !loading && !loadingThumb;
}

void IOThread::stop() {
    queue.push(Request(NULL, Stop));
}


IOThread::IOThread() : QThread(NULL) {
    
}

IOThread *IOThread::_instance = NULL;

IOThread &IOThread::instance() {
    if (!_instance) {
        _instance = new IOThread();
        _instance->start(QThread::IdlePriority);
    }
    return *_instance;
}

void IOThread::run() {
    while (1) {
        Request r = queue.pull();

        switch (r.type) {
        case Stop:
            printf("IOThread stopping\n");
            return;
        case Load:
            if (queue.size() < 8) {
                // start ignoring old loads if I get hammered with requests
                r.image->load();
                emit loadFinished(r.image);
                break;
            }
        case Save:
            r.image->save();
            emit saveFinished(r.image);
            break;
        case LoadThumbnail:
            if (queue.size() < 8) {
                // start ignoring old loads if I get hammered with requests
                r.image->loadThumbnail();
                emit loadFinished(r.image);
                break;
            }
        case Demosaic:
            r.image->demosaic();
            emit demosaicFinished();        
            break;
        }

    }
}

