/****************************************************************************
**
** Copyright (C) 2010  Aimone Davide realsportcars@hotmail.com
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "dictionary.h"
#include <QtGui>

#define USERDICT_FILE "user_"
#define PLAINDICT_FILE "plain.txt"
#define SYMBOL_FILE "/basekey.txt"
#define DATA_PATH "/data/"
#define WORD_SEPARATOR ';'

dictionary::dictionary(QString *sInputFile, QObject *parent) :
        QObject(parent)
{
    m_dict = new QMultiHash<QString,QString>();
    m_userDict = new QMultiHash<QString,QString>();
    m_plainDict = new QMultiHash<QString,QString>();
    m_sLanguage = sInputFile;

    if(init() != 0)
    {
        QMessageBox msgBox;
        msgBox.setText("Some errors occured while loading dictionary");
        msgBox.exec();
    }
}


int dictionary::init()
{
    QString sDictFilePath = QApplication::applicationDirPath() + DATA_PATH + *m_sLanguage;
    QFile *fDict = new QFile(sDictFilePath);

    //This part manage the renaming of user dictionary file//////////
    QString sOldUserFilePath = QApplication::applicationDirPath() + DATA_PATH + "user.idc";
    QFile *fOldUserDict = new QFile(sOldUserFilePath);
    if(fOldUserDict->exists())
    {
        QDir dir (QApplication::applicationDirPath() + DATA_PATH);
        QFileInfo fInfo(*fOldUserDict);
        QString fPath = fInfo.absoluteFilePath();
        QString fPath_ren = fPath;
        fPath_ren.chop(4);
        fPath_ren += "_" + *m_sLanguage;
        dir.rename(fPath, fPath_ren);
    }
    //////////////////////////////////////////////////////////////////

    QString sUserFilePath = QApplication::applicationDirPath() + DATA_PATH + USERDICT_FILE + *m_sLanguage;
    QFile *fUserDict = new QFile(sUserFilePath);
    QString sPlainFilePath = QApplication::applicationDirPath() + DATA_PATH + PLAINDICT_FILE;
    QFile *fPlainDict = new QFile(sPlainFilePath);

    if(fDict->exists())
    {
        if(!fDict->open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QMessageBox msgBox;
            msgBox.setText("Warning!!!\nDictionary file exists but can't be opened:"
                           + sDictFilePath + " error code:" + QString(fDict->error()));
            msgBox.exec();
            return -1;
        }

        if(createTranslater() != 0)
            return -1;

        QTextStream in(fDict);

        while (!in.atEnd())
        {
            QString line = in.readLine();
            loadDictLine(line);
        }
        //Load user custom words
        if(fUserDict->open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QTextStream strUser(fUserDict);

            while (!strUser.atEnd())
            {
                QString line = strUser.readLine();
                loadDictLine(line,true);
            }
        }
        //Load plain dictionary for standard input style
        if(fPlainDict->open(QIODevice::ReadOnly | QIODevice::Text))
        {
            QTextStream strPlain(fPlainDict);

            while (!strPlain.atEnd())
            {
                QString line = strPlain.readLine();
                loadDictLine(line,false,true);
            }
        }
    }
    else
    {
        qDebug() << "Can't find" << sDictFilePath;
        return -1;
    }

    fDict->close();
    fUserDict->close();
    return 0;
}

int dictionary::createTranslater()
{
    QString sSymFilePath = QApplication::applicationDirPath() + DATA_PATH + SYMBOL_FILE;
    QFile *fSymb = new QFile(sSymFilePath);

    if(!fSymb->open(QIODevice::ReadOnly | QIODevice::Text))
    {
        QMessageBox msgBox;
        msgBox.setText("Warning!!!\nCan't open " + sSymFilePath);
        msgBox.exec();
        return -1;
    }

    m_hmTranslater = new QMultiHash <QString,QString>();
    QTextStream in(fSymb);

    while (!in.atEnd())
    {
        QString line = in.readLine();
        QString sCode=line.at(0);

        for(int i=1; i!=line.length(); i++)
        {
            m_hmTranslater->insert(line.at(i),sCode);
        }
    }

    fSymb->close();
    return 0;
}

/**
*   Create 3 dictionary:
*   m_dict is the global one and contains standard words plus user custom ones
*   m_userDict contains only custom words and it is used for marker for new words
*   m_plainDict is used when standard input style is selected
**/
void dictionary::loadDictLine(QString line,bool bUserDict,bool bPlainDict)
{
    if(!line.isEmpty())
    {
        QStringList lToken = line.split(WORD_SEPARATOR);
        QString sCode=lToken.at(0);

        for(int i=1; i!=lToken.size() -1; i++)
        {
            if(!bPlainDict)
            {
                m_dict->insert(sCode,lToken.at(i));

                if(bUserDict)
                    m_userDict->insert(sCode,lToken.at(i));
            }
            else //create a dictionary for standard input style
            {
                m_plainDict->insert(sCode,lToken.at(i));
            }
        }
    }
}

/** Returns words associated with key
*   if bPlain is true, plain dictionary (standard input mode) will be used.
**/
QList<QString> dictionary::getWordList(QString *key,bool bPlain)
{
    QList<QString> list;

    if(!bPlain)
        list = m_dict->values(*key);
    else
        list = m_plainDict->values(*key);


    return list;
}


/** Save all m_dict hashMap to custom user file, preserving words order **/
void dictionary::saveDictionaryToFile()
{
    qDebug() << "Saving dictionary";

    QString sDictFilePath = QApplication::applicationDirPath() + DATA_PATH + USERDICT_FILE + *m_sLanguage;
    QFile *fDict = new QFile(sDictFilePath);

    fDict->remove();
    if(fDict->open(QIODevice::WriteOnly | QIODevice::Text))
    {
        QTextStream out(fDict);
        QString sOldKey = "-1";
        QMultiHash<QString, QString>::iterator it = m_userDict->end();

        it--;
        while (it != m_userDict->begin())
        {
            if(sOldKey.compare(it.key())!=0)
            {
                if(it.key().length() > 0)
                {
                    out << "\n" + it.key() + ";";
                }
                sOldKey = it.key();
            }

            out << it.value()+ ";";
            --it;
        }
        ////////Save last entry
        if(sOldKey.compare(it.key())!=0)
        {
            if(it.key().length() > 0)
            {
                out << "\n" + it.key() + ";";
            }
            sOldKey = it.key();
        }

        out << it.value()+ ";";
        /////////////////

        out.flush();
        fDict->flush();

        fDict->close();
    }
    else
        qDebug() << "Can't save custom word Err.Code:" << fDict->error();


}

/** Add sWord to dictionary. Sub-words are inserted only if no key exists for the sub-words analyzed.
If sWord contains more than a word, only the first will be inserted **/
bool dictionary::addWord(QString *sWord)
{
    bool bInserted = false;

    sWord = new QString(sWord->toLower());
    int nIdx = sWord->indexOf(" ");
    if(nIdx != -1)
    {
        sWord = new QString(sWord->left(nIdx));
    }
    QString *sKey = calculateKey(sWord);

    if(!sKey->isEmpty())
    {
        if(!m_dict->contains(*sKey,*sWord))
        {
            m_dict->insert(*sKey,*sWord);
            m_userDict->insert(*sKey,*sWord);

            bInserted = true;

            for(int i=1;i<=sWord->length();i++)
            {
                QString sAbbrevation = sWord->left(i);
                QString sPartialKey = sKey->left(i);

                if(!m_dict->contains(sPartialKey))
                {
                    m_dict->insert(sPartialKey,sAbbrevation);
                    m_userDict->insert(sPartialKey,sAbbrevation);
                }
            }
        }
    }

    return bInserted;
}

///** sWord become the first choice for its key **/
//void dictionary::updateWordOrder(QString *sWord)
//{
//    QString *sKey = calculateKey(sWord);

//    if(m_dict->contains(*sKey,*sWord))
//    {
//        m_dict->remove(*sKey,*sWord);
//        m_dict->insert(*sKey,*sWord);
//    }
//}

/** Returns key for sWord or empty string if sWord can't be transalted **/
QString* dictionary::calculateKey(QString *sWord)
{
    QString *sKey = new QString();

    for(int i=0;i<sWord->length();i++)
    {
        QChar letter = sWord->at(i);

        QString sValue = m_hmTranslater->value(letter.toLower());
        if(sValue.isEmpty())
        {
            qDebug() << "Cant't recognize character:" << letter ;
            sKey->clear();
            break;
        }
        else
            sKey->append(sValue);
    }

    return sKey;
}

/** Returns true if exists, false otherwise **/
bool dictionary::existsKey(QString *sKey)
{
    return m_dict->contains(*sKey);
}

/** Returns true if car is mapped on "1" button, false otherwise **/
bool dictionary::isSymbol(QChar car)
{
    QString sVal = QString(car);
    return m_hmTranslater->contains(sVal,"1");
}

dictionary::~dictionary()
{
    m_dict->clear();
    delete m_dict;
    m_hmTranslater->clear();
    delete m_hmTranslater;
    m_sLanguage->clear();
    delete m_sLanguage;
    m_userDict->clear();
    delete m_userDict;
    m_plainDict->clear();
    delete m_plainDict;
}
