/*
    Copyright 2010 Marcello V. Mansueto <marcello.m@yahoo.com>

    This file is part of dlh.

    dlh is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    dlh is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with dlh.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <QtGui>
#include <QtSql>

#include "dlh.h"
#include "ui_dlh.h"
#include "about.h"

static bool createConnection()
{
    QSqlDatabase db = QSqlDatabase::addDatabase("QSQLITE");
    db.setDatabaseName(QDir::homePath().append("/MyDocs/dlh.db"));
    if (!db.open()) {
        QMessageBox::critical(0, QObject::tr("Cannot open database"),
        QObject::tr("Unable to establish a database connection.\n\n"
            "Click Cancel to exit."), QMessageBox::Cancel);
        return false;
    }
    QSqlQuery query;
    query.exec("create table dlh (start varchar(20) primary key, "
               "end varchar(20), wb number, wa number, dw number, "
               "mlb number, mla number, dml number, note varchar(500))");
    return true;
}

Widget::Widget(QWidget *parent)
    : QWidget(parent), ui(new Ui::Widget)
{
    start = NULL;
    view = NULL;

    ui->setupUi(this);

    createConnection();
    connect(ui->mainStartButton, SIGNAL(clicked()), this, SLOT(CallStartDialog()));
    connect(ui->mainViewButton, SIGNAL(clicked()), this, SLOT(CallViewDialog()));
    connect(ui->mainAboutButton, SIGNAL(clicked()), this, SLOT(openAboutDialog()));
    connect(ui->mainResetButton, SIGNAL(clicked()), this, SLOT(reset()));
    connect(ui->mainExportButton, SIGNAL(clicked()), this, SLOT(exportDB()));

    QDBusConnection::sessionBus().registerService("com.nokia.dlh");
    QDBusConnection::sessionBus().registerObject("/com/nokia/dlh", this, QDBusConnection::ExportAllContents);

    setAttribute(Qt::WA_Maemo5StackedWindow);
}

Widget::~Widget()
{
    delete ui;
}

void Widget::CallStartDialog() {
    activateWindow();
    if (mutex.tryLock()) {
        start = new Start(this);
        start->setAttribute(Qt::WA_Maemo5StackedWindow);
        start->setWindowFlags(start->windowFlags() | Qt::Window);
        connect(start, SIGNAL(widgetClosed()), this, SLOT(free()));
        start->show();
    }
}

void Widget::CallViewDialog() {
    activateWindow();
    if (mutex.tryLock()) {
        view = new View(this);
        view->setAttribute(Qt::WA_Maemo5StackedWindow);
        view->setWindowFlags(view->windowFlags() | Qt::Window);
        connect(view, SIGNAL(widgetClosed()), this, SLOT(free()));
        view->show();
    }
}

void Widget::free() {
    if (start){
        delete start;
        start = NULL;
    }
    if (view){
        delete view;
        view = NULL;
    }
    mutex.unlock();
}

void Widget::openAboutDialog() {
    About about;
    about.exec();
}

void Widget::reset() {
    QMessageBox::StandardButton answer =
    QMessageBox::question(this, tr("Are you sure?"),
        tr("This operation is undoable!!!"),
        QMessageBox::Yes | QMessageBox::No,
        QMessageBox::No);
    if (answer == QMessageBox::Yes) {
        QSqlQuery query;
        QString command = QString("delete from dlh");
        query.exec(command);
    }
}

void Widget::exportDB() {
    QString fileName = QFileDialog::getSaveFileName(this, tr("Select export file"),
                                                          QString("/home/user/MyDocs/.documents/dlh.csv"));
    if(fileName.isEmpty()) return;
    QFile file(fileName);
    if(!file.open(QFile::WriteOnly | QFile::Truncate)) {
        QMessageBox::critical(this, tr("Unable to open the export file"),
                                    tr("Do you have the permission to write this file?"));
        return;
    }
    QTextStream csv(&file);
    csv << tr("Start time,Finish time,Weight before,Weight after,Weight difference,ml before,ml after,ml difference,Note") << endl;
    QSqlTableModel model;
    model.setTable("dlh");
    model.setEditStrategy(QSqlTableModel::OnManualSubmit);
    model.select();
    for(int row = 0; row < model.rowCount(); row++) {
        csv << model.record(row).value("start").toString() << QString(",");
        csv << model.record(row).value("end").toString() << QString(",");
        csv << model.record(row).value("wb").toString() << QString(",");
        csv << model.record(row).value("wa").toString() << QString(",");
        csv << model.record(row).value("dw").toString() << QString(",");
        csv << model.record(row).value("mlb").toString() << QString(",");
        csv << model.record(row).value("mla").toString() << QString(",");
        csv << model.record(row).value("dml").toString() << QString(",");
        csv << QString("\"") << model.record(row).value("note").toString() << QString("\"") << endl;
    }
    file.close();
}
