#include "database.h"
#include "settings.h"
#include "utils.h"
#include <QFileInfo>
#include <QDir>
#include <QSqlQuery>
#include <QSqlRecord>
#include <QSqlError>
#include <QRegExp>
#include <QDateTime>
#include <QUrl>
#include <QDebug>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif

Database* databaseInstance = 0;

Database::Database(QObject *parent) :
    QObject(parent),
    m_database(QSqlDatabase::addDatabase("QSQLITE"))
{
    if (!databaseInstance) {
        databaseInstance = this;
    }
#if (defined (Q_WS_MAEMO_5)) || (defined (MEEGO_EDITION_HARMATTAN))
    QDir dir;
    dir.mkpath("/home/user/cuteTube/");
    m_database.setDatabaseName("/home/user/cuteTube/cuteTube.db");
#elif (defined (SYMBIAN_OS))
    m_database.setDatabaseName("cuteTube.db");
#elif QT_VERSION >= 0x050000
    QDir dir(QStandardPaths::writableLocation(QStandardPaths::HomeLocation) + "/cuteTube/");
    dir.mkpath(dir.path());
    m_database.setDatabaseName(dir.path() + "/cuteTube.db");
#else
    QDir dir(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/cuteTube/");
    dir.mkpath(dir.path());
    m_database.setDatabaseName(dir.path() + "/cuteTube.db");
#endif
    this->initialize();
}

Database* Database::instance() {
    return databaseInstance;
}

void Database::initialize() {
    if (m_database.open()) {
        m_database.exec("DROP TABLE IF EXISTS downloads");
        m_database.exec("CREATE TABLE IF NOT EXISTS storedDownloads (id TEXT UNIQUE, videoId TEXT, title TEXT, thumbnailUrl TEXT, downloadPath TEXT, tempFileName TEXT, fileName TEXT, service INTEGER, videoFormat INTEGER, priority INTEGER, saveAsAudio INTEGER, size TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS youtubeAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS dailymotionAccounts (name TEXT UNIQUE, username TEXT, token TEXT, refresh TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS vimeoAccounts (name TEXT UNIQUE, username TEXT, token TEXT, secret TEXT, active INTEGER)");
        m_database.exec("CREATE TABLE IF NOT EXISTS facebookAccount (name TEXT UNIQUE, token TEXT)");
        m_database.exec("CREATE TABLE IF NOT EXISTS twitterAccount (name TEXT UNIQUE, token TEXT, secret TEXT)");
        m_database.close();
    }
    else {
        qWarning() << m_database.lastError().text();
    }
}

void Database::restoreAccounts() {
    this->restoreYouTubeAccount();
    this->restoreDailymotionAccount();
    this->restoreVimeoAccount();
    this->restoreFacebookAccount();
    this->restoreTwitterAccount();
}

bool Database::setYouTubeUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

bool Database::setYouTubeAccessToken(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE youtubeAccounts SET token = ? WHERE active = 1");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store YouTube account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store YouTube account details"));

        return false;
    }
}

void Database::restoreYouTubeAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh FROM youtubeAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotYouTubeAccount(user, token, refresh);
        }
    }
}

bool Database::setDailymotionUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

bool Database::setDailymotionAccessToken(const QString &token, const QString &refresh) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE dailymotionAccounts SET token = ?, refresh = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(refresh);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Dailymotion account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Dailymotion account details"));

        return false;
    }
}

void Database::restoreDailymotionAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString refresh;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, refresh FROM dailymotionAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                refresh = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotDailymotionAccount(user, token, refresh);
        }
    }
}

bool Database::setVimeoUsername(const QString &user) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET username = ? WHERE active = 1");
        query.addBindValue(user);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

bool Database::setVimeoAccessToken(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("UPDATE vimeoAccounts SET token = ?, secret = ? WHERE active = 1");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store Vimeo account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store Vimeo account details"));

        return false;
    }
}

void Database::restoreVimeoAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString user;
        QString token;
        QString secret;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT username, token, secret FROM vimeoAccounts WHERE active = 1");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                user = query.value(0).toString();
                token = query.value(1).toString();
                secret = query.value(2).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotVimeoAccount(user, token, secret);
        }
    }
}

bool Database::storeFacebookAccount(const QString &token) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO facebookAccount VALUES (?, ?)");
        query.addBindValue("facebook");
        query.addBindValue(token);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store facebook account details"));
        }
        else {
            emit gotFacebookAccount(token);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store facebook account details"));

        return false;
    }
}

bool Database::deleteFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM facebookAccount WHERE name = ?");
        query.addBindValue("facebook");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("facebook account deleted. Please visit the facebook website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete facebook account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete facebook account details"));

        return false;
    }
}

void Database::restoreFacebookAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token FROM facebookAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotFacebookAccount(token);
        }
    }
}

bool Database::storeTwitterAccount(const QString &token, const QString &secret) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR REPLACE INTO twitterAccount VALUES (?, ?, ?)");
        query.addBindValue("twitter");
        query.addBindValue(token);
        query.addBindValue(secret);

        bool success = query.exec();

        if (!success) {
            emit error(tr("Database error. Unable to store twitter account details"));
        }
        else {
            emit gotTwitterAccount(token, secret);
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to store twitter account details"));

        return false;
    }
}

bool Database::deleteTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM twitterAccount WHERE name = ?");
        query.addBindValue("twitter");

        bool success = query.exec();

        if (success) {
            emit gotFacebookAccount(QString());
            emit info(tr("twitter account deleted. Please visit the twitter website to revoke access"));
        }
        else {
            emit error(tr("Database error. Unable to delete twitter account details"));
        }

        m_database.close();

        return success;
    }
    else {
        emit error(tr("Database error. Unable to delete twitter account details"));

        return false;
    }
}

void Database::restoreTwitterAccount() {
    if ((m_database.isOpen()) || (m_database.open())) {
        QString token;
        QString secret;
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT token, secret FROM twitterAccount");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                token = query.value(0).toString();
                secret = query.value(1).toString();
            }
        }

        m_database.close();

        if (!token.isEmpty()) {
            emit gotTwitterAccount(token, secret);
        }
    }
}

bool Database::storeDownload(QSharedPointer<TransferItem> transfer) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("INSERT OR IGNORE INTO storedDownloads VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        query.addBindValue(transfer.data()->id());
        query.addBindValue(transfer.data()->videoId());
        query.addBindValue(transfer.data()->title());
        query.addBindValue(transfer.data()->thumbnailUrl());
        query.addBindValue(transfer.data()->downloadPath());
        query.addBindValue(transfer.data()->tempFileName());
        query.addBindValue(transfer.data()->fileName());
        query.addBindValue(Services::VideoService(transfer.data()->service()));
        query.addBindValue(Videos::VideoFormat(transfer.data()->videoFormat()));
#ifdef MEEGO_EDITION_HARMATTAN
        query.addBindValue(Transfers::TransferPriority(Transfers::NormalPriority));
#else
        query.addBindValue(Transfers::TransferPriority(transfer.data()->priority()));
#endif
        query.addBindValue(transfer.data()->saveAsAudio());
        query.addBindValue(0);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::removeStoredDownload(const QString &id) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare("DELETE FROM storedDownloads WHERE id = ?");
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

bool Database::updateStoredDownload(const QString &id, const QString &key, const QVariant &value) {
    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        query.prepare(QString("UPDATE storedDownloads SET %1 = ? WHERE id = ?").arg(key));
        query.addBindValue(value);
        query.addBindValue(id);
        bool success = query.exec();
        m_database.close();

        return success;
    }

    return false;
}

QList< QSharedPointer<TransferItem> > Database::getStoredDownloads() {
    QList< QSharedPointer<TransferItem> > transfers;

    if ((m_database.isOpen()) || (m_database.open())) {
        QSqlQuery query;
        QSqlRecord record;
        query.exec("SELECT * FROM storedDownloads");
        record = query.record();

        if (record.count() > 0) {
            while (query.next()) {
                QSharedPointer<TransferItem> transfer(new TransferItem);
                transfer.data()->setId(query.value(0).toString());
                transfer.data()->setVideoId(query.value(1).toString());
                transfer.data()->setTitle(query.value(2).toString());
                transfer.data()->setThumbnailUrl(query.value(3).toUrl());
                transfer.data()->setDownloadPath(query.value(4).toString());
                transfer.data()->setTempFileName(query.value(5).toString());
                transfer.data()->setFileName(query.value(6).toString());
                transfer.data()->setService(static_cast<Services::VideoService>(query.value(7).toInt()));
                transfer.data()->setVideoFormat(static_cast<Videos::VideoFormat>(query.value(8).toInt()));
#ifndef MEEGO_EDITION_HARMATTAN
                transfer.data()->setPriority(static_cast<Transfers::TransferPriority>(query.value(9).toInt()));
#endif
                transfer.data()->setSaveAsAudio(query.value(10).toBool());
                transfer.data()->setSize(query.value(11).toLongLong());

                QFileInfo file(transfer.data()->downloadPath() + transfer.data()->tempFileName());

                if ((file.size() > 0) && (transfer.data()->size() > 0)) {
                    transfer.data()->setProgress(file.size() * 100 / transfer.data()->size());
                }

                transfers.append(transfer);
            }
        }

        m_database.close();
    }

    return transfers;
}
