#include "youtubecommentlistmodel.h"
#include "feedurls.h"
#include "youtube.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

YouTubeCommentListModel::YouTubeCommentListModel(QObject *parent) :
    AbstractCommentListModel(parent),
    m_loading(true),
    m_replyRow(0),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    this->connect(YouTube::instance(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
    this->connect(YouTube::instance(), SIGNAL(gotComment(QSharedPointer<CommentItem>)), this, SLOT(insertInitialComment(QSharedPointer<CommentItem>)));
}

void YouTubeCommentListModel::clear() {
    AbstractCommentListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void YouTubeCommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

void YouTubeCommentListModel::setVideoId(const QString &id) {
    if (id != this->videoId()) {
        m_videoId = id;
        this->setFeed(QUrl(QString("%1/%2/comments?v=2.1&max-results=30").arg(YOUTUBE_VIDEOS_BASE_URL).arg(id)));
    }
}

void YouTubeCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setVideoId(id);
    }

    this->setLoading(true);
    QUrl feed = this->feed();
    feed.addQueryItem("fields", YOUTUBE_COMMENT_FIELDS);
    QNetworkReply *reply = YouTube::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void YouTubeCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getComments()));
        this->connect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        YouTube::instance()->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            CommentItem *comment = new CommentItem;
            comment->loadYouTubeComment(entries.at(i));
            QMetaObject::invokeMethod(this, "appendComment", Qt::QueuedConnection, Q_ARG(QSharedPointer<CommentItem>, QSharedPointer<CommentItem>(comment)));
        }

        QTimer::singleShot(1000, this, SLOT(stopLoading()));
        this->setMoreResults((!doc.namedItem("feed").namedItem("link").isNull()) && (!entries.isEmpty()));
        this->setOffset(this->offset() + 30);

        this->disconnect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getComments()));
        this->disconnect(YouTube::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void YouTubeCommentListModel::getInitialComment(int row) {
    m_replyRow = row;
    QSharedPointer<CommentItem> comment = this->get(row);
    comment.data()->setLoading(true);
    YouTube::instance()->getFullComment(comment.data()->videoId(), comment.data()->replyId());
    this->connect(YouTube::instance(), SIGNAL(error(QString)), this, SLOT(onCommentError()));
    emit dataChanged(this->index(row), this->index(row));
}

void YouTubeCommentListModel::insertInitialComment(QSharedPointer<CommentItem> comment) {
    this->disconnect(YouTube::instance(), SIGNAL(error(QString)), this, SLOT(onCommentError()));

    if (comment.data()->videoId() == this->videoId()) {
        QSharedPointer<CommentItem> reply = this->get(m_replyRow);
        reply.data()->setLoading(false);
        reply.data()->setIndent(reply.data()->indent() + 1);

        int i = m_replyRow + 1;
        int indent = reply.data()->indent();
        bool isReply = true;

        while ((i < m_list.size()) && (isReply)) {
            reply = this->get(i);
            isReply = (reply.data()->indent() >= indent);

            if (isReply) {
                reply.data()->setIndent(reply.data()->indent() + 1);
                emit dataChanged(this->index(i), this->index(i));
            }

            i++;
        }

        emit dataChanged(index(m_replyRow), index(i));
        this->insertComment(m_replyRow, comment);
        emit countChanged(this->rowCount());
    }
}

void YouTubeCommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void YouTubeCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if (!comment.data()->replyId().isEmpty()) {
        comment.data()->setIndent(this->data(this->index(m_replyRow), IndentRole).toInt() + 1);

        if (m_list.size() > m_replyRow) {
            this->insertComment(m_replyRow + 1, comment);
        }
        else {
            this->appendComment(comment);
        }
    }
    else {
        this->insertComment(0, comment);
    }

    emit countChanged(this->rowCount());
}

void YouTubeCommentListModel::onCommentError() {
    if (m_replyRow < m_list.size()) {
        this->get(m_replyRow).data()->setLoading(false);
        emit dataChanged(this->index(m_replyRow), this->index(m_replyRow));
    }

    this->disconnect(YouTube::instance(), SIGNAL(error(QString)), this, SLOT(onCommentError()));
}
