#include "vimeocommentlistmodel.h"
#include "feedurls.h"
#include "vimeo.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>

VimeoCommentListModel::VimeoCommentListModel(QObject *parent) :
    AbstractCommentListModel(parent),
    m_loading(true),
    m_replyRow(0),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
    this->connect(Vimeo::instance(), SIGNAL(commentAdded(QSharedPointer<CommentItem>)), this, SLOT(onCommentAdded(QSharedPointer<CommentItem>)));
}

void VimeoCommentListModel::clear() {
    AbstractCommentListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void VimeoCommentListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMoreComments();
    }
}

void VimeoCommentListModel::setVideoId(const QString &id) {
    if (id != this->videoId()) {
        m_videoId = id;
        QUrl feed(VIMEO_BASE_URL);
        feed.addQueryItem("method", "vimeo.videos.comments.getList");
        feed.addQueryItem("video_id", id);
        feed.addQueryItem("per_page", "30");
        this->setFeed(feed);
    }
}

void VimeoCommentListModel::getComments(const QString &id) {
    if (!id.isEmpty()) {
        this->setVideoId(id);
    }

    this->setLoading(true);
    QNetworkReply *reply = Vimeo::instance()->createReply(this->feed(), this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addComments()));
}

void VimeoCommentListModel::addComments() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("comment");

        for (int i = 0; i < entries.size(); i++) {
            CommentItem *comment = new CommentItem;
            comment->loadVimeoComment(entries.at(i).toElement());
            QMetaObject::invokeMethod(this, "appendComment", Qt::QueuedConnection, Q_ARG(QSharedPointer<CommentItem>, QSharedPointer<CommentItem>(comment)));
        }

        QDomElement statNode = responseNode.firstChildElement("comments");

        QTimer::singleShot(1000, this, SLOT(stopLoading()));
        this->setMoreResults(statNode.attribute("total").toInt() > this->rowCount());
        this->setOffset(statNode.attribute("page").toInt() + 1);

        emit countChanged(this->rowCount());
    }
    else {
        this->setLoading(false);
        this->setError(true);
    }

    reply->deleteLater();
}

void VimeoCommentListModel::getMoreComments() {
    if ((this->moreResults()) && (!this->loading())) {
        this->getComments();
    }
}

void VimeoCommentListModel::onCommentAdded(QSharedPointer<CommentItem> comment) {
    if (!comment.data()->replyId().isEmpty()) {
        comment.data()->setIndent(this->data(this->index(m_replyRow), IndentRole).toInt() + 1);

        if (m_list.size() > m_replyRow) {
            this->insertComment(m_replyRow + 1, comment);
        }
        else {
            this->appendComment(comment);
        }
    }
    else {
        this->insertComment(0, comment);
    }

    emit countChanged(this->rowCount());
}

void VimeoCommentListModel::onCommentError() {
    if (m_replyRow < m_list.size()) {
        this->get(m_replyRow).data()->setLoading(false);
        emit dataChanged(this->index(m_replyRow), this->index(m_replyRow));
    }

    this->disconnect(Vimeo::instance(), SIGNAL(error(QString)), this, SLOT(onCommentError()));
}
