#include "abstractcommentlistmodel.h"
#ifndef QML_USER_INTERFACE
#include "thumbnailcache.h"
#endif

AbstractCommentListModel::AbstractCommentListModel(QObject *parent) :
    QAbstractListModel(parent)
  #ifndef QML_USER_INTERFACE
    ,m_cache(new ThumbnailCache),
    m_avatarPressedRow(-1),
    m_loaderPressedRow(-1)
  #endif
{
    QHash<int, QByteArray> roles;
    roles[IdRole] = "id";
    roles[BodyRole] = "body";
    roles[ArtistRole] = "artist";
    roles[ArtistIdRole] = "artistId";
    roles[ArtistAvatarUrlRole] = "artistAvatarUrl";
    roles[DateRole] = "date";
    roles[VideoIdRole] = "videoId";
    roles[ReplyIdRole] = "replyId";
    roles[IndentRole] = "indent";
    roles[LoadingRole] = "loading";
#ifndef QML_USER_INTERFACE
    roles[ArtistAvatarRole] = "artistAvatar";
    roles[AvatarPressedRole] = "avatarPressed";
    roles[LoaderPressedRole] = "loaderPressed";
#endif
    this->setRoleNames(roles);
#ifndef QML_USER_INTERFACE
    this->connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
#endif
}

AbstractCommentListModel::~AbstractCommentListModel() {
    m_list.clear();
#ifndef QML_USER_INTERFACE
    delete m_cache;
    m_cache = 0;
#endif
}

void AbstractCommentListModel::clear() {
    this->beginResetModel();
    m_list.clear();
    this->endResetModel();
}

int AbstractCommentListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent)

    return m_list.size();
}

QVariant AbstractCommentListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case BodyRole:
        return m_list.at(index.row()).data()->body();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case ArtistIdRole:
        return m_list.at(index.row()).data()->artistId();
#ifndef QML_USER_INTERFACE
    case ArtistAvatarRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->artistAvatarUrl(), QSize(50, 50));
    case AvatarPressedRole:
        return m_avatarPressedRow == index.row();
    case LoaderPressedRole:
        return m_loaderPressedRow == index.row();
#endif
    case ArtistAvatarUrlRole:
        return m_list.at(index.row()).data()->artistAvatarUrl();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case VideoIdRole:
        return m_list.at(index.row()).data()->videoId();
    case ReplyIdRole:
        return m_list.at(index.row()).data()->replyId();
    case IndentRole:
        return m_list.at(index.row()).data()->indent();
    case LoadingRole:
        return m_list.at(index.row()).data()->loading();
    default:
        return QVariant();
    }
}

QVariant AbstractCommentListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row), this->roleNames().key(role));
}

#ifndef QML_USER_INTERFACE
bool AbstractCommentListModel::setData(const QModelIndex &index, const QVariant &value, int role) {
    switch (role) {
    case AvatarPressedRole:
        m_avatarPressedRow = value.toInt();
        break;
    case LoaderPressedRole:
        m_loaderPressedRow = value.toInt();
        break;
    default:
        return false;
    }

    emit dataChanged(index, index);
    return true;
}
#endif

QSharedPointer<CommentItem> AbstractCommentListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<CommentItem>(m_list.at(row));
    }

    return QSharedPointer<CommentItem>();
}

#ifdef QML_USER_INTERFACE
CommentItem* AbstractCommentListModel::getFromQML(int row) const {

    if ((row >= 0) && (row < m_list.size())) {
        return m_list.at(row).data();
    }

    return 0;
}
#endif

void AbstractCommentListModel::insertComment(int row, QSharedPointer<CommentItem> comment) {
    this->beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, comment);
    this->endInsertRows();
}

void AbstractCommentListModel::appendComment(QSharedPointer<CommentItem> comment) {
    this->beginInsertRows(QModelIndex(), this->rowCount(), this->rowCount());
    m_list.append(comment);
    this->endInsertRows();
}

bool AbstractCommentListModel::removeComment(const QVariant &value, int role) {
    QModelIndexList indexes = this->match(this->index(0), role, value, 1, Qt::MatchExactly);

    if (indexes.isEmpty()) {
        return false;
    }

    return this->removeRow(indexes.first().row());
}

bool AbstractCommentListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent)

    if ((row >= 0) && (row < m_list.size())) {
        this->beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        this->endRemoveRows();

        return true;
    }

    return false;
}

#ifndef QML_USER_INTERFACE
void AbstractCommentListModel::onThumbnailReady() {
    emit dataChanged(this->index(0), this->index(this->rowCount() - 1));
}
#endif
