#include "events.h"
#include "feedurls.h"
#include "json.h"
#include "alarm.h"
#include <QUrl>
#include <QSettings>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QDBusInterface>
#include <QDomDocument>
#include <QDomElement>
#include <QCoreApplication>

using namespace QtJson;

Events::Events(QObject *parent) :
    QObject(parent),
    m_nam(new QNetworkAccessManager(this)),
    m_database(new Database(this)),
    m_youtube(new YouTube(this)),
    m_dailymotion(new Dailymotion(this)),
    m_initialized(false)
{
}

void Events::init() {
    m_youtube->setNetworkAccessManager(m_nam);
    m_dailymotion->setNetworkAccessManager(m_nam);
    connect(m_database, SIGNAL(gotYouTubeAccount(QString,QString,QString)), m_youtube, SLOT(setAccount(QString,QString,QString)));
    connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), m_database, SLOT(setYouTubeAccessToken(QString)));
    connect(m_database, SIGNAL(gotDailymotionAccount(QString,QString,QString)), m_dailymotion, SLOT(setAccount(QString,QString,QString)));
    connect(m_dailymotion, SIGNAL(accessTokenRefreshed(QString,QString)), m_database, SLOT(setDailymotionAccessToken(QString,QString)));
    connect(m_youtube, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getYouTubeEvents()));
    connect(m_dailymotion, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getDailymotionEvents()));
    connect(m_youtube, SIGNAL(refreshError()), this, SLOT(syncFailed()));
    connect(m_dailymotion, SIGNAL(refreshError()), this, SLOT(syncFailed()));
    connect(m_database, SIGNAL(databaseError()), this, SLOT(syncFailed()));

    m_database->restoreAccounts();
    m_initialized = true;
}

void Events::enableSync() {
    qint64 cookie = add_alarm();

    if (cookie > 0) {
        QSettings settings("Marxian", "cuteTube");
        settings.setValue("Events/eventsAlarmCookie", cookie);
        settings.setValue("Events/eventFeedEnabled", true);
        QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
        iface.call(QDBus::Block, "addRefreshAction", "cutetube", "dbus:session com.maemo.cuteTubeEvents / com.maemo.cuteTubeEvents startSync");
    }

    QCoreApplication::instance()->quit();
}

void Events::disableSync() {
    qint64 cookie = QSettings("Marxian", "cuteTube").value("Events/eventsAlarmCookie", 0).toLongLong();

    if (delete_alarm(cookie)) {
        QSettings settings("Marxian", "cuteTube");
        settings.setValue("Events/eventsAlarmCookie", 0);
        settings.setValue("Events/lastRun", 0);
        settings.setValue("Events/eventFeedEnabled", false);
        QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");
        iface.call(QDBus::Block, "removeEventsBySourceName", "cutetube");
        iface.call(QDBus::Block, "removeRefreshAction", "cutetube");
    }

    QCoreApplication::instance()->quit();
}

void Events::startSync() {
    if (!m_initialized) {
        init();
    }

    m_lastRun = QDateTime::fromMSecsSinceEpoch(QSettings("Marxian", "cuteTube").value("Events/lastRun", 0).toLongLong()).toLocalTime();
    m_thisRun = QDateTime::currentDateTime().toLocalTime();

    getEvents();
}

void Events::abortSync() {
    QCoreApplication::instance()->quit();
}

void Events::syncSuccess() {
    QSettings("Marxian", "cuteTube").setValue("Events/lastRun", m_thisRun.toMSecsSinceEpoch());
    QCoreApplication::instance()->quit();
}

void Events::syncFailed() {
    QCoreApplication::instance()->quit();
}

void Events::getEvents() {
    if (m_youtube->userSignedIn()) {
        getYouTubeEvents();
    }
    else if (m_dailymotion->userSignedIn()) {
        getDailymotionEvents();
    }
}

void Events::getYouTubeEvents() {
    QUrl url(YOUTUBE_NEW_SUBSCRIPTION_VIDEOS_FEED);
    url.addQueryItem("fields", YOUTUBE_VIDEO_FIELDS);
    QNetworkReply *reply = m_youtube->createReply(url);
    connect(reply, SIGNAL(finished()), this, SLOT(processYouTubeEvents()));
}

void Events::processYouTubeEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        syncFailed();
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        m_youtube->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            QDomNode entry = entries.at(i);
            QDomElement mediaGroup = entry.firstChildElement("media:group");
            QDateTime date = QDateTime::fromString(mediaGroup.firstChildElement("yt:uploaded").text(), Qt::ISODate).toLocalTime();

            if (date > m_lastRun) {
                Event event;
                event.setId(mediaGroup.firstChildElement("yt:videoid").text());
                event.setSourceName("cutetube");
                event.setSourceDisplayName("cuteTube");
                event.setIconPath("/usr/share/icons/hicolor/64x64/apps/cutetube64.png");
                event.setTitle(mediaGroup.firstChildElement("media:title").text());
                event.setBody(mediaGroup.firstChildElement("media:description").text());
                event.setImagePathOne(mediaGroup.firstChildElement("media:thumbnail").attribute("url").section('/', 0, -2).append("/mqdefault.jpg"));
                event.setFooter(mediaGroup.firstChildElement("media:credit").attribute("yt:display"));
                event.setTimestamp(mediaGroup.firstChildElement("yt:uploaded").text());
                event.setAction("dbus:session com.maemo.cuteTube / com.maemo.cuteTube displayVideo http://youtu.be/" + event.id());

                m_events.append(event);
            }
        }

        if (m_dailymotion->userSignedIn()) {
            getDailymotionEvents();
        }
        else {
            addEvents();
        }
    }

    reply->deleteLater();
}

void Events::getDailymotionEvents() {
    QUrl url(DAILYMOTION_NEW_SUBSCRIPTION_VIDEOS_FEED);
    url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    QNetworkReply *reply = m_dailymotion->createReply(url);
    connect(reply, SIGNAL(finished()), this, SLOT(processDailymotionEvents()));
}

void Events::processDailymotionEvents() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(sender());

    if (!reply) {
        syncFailed();
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        m_dailymotion->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            syncFailed();
            return;
        }

        QVariantList entries = res.value("list").toList();

        for (int i = 0; i < entries.size(); i++) {
            QVariantMap entry = entries.at(i).toMap();
            QDateTime date = QDateTime::fromMSecsSinceEpoch(entry.value("created_time").toLongLong() * 1000).toLocalTime();

            if (date > m_lastRun) {
                Event event;
                event.setId(entry.value("id").toString());
                event.setSourceName("cutetube");
                event.setSourceDisplayName("cuteTube");
                event.setIconPath("/usr/share/icons/hicolor/64x64/apps/cutetube64.png");
                event.setTitle(entry.value("title").toString());
                event.setBody(entry.value("description").toString());
                event.setImagePathOne(entry.value("thumbnail_medium_url").toString());
                event.setFooter(entry.value("owner.username").toString());
                event.setTimestamp(QString::number(entry.value("created_time").toLongLong() * 1000));
                event.setAction("dbus:session com.maemo.cuteTube / com.maemo.cuteTube displayVideo http://dai.ly/" + event.id());

                m_events.append(event);
            }
        }

        addEvents();
    }

    reply->deleteLater();
}

void Events::addEvents() {
    QDBusInterface iface("com.maemo.eventFeed", "/", "com.maemo.eventFeed");

    while (!m_events.isEmpty()) {
        QVariantList args;
        Event event = m_events.takeFirst();
        args.append(event.sourceName());
        args.append(event.sourceDisplayName());
        args.append(event.iconPath());
        args.append(event.title());
        args.append(event.body());
        args.append(QStringList() << event.imagePathOne());
        args.append(event.footer());
        args.append(event.timestamp());
        args.append(event.action());

        iface.callWithArgumentList(QDBus::Block, "addEvent", args);
    }

    syncSuccess();
}
