#include "xtubevideolistwindow.h"
#include "xtubevideolistdelegate.h"
#include "videoselectionwindow.h"
#include "videoplaybackwindow.h"
#include <QMenuBar>

XTubeVideoListWindow::XTubeVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new XTubeVideoListModel(m_session, this)),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    setCentralWidget(m_listView);
    m_listView->setModel(m_model);
    m_listView->setItemDelegate(new XTubeVideoListDelegate(m_listView));
    m_noResultsLabel->hide();

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    onOrientationChanged();
}

void XTubeVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void XTubeVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void XTubeVideoListWindow::search(const QString &query, Services::VideoService service, Queries::QueryOrder order, Queries::TimeFilter time, Queries::DurationFilter duration) {
    setWindowTitle(tr("Search") + QString(" ('%1')").arg(query));
    m_model->search(query, service, order, time, duration);
}

void XTubeVideoListWindow::onLoadingChanged() {
    toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void XTubeVideoListWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::NoService, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void XTubeVideoListWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::NoService, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void XTubeVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_videoDownloadContextAction);
}

void XTubeVideoListWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void XTubeVideoListWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void XTubeVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        if (m_session->settings()->mediaPlayer() == "cutetube") {
            VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
            window->show();
        }
        else {
            m_session->urlGrabber()->getVideoUrl(static_cast<Services::VideoService>(index.data(AbstractVideoListModel::ServiceRole).toInt()), index.data(AbstractVideoListModel::IdRole).toString());
        }
    }
}
