#include "vimeovideoinfowindow.h"
#include "sharingdialog.h"
#include "sharetofacebookdialog.h"
#include "sharetotwitterdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "vimeovideolistwindow.h"
#include "videoplaybackwindow.h"
#include <QMenuBar>

VimeoVideoInfoWindow::VimeoVideoInfoWindow(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    StackedWindow(session, parent),
    m_video(video),
    m_queueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(addVideoToPlaybackQueue()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_shareAction(menuBar()->addAction(tr("Share"), this, SLOT(showShareDialog()))),
    m_thumbnail(new ThumbnailLabel(m_video.data()->duration(), this)),
    m_avatar(new ThumbnailLabel(this)),
    m_titleLabel(new QLabel(m_video.data()->title(), this)),
    m_dateLabel(new QLabel(QString("%1 - %2").arg(m_video.data()->date()).arg(m_video.data()->artist()), this)),
    m_ratingLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Likes")).arg(m_video.data()->likes()), this)),
    m_viewsLabel(new QLabel(QString("%1   <font color='#4d4d4d'>%2</font>").arg(tr("Plays")).arg(m_video.data()->viewCount()), this)),
    m_descriptionLabel(new QLabel(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()), this)),
    m_tagsLabel(new QLabel(tagsToString(m_video.data()->tags()), this)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(0)
{
    setWindowTitle(tr("Info"));
    setCentralWidget(new QWidget(this));

    m_grid = new QGridLayout(centralWidget());

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);
    QFont smallFont;
    smallFont.setPixelSize(20);
    m_avatar->setFixedSize(60, 60);
    m_avatar->setInteractive(false);
    m_titleLabel->setWordWrap(true);
    m_dateLabel->setWordWrap(true);
    m_dateLabel->setFont(smallFont);
    m_ratingLabel->setFont(smallFont);
    m_viewsLabel->setFont(smallFont);
    m_descriptionLabel->setWordWrap(true);
    m_descriptionLabel->setFont(smallFont);
    m_tagsLabel->setWordWrap(true);
    m_tagsLabel->setFont(smallFont);

    grid->addWidget(m_titleLabel, 0, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_avatar, 1, 0);
    grid->addWidget(m_dateLabel, 1, 1, Qt::AlignTop);
    grid->addWidget(m_ratingLabel, 2, 0, 1, 2);
    grid->addWidget(m_viewsLabel, 3, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_descriptionLabel, 4, 0, 1, 2, Qt::AlignTop);
    grid->addWidget(m_tagsLabel, 5, 0, 1, 2);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_thumbnail, SIGNAL(clicked()), this, SLOT(playVideo()));
    connect(m_tagsLabel, SIGNAL(linkActivated(QString)), this, SLOT(search(QString)));

    if (m_video.data()->description().isEmpty()) {
        m_session->vimeo()->getVideoInfo(m_video);
        connect(m_session->vimeo(), SIGNAL(gotVideoInfo()), this, SLOT(setVideoInfo()));
    }
    else {
        setThumbnails();
    }

    onOrientationChanged();
}

void VimeoVideoInfoWindow::setLandscapeLayout() {
    m_thumbnail->setFixedSize(280, 210);
    m_scrollArea->setFixedHeight(415);
    m_grid->setContentsMargins(10, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 0, 1, Qt::AlignTop);
    }
}

void VimeoVideoInfoWindow::setPortraitLayout() {
    m_thumbnail->setFixedSize(460, 345);
    m_scrollArea->setFixedHeight(380);
    m_grid->setContentsMargins(0, 10, 0, 0);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_thumbnail, 0, 0, Qt::AlignTop | Qt::AlignHCenter);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
    else {
        m_grid->removeWidget(m_scrollArea);
        m_grid->addWidget(m_scrollArea, 1, 0, Qt::AlignTop);
    }
}

QString VimeoVideoInfoWindow::tagsToString(QStringList tags) {
    QString tagString("Tags<br>");

    if (tags.isEmpty()) {
        tagString.append(QString("<p style='color:#4d4d4d;'>%1</p>").arg(tr("No tags")));
    }
    else {
        while (!tags.isEmpty()) {
            tagString.append(QString("<a href='%1'>%1</a>   ").arg(tags.takeFirst()));
        }
    }

    return tagString;
}

void VimeoVideoInfoWindow::setThumbnails() {
    QImage thumbnail = m_session->thumbnailCache()->thumbnail(m_video.data()->largeThumbnailUrl());
    QImage avatar = m_session->thumbnailCache()->thumbnail(m_video.data()->artistAvatarUrl(), QSize(60, 60));

    if ((thumbnail.isNull()) || (avatar.isNull())) {
        connect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    else {
        disconnect(m_session->thumbnailCache(), SIGNAL(thumbnailReady()), this, SLOT(setThumbnails()));
    }
    if (!thumbnail.isNull()) {
        m_thumbnail->setImage(thumbnail);
    }
    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
    }
}

void VimeoVideoInfoWindow::setVideoInfo() {
    setThumbnails();
    m_descriptionLabel->setText(QString("Description<br><font color='#4d4d4d'>%1</font>").arg(m_video.data()->description().isEmpty() ? tr("No description") : m_video.data()->description()));
    m_tagsLabel->setText(tagsToString(m_video.data()->tags()));

    disconnect(m_session->vimeo(), SIGNAL(gotVideoInfo()), this, SLOT(setVideoInfo()));
}

void VimeoVideoInfoWindow::showShareDialog() {
    SharingDialog *dialog = new SharingDialog(this);
    dialog->open();

    connect(dialog, SIGNAL(valueChanged(Services::SharingService)), this, SLOT(shareVideo(Services::SharingService)));
}

void VimeoVideoInfoWindow::shareVideo(Services::SharingService service) {
    switch (service) {
    case Services::Facebook:
        showFacebookDialog();
        break;
    case Services::Twitter:
        showTwitterDialog();
        break;
    case Services::Email:
        m_session->sharing()->shareViaEmail(m_video);
        break;
    case Services::Clipboard:
        m_session->sharing()->copyToClipboard(m_video.data()->url());
        break;
    default:
        return;
    }
}

void VimeoVideoInfoWindow::showFacebookDialog() {
    if (m_session->sharing()->userSignedInToFacebook()) {
        ShareToFacebookDialog *dialog = new ShareToFacebookDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        FacebookAuthDialog *dialog = new FacebookAuthDialog(m_session, this);
        dialog->open();
    }
}

void VimeoVideoInfoWindow::showTwitterDialog() {
    if (m_session->sharing()->userSignedInToTwitter()) {
        ShareToTwitterDialog *dialog = new ShareToTwitterDialog(m_session, m_video, this);
        dialog->open();
    }
    else {
        TwitterAuthDialog *dialog = new TwitterAuthDialog(m_session, this);
        dialog->open();
    }
}

void VimeoVideoInfoWindow::addVideoToPlaybackQueue() {
    m_session->playbackQueue()->addVideo(m_video);
}

void VimeoVideoInfoWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video);
}

void VimeoVideoInfoWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_video, true);
}

void VimeoVideoInfoWindow::playVideo() {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_video, this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getVimeoVideoUrl(m_video.data()->videoId());
    }
}

void VimeoVideoInfoWindow::search(const QString &query) {
    VimeoVideoListWindow *window = new VimeoVideoListWindow(m_session, this);
    window->search(query, m_session->settings()->searchOrder());
    window->show();
}
