#include "videoselectionwindow.h"
#include "videoselectiondelegate.h"
#include "xtubevideoselectiondelegate.h"
#include "youtubeplaylistdialog.h"
#include "dailymotionplaylistdialog.h"

VideoSelectionWindow::VideoSelectionWindow(Session *session, AbstractVideoListModel *model, Services::VideoService service, const QString &text, const char* slot, QWidget *parent) :
    QWidget(parent),
    m_session(session),
    m_model(model),
    m_service(service),
    m_listView(new ListView(this)),
    m_editBar(new QMaemo5EditBar(tr("Select videos"), this)),
    m_acceptButton(new QPushButton(text, this)),
    m_vbox(new QVBoxLayout(this))
{
    setWindowFlags(Qt::Window);
    setWindowState(Qt::WindowFullScreen);
    setAttribute(Qt::WA_Maemo5StackedWindow, true);
    setAttribute(Qt::WA_DeleteOnClose, true);

    m_listView->setModel(m_model);

    if ((service == Services::YouTube) || (service == Services::Dailymotion) || (service == Services::Vimeo)) {
        m_listView->setItemDelegate(new VideoSelectionDelegate(m_listView));
    }
    else {
        m_listView->setItemDelegate(new XTubeVideoSelectionDelegate(m_listView));
    }

    m_editBar->addButton(m_acceptButton);
    m_vbox->setContentsMargins(0, 0, 0, 0);
    m_vbox->addWidget(m_editBar);
    m_vbox->addWidget(m_listView);

    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_acceptButton, SIGNAL(clicked()), this, SLOT(onButtonClicked()));
    connect(this, SIGNAL(accepted()), this, slot);
}

VideoSelectionWindow::~VideoSelectionWindow() {
    if (m_model->itemsChecked()) {
        m_model->checkNone();
    }
}

void VideoSelectionWindow::positionAtIndex(const QModelIndex &index) {
    m_listView->scrollTo(index, QListView::PositionAtTop);
}

void VideoSelectionWindow::onItemClicked(const QModelIndex &index) {
    m_model->toggleChecked(index.row());
}

void VideoSelectionWindow::onButtonClicked() {
    if (m_model->itemsChecked()) {
        emit accepted();
    }
}

void VideoSelectionWindow::deleteVideosFromUploads() {
    if (m_service == Services::YouTube) {
        m_session->youtube()->deleteFromUploads(m_model->checkedItems());
    }
    else if (m_service == Services::Dailymotion) {
        m_session->dailymotion()->deleteFromUploads(m_model->checkedItems());
    }

    m_model->checkNone();
}

void VideoSelectionWindow::addVideosToFavourites() {
    if (m_service == Services::YouTube) {
        m_session->youtube()->addToFavourites(m_model->checkedItems());
    }
    else if (m_service == Services::Dailymotion) {
        m_session->dailymotion()->addToFavourites(m_model->checkedItems());
    }

    m_model->checkNone();
}

void VideoSelectionWindow::deleteVideosFromFavourites() {
    if (m_service == Services::YouTube) {
        m_session->youtube()->deleteFromFavourites(m_model->checkedItems());
    }
    else if (m_service == Services::Dailymotion) {
        m_session->dailymotion()->deleteFromFavourites(m_model->checkedItems());
    }

    m_model->checkNone();
}

void VideoSelectionWindow::showPlaylistDialog() {
    if (m_service == Services::YouTube) {
        YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
        dialog->open();

        connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideosToPlaylist(QString)));
        connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideosToNewPlaylist(NewPlaylist)));
    }
    else if (m_service == Services::Dailymotion) {
        DailymotionPlaylistDialog *dialog = new DailymotionPlaylistDialog(m_session, this);
        dialog->open();

        connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideosToPlaylist(QString)));
        connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideosToNewPlaylist(NewPlaylist)));
    }
}

void VideoSelectionWindow::addVideosToPlaylist(const QString &id) {
    if (m_service == Services::YouTube) {
        m_session->youtube()->addToPlaylist(m_model->checkedItems(), id);
    }
    else if (m_service == Services::Dailymotion) {
        m_session->dailymotion()->addToPlaylist(m_model->checkedItems(), id);
    }

    m_model->checkNone();
}

void VideoSelectionWindow::addVideosToNewPlaylist(const NewPlaylist &playlist) {
    if (m_service == Services::YouTube) {
        m_session->youtube()->createPlaylist(playlist, QSharedPointer<VideoItem>(), m_model->checkedItems());
    }
    else if (m_service == Services::Dailymotion) {
        m_session->dailymotion()->createPlaylist(playlist, QSharedPointer<VideoItem>(), m_model->checkedItems());
    }

    m_model->checkNone();
}

void VideoSelectionWindow::addVideosToWatchLaterPlaylist() {
    if (m_service == Services::YouTube) {
        m_session->youtube()->addToWatchLaterPlaylist(m_model->checkedItems());
    }

    m_model->checkNone();
}

void VideoSelectionWindow::addVideoDownloads() {
    m_session->transferManager()->addDownloadTransfers(m_model->checkedItems());
    m_model->checkNone();
}

void VideoSelectionWindow::addAudioDownloads() {
    m_session->transferManager()->addDownloadTransfers(m_model->checkedItems(), true);
    m_model->checkNone();
}

void VideoSelectionWindow::queueVideos() {
    m_session->playbackQueue()->addVideos(m_model->checkedItems());
    m_model->checkNone();
}
