#include "thumbnaillabel.h"
#include "utils.h"
#include <QPainter>
#include <QtCore/qmath.h>

ThumbnailLabel::ThumbnailLabel(QWidget *parent) :
    QWidget(parent),
    m_duration(0),
    m_interactive(true),
    m_pressed(false)
{
}

ThumbnailLabel::ThumbnailLabel(int duration, QWidget *parent) :
    QWidget(parent),
    m_duration(duration),
    m_interactive(true),
    m_pressed(false)
{
}

void ThumbnailLabel::setImage(const QImage &image) {
    m_image = image;
    repaint();
}

void ThumbnailLabel::setDuration(int duration) {
    m_duration = duration;
    repaint();
}

void ThumbnailLabel::mousePressEvent(QMouseEvent *event) {
    if (interactive()) {
        event->accept();
        m_pressed = true;
        repaint();
    }
    else {
        event->ignore();
    }
}

void ThumbnailLabel::mouseReleaseEvent(QMouseEvent *event) {
    if ((interactive()) && (rect().contains(event->pos()))) {
        event->accept();
        m_pressed = false;
        repaint();
        emit clicked();
    }
    else {
        event->ignore();
    }
}

void ThumbnailLabel::leaveEvent(QEvent *event) {
    event->accept();
    m_pressed = false;
    repaint();
}

void ThumbnailLabel::paintEvent(QPaintEvent *event) {
    Q_UNUSED(event);

    QPainter painter(this);

    if (m_pressed) {
        painter.setOpacity(0.5);
    }

    if (!m_image.isNull()) {
        painter.drawImage(rect(), m_image);
    }

    QRect backgroundRect = rect();
    backgroundRect.setBottomRight(QPoint(backgroundRect.right() - 1, backgroundRect.bottom() - 1));

    painter.setPen(QColor("#4d4d4d"));
    painter.drawRect(backgroundRect);

    if (m_duration) {
        int backgroundWidth = m_duration >= 3600 ? qFloor(width() / 2.5) : qFloor(width() / 4);

        backgroundRect.setLeft(backgroundRect.right() - backgroundWidth);
        backgroundRect.setTop(backgroundRect.bottom() - qFloor(height() / 6));

        QFont font;
        font.setPixelSize(qFloor(height() / 8));

        painter.setOpacity(0.5);
        painter.fillRect(backgroundRect, Qt::black);
        painter.setPen(Qt::white);
        painter.setFont(font);
        painter.setOpacity(1);
        painter.drawText(backgroundRect, Qt::AlignCenter, Utils::durationFromSecs(m_duration));
    }
}
