#ifndef SHARING_H
#define SHARING_H

#include "videoitem.h"
#include "videoshare.h"
#include <QObject>
#include <QNetworkAccessManager>
#include <QStringList>
#include <QUrl>
#include <QDateTime>

class Sharing : public QObject {
    Q_OBJECT

public:
    explicit Sharing(QObject *parent = 0);
    ~Sharing();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    inline void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline bool userSignedInToFacebook() { return !facebookToken().isEmpty(); }
    inline bool userSignedInToTwitter() { return (!twitterToken().isEmpty()) && (!twitterTokenSecret().isEmpty()); }
    void shareViaEmail(QSharedPointer<VideoItem> video);
    void copyToClipboard(const QUrl &url);
    void signInToFacebook(const QUrl &response);
    void signOutOfFacebook();
    void signInToTwitter(const QUrl &response);
    void signOutOfTwitter();
    void getTwitterAuthUrl();
    QUrl facebookAuthUrl() const;
    inline QList< QSharedPointer<VideoItem> > * videos() const { return m_videoCache; }
    inline bool busy() const { return m_busy; }

public slots:
    void postToFacebook(const VideoShare &post = VideoShare());
    void postToTwitter(const VideoShare &post = VideoShare());
    void setFacebookAccessToken(const QString &token);
    void setTwitterAccount(const QString &token, const QString &secret);
    void getVideos(bool forceRefresh = false);
    void addVideosToCache(QList< QSharedPointer<VideoItem> > videos);

private slots:
    void facebookPostFinished();
    void twitterPostFinished();
    void getVideosFromFacebook();
    void getVideosFromTwitter();
    void checkFacebookVideos();
    void checkTwitterVideos();
    void checkTwitterAuthUrl();
    void checkIfSignedInToTwitter();

private:
    void clearCache();
    inline QString facebookToken() const { return m_facebookToken; }
    inline QString twitterToken() const { return m_twitterToken; }
    inline void setTwitterAccessToken(const QString &token) { m_twitterToken = token; }
    inline QString twitterTokenSecret() const { return m_twitterSecret; }
    inline void setTwitterTokenSecret(const QString &secret) { m_twitterSecret = secret; }
    inline QString twitterRequestToken() const { return m_twitterRequestToken; }
    inline void setTwitterRequestToken(const QString &token) { m_twitterRequestToken = token; }
    inline QString twitterRequestSecret() const { return m_twitterRequestSecret; }
    inline void setTwitterRequestSecret(const QString &secret) { m_twitterRequestSecret = secret; }
    QByteArray getTwitterOAuthHeader(const QString &method, const QString &url, QMap<QString, QString> params);
    QString lastTweet() const { return m_lastTweetId; }
    void setLastTweet(const QString &id) { m_lastTweetId = id; }
    inline qint64 facebookTimeStamp() const { return m_facebookTimestamp; }
    inline void setFacebookTimeStamp(qint64 timestamp) { m_facebookTimestamp = timestamp; }
    inline bool videoRefreshIsDue() const { return (m_nextVideosRefresh.elapsed() == 0) || (m_nextVideosRefresh.elapsed() > 600000); }
    inline void resetVideoRefreshTime() { m_nextVideosRefresh.restart(); }
    inline void setBusy(bool busy) { m_busy = busy; emit busyChanged(); }

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged();
    void info(const QString &message);
    void warning(const QString &message);
    void userSignedInToFacebookChanged();
    void userSignedInToTwitterChanged();
    void postSuccessful();
    void signedInToFacebook(const QString &token);
    void signedOutOfFacebook();
    void gotTwitterAuthUrl(const QUrl &url);
    void signedInToTwitter(const QString &token, const QString &secret);
    void signedOutOfTwitter();
    void reauthenticateForFacebook();
    void reauthenticateForTwitter();
    void gotYouTubeIds(const QStringList &ids);
    void gotDailymotionIds(const QStringList &ids);
    void videoAddedToCache(int row);
    void videosAdded();

private:
    QNetworkAccessManager *m_nam;
    QString m_facebookClientId;
    QString m_facebookClientSecret;
    QString m_twitterClientId;
    QString m_twitterClientSecret;
    QString m_redirectUri;
    QString m_facebookToken;
    QString m_twitterToken;
    QString m_twitterSecret;
    QString m_twitterRequestToken;
    QString m_twitterRequestSecret;
    qint64 m_facebookTimestamp;
    QString m_lastTweetId;
    QTime m_nextVideosRefresh;
    VideoShare m_post;
    QList< QSharedPointer<VideoItem> > *m_videoCache;
    bool m_busy;
};

#endif // SHARING_H
