#include "playbackqueuewindow.h"
#include "videolistdelegate.h"
#include "youtubevideoinfowindow.h"
#include "dailymotionvideoinfowindow.h"
#include "vimeovideoinfowindow.h"
#include "playbackvideoselectionwindow.h"
#include <QMenuBar>

PlaybackQueueWindow::PlaybackQueueWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_removeAction(menuBar()->addAction(tr("Remove"), this, SLOT(removeVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_removeContextAction(m_contextMenu->addAction(tr("Remove"), this, SLOT(removeVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos queued")), this))
{
    setWindowTitle(tr("Playback queue"));
    setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_session->playbackQueue());
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->setVisible(m_session->playbackQueue()->rowCount() == 0);

    connect(m_session->playbackQueue(), SIGNAL(countChanged()), this, SLOT(onCountChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));

    onOrientationChanged();
}

void PlaybackQueueWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void PlaybackQueueWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void PlaybackQueueWindow::onCountChanged() {
    m_noResultsLabel->setVisible(m_session->playbackQueue()->rowCount() == 0);
}

void PlaybackQueueWindow::removeVideos() {
    PlaybackVideoSelectionWindow *window = new PlaybackVideoSelectionWindow(m_session, tr("Remove"), SLOT(removeVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void PlaybackQueueWindow::addVideoDownloads() {
    PlaybackVideoSelectionWindow *window = new PlaybackVideoSelectionWindow(m_session, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void PlaybackQueueWindow::addAudioDownloads() {
    PlaybackVideoSelectionWindow *window = new PlaybackVideoSelectionWindow(m_session, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void PlaybackQueueWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_removeContextAction);
}

void PlaybackQueueWindow::removeVideo() {
    m_session->playbackQueue()->removeVideo(m_listView->currentIndex().row());
}

void PlaybackQueueWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_session->playbackQueue()->get(m_listView->currentIndex().row()));
}

void PlaybackQueueWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_session->playbackQueue()->get(m_listView->currentIndex().row()), true);
}

void PlaybackQueueWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        Services::VideoService service = static_cast<Services::VideoService>(index.data(AbstractVideoListModel::ServiceRole).toInt());

        switch (service) {
        case Services::YouTube:
            showYouTubeVideoInfo(m_session->playbackQueue()->get(index.row()));
            break;
        case Services::Dailymotion:
            showDailymotionVideoInfo(m_session->playbackQueue()->get(index.row()));
            break;
        case Services::Vimeo:
            showVimeoVideoInfo(m_session->playbackQueue()->get(index.row()));
            break;
        default:
            return;
        }
    }
}

void PlaybackQueueWindow::showYouTubeVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *infoWindow = new YouTubeVideoInfoWindow(m_session, video, this);
    infoWindow->show();
}

void PlaybackQueueWindow::showDailymotionVideoInfo(QSharedPointer<VideoItem> video) {
    DailymotionVideoInfoWindow *infoWindow = new DailymotionVideoInfoWindow(m_session, video, this);
    infoWindow->show();
}

void PlaybackQueueWindow::showVimeoVideoInfo(QSharedPointer<VideoItem> video) {
    VimeoVideoInfoWindow *infoWindow = new VimeoVideoInfoWindow(m_session, video, this);
    infoWindow->show();
}

void PlaybackQueueWindow::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getVideoUrl(static_cast<Services::VideoService>(index.data(AbstractVideoListModel::ServiceRole).toInt()), index.data(AbstractVideoListModel::VideoIdRole).toString());
}
