#include "dailymotionaccountslistmodel.h"
#include <QSqlRecord>
#include <QSqlField>

DailymotionAccountsListModel::DailymotionAccountsListModel(Dailymotion *dailymotion, CookieJar *cookieJar, QObject *parent) :
    QSqlTableModel(parent, QSqlDatabase::database()),
    m_dailymotion(dailymotion),
    m_jar(cookieJar)
{
    QHash<int, QByteArray> roles;
    roles[DisplayNameRole] = "name";
    roles[UsernameRole] = "username";
    roles[AccessTokenRole] = "token";
    roles[RefreshTokenRole] = "refresh";
    roles[CookiesRole] = "cookies";
    roles[ActiveRole] = "active";
    setRoleNames(roles);
    setTable("dailymotionAccounts");
    setEditStrategy(QSqlTableModel::OnManualSubmit);
    select();

    connect(m_dailymotion, SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountSelected(QString,QString,QString)), m_dailymotion, SLOT(setAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountAdded(QString,QString,QString)), m_dailymotion, SLOT(setAccount(QString,QString,QString)));
    connect(this, SIGNAL(accountDeleted()), m_dailymotion, SLOT(signOut()));
    connect(this, SIGNAL(gotCookies(QByteArray)), m_jar, SLOT(setDailymotionCookies(QByteArray)));
    connect(this, SIGNAL(accountAdded(QString,QString,QString)), m_jar, SLOT(clearDailymotionCookies()));
    connect(this, SIGNAL(accountDeleted()), m_jar, SLOT(clearDailymotionCookiesFromDB()));
}

DailymotionAccountsListModel::~DailymotionAccountsListModel() {
    clear();
}

void DailymotionAccountsListModel::setActiveAccount(int row) {
    database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(data(index(row, 0), DisplayNameRole));
    usernameField.setValue(data(index(row, 0), UsernameRole));
    tokenField.setValue(data(index(row, 0), AccessTokenRole));
    refreshField.setValue(data(index(row, 0), RefreshTokenRole));
    cookiesField.setValue(data(index(row, 0), CookiesRole));
    activeField.setValue(1);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(cookiesField);
    rec.append(activeField);

    setRecord(row, rec);

    for (int i = 0; i < rowCount(); i++) {
        if (i != row) {
            nameField.setValue(data(index(i, 0), DisplayNameRole));
            usernameField.setValue(data(index(i, 0), UsernameRole));
            tokenField.setValue(data(index(i, 0), AccessTokenRole));
            refreshField.setValue(data(index(i, 0), RefreshTokenRole));
            cookiesField.setValue(data(index(i, 0), CookiesRole));
            activeField.setValue(0);

            rec.clear();
            rec.append(nameField);
            rec.append(usernameField);
            rec.append(tokenField);
            rec.append(refreshField);
            rec.append(cookiesField);
            rec.append(activeField);

            setRecord(i, rec);
        }
    }

    submitAll();
}

void DailymotionAccountsListModel::switchAccount(int row) {
    setActiveAccount(row);

    database().open();
    QString user = data(index(row, 0), UsernameRole).toString();
    QString token = data(index(row, 0), AccessTokenRole).toString();
    QString refresh = data(index(row, 0), RefreshTokenRole).toString();
    QByteArray cookieString = data(index(row, 0), CookiesRole).toByteArray();

    emit accountSelected(user, token, refresh);
    emit gotCookies(cookieString);
}

QVariant DailymotionAccountsListModel::data(const QModelIndex &idx, int role) const {
    return QSqlTableModel::data(index(idx.row(), role - (Qt::DisplayRole)));
}

void DailymotionAccountsListModel::addAccount(const QString &displayName, const QString &token, const QString &refresh) {
    database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField cookiesField("cookies", QVariant::ByteArray);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(displayName);
    usernameField.setValue(QVariant::String);
    tokenField.setValue(token);
    refreshField.setValue(refresh);
    cookiesField.setValue(QVariant::ByteArray);
    activeField.setValue(0);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(cookiesField);
    rec.append(activeField);
    insertRecord(-1, rec);
    submitAll();

    setActiveAccount(rowCount() -1);

    emit accountAdded(QString(), token, refresh);
}

void DailymotionAccountsListModel::deleteAccount(int row) {
    database().open();
    bool signOut = data(index(row, 0), ActiveRole).toBool();

    if ((removeRow(row)) && (submitAll())) {
        if (signOut) {
            emit accountDeleted();
        }
    }
    else {
        emit warning(tr("Database error. Unable to delete account"));
    }
}
