#include "urlgrabber.h"
#include "cookiejar.h"
#include "json.h"
#include "definitions.h"
#include "utils.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QStringList>
#include <QRegExp>
#include <QScriptEngine>
#include <QDomDocument>
#include <QDomElement>
#include <QDebug>
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

using namespace QtJson;

UrlGrabber* UrlGrabber::self = 0;
QScriptEngine* UrlGrabber::youtubeDecryptionEngine = 0;
QHash<QUrl, QScriptValue> UrlGrabber::youtubeDecryptionCache;

UrlGrabber::UrlGrabber(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_format(Videos::Unknown),
    m_language("en"),
    m_busy(false),
    m_cancelled(false)
{
    if (!self) {
        self = this;
    }

    if (!youtubeDecryptionEngine) {
        youtubeDecryptionEngine = new QScriptEngine;
    }
}

UrlGrabber* UrlGrabber::instance() {
    return !self ? new UrlGrabber : self;
}

void UrlGrabber::setYouTubeFormats(QSet<int> formats) {
    m_youtubeFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::MobileLQ:
            videoFormat.value = 17;
            videoFormat.displayName = "144P AVC1";
            break;
        case Videos::MobileNormal:
            videoFormat.value = 36;
            videoFormat.displayName = "240P AVC1";
            break;
        case Videos::LQ:
            videoFormat.value = 18;
            videoFormat.displayName = QString("%1 AVC1").arg(tr("Normal (upto 360P)"));
            break;
        case Videos::Normal:
            videoFormat.value = 34;
            videoFormat.displayName = "360P FLV";
            break;
        case Videos::HQ:
            videoFormat.value = 35;
            videoFormat.displayName = "480P FLV";
            break;
        case Videos::HD:
            videoFormat.value = 22;
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = 37;
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setYouTubeFormats(): Invalid video format";
            break;
        }

        m_youtubeFormats.append(videoFormat);
    }
}

void UrlGrabber::setDailymotionFormats(QSet<int> formats) {
    m_dailymotionFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::MobileLQ:
        case Videos::MobileNormal:
        case Videos::LQ:
            videoFormat.value = "stream_h264_ld_url";
            videoFormat.displayName = "240P AVC1";
            break;
        case Videos::Normal:
            videoFormat.value = "stream_h264_url";
            videoFormat.displayName = "384P AVC1";
            break;
        case Videos::HQ:
            videoFormat.value = "stream_h264_hq_url";
            videoFormat.displayName = "480P AVC1";
            break;
        case Videos::HD:
            videoFormat.value = "stream_h264_hd_url";
            videoFormat.displayName = "720P AVC1";
            break;
        case Videos::SuperHD:
            videoFormat.value = "stream_h264_hd1080_url";
            videoFormat.displayName = "1080P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setDailymotionFormats(): Invalid video format";
            break;
        }

        m_dailymotionFormats.append(videoFormat);
    }
}

void UrlGrabber::setVimeoFormats(QSet<int> formats) {
    m_vimeoFormats.clear();
    QList<int> formatList = formats.toList();
    qSort(formatList.begin(), formatList.end(), qGreater<int>());

    while (!formatList.isEmpty()) {
        int format = formatList.takeFirst();
        VideoFormat videoFormat;
        videoFormat.format = static_cast<Videos::VideoFormat>(format);

        switch (format) {
        case Videos::MobileLQ:
        case Videos::MobileNormal:
        case Videos::LQ:
            videoFormat.value = "mobile";
            videoFormat.displayName = tr("Mobile");
            break;
        case Videos::Normal:
        case Videos::HQ:
            videoFormat.value = "sd";
            videoFormat.displayName = "360P AVC1";
            break;
        case Videos::HD:
        case Videos::SuperHD:
            videoFormat.value = "hd";
            videoFormat.displayName = "720P AVC1";
            break;
        default:
            qWarning() << "UrlGrabber::setVimeoFormats(): Invalid video format";
            break;
        }

        m_vimeoFormats.append(videoFormat);
    }
}

void UrlGrabber::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;
        emit busyChanged(isBusy);
    }

    if (isBusy) {
        this->setCancelled(false);
        emit busy(message, numberOfOperations);
    }
    else if (!this->cancelled()) {
        emit busyProgressChanged(numberOfOperations);
    }
}

void UrlGrabber::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
    emit currentOperationCancelled();
}

void UrlGrabber::getVideoUrl(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeVideoUrl(id);
        break;
    case Services::Dailymotion:
        this->getDailymotionVideoUrl(id);
        break;
    case Services::Vimeo:
        this->getVimeoVideoUrl(id);
        break;
    default:
        qWarning() << "UrlGrabber::getVideoUrl(): No/invalid service specied";
        return;
    }
}

QScriptValue UrlGrabber::getYouTubeDecryptionFunction(const QUrl &playerUrl) {
    if (youtubeDecryptionCache.contains(playerUrl)) {
        return youtubeDecryptionCache.value(playerUrl);
    }

    QNetworkRequest request(playerUrl);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addYouTubeDecryptionFunctionToCache()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));

    return QScriptValue();
}

void UrlGrabber::addYouTubeDecryptionFunctionToCache() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
        return;
    }

    QString response(reply->readAll());
    QRegExp re("\\.sig\\|\\|\\w+(?=\\()");

    if (re.indexIn(response) != -1) {
        QString funcName = re.cap().section("||", -1);
        QString var = response.section("function " + funcName, 0, 0).section(";var", -1);
        QString funcBody = QString("function %2%3").arg(funcName).arg(response.section("function " + funcName, 1, 1).section(";function", 0, 0));
        QString js = QString("var%1 %2").arg(var).arg(funcBody);
        qDebug() << "Found decryption function:" << js;
        youtubeDecryptionEngine->evaluate(js);

        QScriptValue global = youtubeDecryptionEngine->globalObject();
        QScriptValue decryptionFunction = global.property(funcName);

        if (decryptionFunction.isFunction()) {
            youtubeDecryptionCache[reply->request().url()] = decryptionFunction;
            emit youtubeDecryptionFunctionReady(decryptionFunction);
        }
        else {
            this->setBusy(false);
            emit error(tr("Unable to decrypt video signature"));
            this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
        }
    }
    else {
        this->setBusy(false);
        emit error(tr("Unable to decrypt video signature"));
        this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), 0, 0);
    }

    reply->deleteLater();
}

QMap<int, QUrl> UrlGrabber::getYouTubeVideoUrlMap(QString page, QScriptValue decryptionFunction) {
    QMap<int, QUrl> urlMap;
    QStringList parts = page.split(',', QString::SkipEmptyParts);

    if (decryptionFunction.isFunction()) {
        foreach (QString part, parts) {
            part = Utils::unescape(part);
            part.replace(QRegExp("(^|&)s="), "&signature=");
            QString oldSig = part.section("signature=", 1, 1).section('&', 0, 0);
            part.replace(oldSig, decryptionFunction.call(QScriptValue(), QScriptValueList() << oldSig).toString());
            QStringList splitPart = part.split("url=");

            if (!splitPart.isEmpty()) {
                QString urlString = splitPart.last();
                QStringList params = urlString.mid(urlString.indexOf('?') + 1).split('&', QString::SkipEmptyParts);
                params.removeDuplicates();

                QUrl url(urlString.left(urlString.indexOf('?')));
#if QT_VERSION >= 0x050000
                QUrlQuery query;

                foreach (QString param, params) {
                    query.addQueryItem(param.section('=', 0, 0), param.section('=', -1));
                }

                if (!query.hasQueryItem("signature")) {
                    query.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                url.setQuery(query);

                urlMap[query.queryItemValue("itag").toInt()] = url;
#else
                foreach (QString param, params) {
                    url.addQueryItem(param.section('=', 0, 0), param.section('=', -1));
                }

                if (!url.hasQueryItem("signature")) {
                    url.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                urlMap[url.queryItemValue("itag").toInt()] = url;
#endif
            }
        }
    }
    else {
        foreach (QString part, parts) {
            part = Utils::unescape(part);
            part.replace(QRegExp("(^|&)sig="), "&signature=");
            QStringList splitPart = part.split("url=");

            if (!splitPart.isEmpty()) {
                QString urlString = splitPart.last();
                QStringList params = urlString.mid(urlString.indexOf('?') + 1).split('&', QString::SkipEmptyParts);
                params.removeDuplicates();

                QUrl url(urlString.left(urlString.indexOf('?')));
#if QT_VERSION >= 0x050000
                QUrlQuery query;

                foreach (QString param, params) {
                    query.addQueryItem(param.section('=', 0, 0), param.section('=', -1));
                }

                if (!query.hasQueryItem("signature")) {
                    query.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                url.setQuery(query);

                urlMap[query.queryItemValue("itag").toInt()] = url;
#else
                foreach (QString param, params) {
                    url.addQueryItem(param.section('=', 0, 0), param.section('=', -1));
                }

                if (!url.hasQueryItem("signature")) {
                    url.addQueryItem("signature", splitPart.first().section("signature=", 1, 1).section('&', 0, 0));
                }

                urlMap[url.queryItemValue("itag").toInt()] = url;
#endif
            }
        }
    }

    return urlMap;
}

void UrlGrabber::getYouTubeVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    this->getYouTubeVideoInfoPage(id, SLOT(checkYouTubeVideoInfoPage()));
}

void UrlGrabber::getYouTubeVideoInfoPage(const QString &id, const char *slot) {
    QUrl url("https://www.youtube.com/get_video_info");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("video_id", id);
    query.addQueryItem("el", "detailpage");
    query.addQueryItem("ps", "default");
    query.addQueryItem("eurl", "gl");
    query.addQueryItem("gl", "US");
    query.addQueryItem("hl", "en");
    url.setQuery(query);
#else
    url.addQueryItem("video_id", id);
    url.addQueryItem("el", "detailpage");
    url.addQueryItem("ps", "default");
    url.addQueryItem("eurl", "gl");
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
#endif
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, slot);
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeVideoInfoPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << "No format map in YouTube video info page. Retrieving the web page";
#if QT_VERSION >= 0x050000
        this->getYouTubeVideoWebPage(QUrlQuery(reply->request().url()).queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
#else
        this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
#endif
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        QString separator = response.left(response.indexOf('%'));

        if ((separator == "s") || (response.contains("%26s%3D"))) {
            qDebug() << "YouTube video has encrypted signatures. Retrieving the web page";
#if QT_VERSION >= 0x050000
            this->getYouTubeVideoWebPage(QUrlQuery(reply->request().url()).queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
#else
            this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPage()));
#endif
        }
        else {
            qDebug() << "YouTube video info OK. Parsing the page";
            response = response.section('&', 0, 0).replace("%2C", ",");
            this->parseYouTubeVideoPage(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getYouTubeVideoWebPage(const QString &id, const char *slot) {
    QUrl url("http://www.youtube.com/watch");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("v", id);
    query.addQueryItem("gl", "US");
    query.addQueryItem("hl", "en");
    query.addQueryItem("has_verified", "1");
    url.setQuery(query);
#else
    url.addQueryItem("v", id);
    url.addQueryItem("gl", "US");
    url.addQueryItem("hl", "en");
    url.addQueryItem("has_verified", "1");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", "Mozilla/5.0 (X11; Linux x86_64; rv:28.0) Gecko/20100101  Firefox/28.0");
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, slot);
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeWebPage() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        this->setBusy(false);
        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
    }
    else {
        QVariantMap assets = Json::parse(QString("%1}").arg(response.section("\"assets\": ", 1, 1).section('}', 0, 0))).toMap();
        QUrl playerUrl = assets.value("js").toUrl();

        if (playerUrl.scheme().isEmpty()) {
            playerUrl.setScheme("http");
        }

        response = response.section("url_encoded_fmt_stream_map\": \"", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&").remove(QRegExp("itag=\\d+"));

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            if (playerUrl.isValid()) {
                QScriptValue decryptionFunction = this->getYouTubeDecryptionFunction(playerUrl);

                if (decryptionFunction.isFunction()) {
                    this->parseYouTubeVideoPage(decryptionFunction, response);
                }
                else {
                    m_youtubePage = response;
                    this->connect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoPage(QScriptValue)));
                }
            }
            else {
                this->setBusy(false);
                emit error(tr("Unable to decrypt video signature"));
            }
        }
        else {
            this->parseYouTubeVideoPage(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::parseYouTubeVideoPage(QScriptValue decryptionFunction, QString page) {
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoPage(QScriptValue)));
    this->setBusy(false);

    if (page.isEmpty()) {
        page = m_youtubePage;
    }

    QMap<int, QUrl> urlMap = this->getYouTubeVideoUrlMap(page, decryptionFunction);
    QUrl videoUrl;
    int index = 0;

    while ((videoUrl.isEmpty()) && (index < m_youtubeFormats.size())) {
        videoUrl = urlMap.value(m_youtubeFormats.at(index).value.toInt());
        m_format = m_youtubeFormats.at(index).format;
        index++;
    }

    if (videoUrl.isEmpty()) {
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        qDebug() << videoUrl;
        emit gotVideoUrl(videoUrl, m_format);
    }
}

void UrlGrabber::getDailymotionVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseDailymotionVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseDailymotionVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QUrl url;
    int i = 0;

    while ((url.isEmpty()) && (i < m_dailymotionFormats.size())) {
        url.setUrl(info.value(m_dailymotionFormats.at(i).value.toString(), "").toString());
        m_format = m_dailymotionFormats.at(i).format;
        i++;
    }

    if (url.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoUrl(url, m_format);
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoVideoUrl(const QString &id) {
    this->setBusy(true, tr("Loading video"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(parseVimeoVideoPage()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::parseVimeoVideoPage() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("\"h264\":", 1, 1).section(",\"hls\":", 0, 0);
    QVariantMap formatMap = Json::parse(params).toMap();

    if (formatMap.isEmpty()) {
        qDebug() << response;
        emit error(tr("Unable to retrieve video. Access may be restricted"));
    }
    else {
        QUrl url;
        int i = 0;

        while ((url.isEmpty()) && (i  < m_vimeoFormats.size())) {
            url.setUrl(formatMap.value(m_vimeoFormats.at(i).value.toString(), "").toMap().value("url").toString());
            m_format = m_vimeoFormats.at(i).format;
            i++;
        }

        if (url.isEmpty()) {
            qDebug() << response;
            emit error(tr("Unable to retrieve video. Access may be restricted"));
        }
        else {
            emit gotVideoUrl(url, m_format);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVideoFormats(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeVideoFormats(id);
        break;
    case Services::Dailymotion:
        this->getAvailableDailymotionVideoFormats(id);
        break;
    case Services::Vimeo:
        this->getAvailableVimeoVideoFormats(id);
        break;
    default:
        return;
    }
}

void UrlGrabber::getAvailableYouTubeVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    this->getYouTubeVideoInfoPage(id, SLOT(checkYouTubeVideoInfoFormats()));
}

void UrlGrabber::checkYouTubeVideoInfoFormats() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map=")) {
        qDebug() << "No format map in YouTube video info page. Retrieving the web page";
#if QT_VERSION >= 0x050000
        this->getYouTubeVideoWebPage(QUrlQuery(reply->request().url()).queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
#else
        this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
#endif
    }
    else {
        response = response.section("url_encoded_fmt_stream_map=", 1, 1);
        QString separator = response.left(response.indexOf('%'));

        if ((separator == "s") || (response.contains("%26s%3D"))) {
            qDebug() << "YouTube video has encrypted signatures. Retrieving the web page";
#if QT_VERSION >= 0x050000
            this->getYouTubeVideoWebPage(QUrlQuery(reply->request().url()).queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
#else
            this->getYouTubeVideoWebPage(reply->request().url().queryItemValue("video_id"), SLOT(checkYouTubeWebPageFormats()));
#endif
        }
        else {
            qDebug() << "YouTube video info OK. Parsing the page";
            response = response.section('&', 0, 0).replace(QRegExp("%2C(\\w+%3D1%26|)" + separator), "," + separator);
            this->parseYouTubeVideoFormats(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::checkYouTubeWebPageFormats() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());

    if (!response.contains("url_encoded_fmt_stream_map\":")) {
        qDebug() << response;

        QString message = response.section("unavailable-message\" class=\"message\">", 1, 1).section('<', 0, 0).trimmed();
        emit error(message.isEmpty() ? tr("Unable to retrieve video. Access may be restricted") : message);
    }
    else {
        QVariantMap assets = Json::parse(QString("%1}").arg(response.section("\"assets\": ", 1, 1).section('}', 0, 0))).toMap();
        QUrl playerUrl = assets.value("js").toUrl();

        if (playerUrl.scheme().isEmpty()) {
            playerUrl.setScheme("http");
        }

        response = response.section("url_encoded_fmt_stream_map\": \"", 1, 1).section(", \"", 0, 0).trimmed().replace("\\u0026", "&").remove(QRegExp("itag=\\d+"));

        bool encryptedSignatures = !response.contains("sig=");

        if (encryptedSignatures) {
            if (playerUrl.isValid()) {
                QScriptValue decryptionFunction = this->getYouTubeDecryptionFunction(playerUrl);

                if (decryptionFunction.isFunction()) {
                    this->parseYouTubeVideoFormats(decryptionFunction, response);
                }
                else {
                    m_youtubePage = response;
                    this->connect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoFormats(QScriptValue)));
                }
            }
            else {
                this->setBusy(false);
                emit error(tr("Unable to decrypt video signature"));
            }
        }
        else {
            this->parseYouTubeVideoFormats(QScriptValue(), response);
        }
    }

    reply->deleteLater();
}

void UrlGrabber::parseYouTubeVideoFormats(QScriptValue decryptionFunction, QString page) {
    this->disconnect(this, SIGNAL(youtubeDecryptionFunctionReady(QScriptValue)), this, SLOT(parseYouTubeVideoFormats(QScriptValue)));
    this->setBusy(false);

    if (page.isEmpty()) {
        page = m_youtubePage;
    }

    QVariantList formats;
    QList<int> keys;
#ifdef SYMBIAN_OS
    keys << 37 << 22 << 18 << 36 << 17;
#else
    keys << 37 << 22 << 35 << 34 << 18 << 36 << 17;
#endif
    QMap<int, QUrl> urlMap = this->getYouTubeVideoUrlMap(page, decryptionFunction);

    foreach (int key, keys) {
        if (urlMap.contains(key)) {
            QVariantMap format;
            format["value"] = key;
            format["url"] = urlMap.value(key);

            switch (key) {
            case 37:
                format["name"] = "1080P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 22:
                format["name"] = "720P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
#ifndef SYMBIAN_OS
            case 35:
                format["name"] = "480P FLV";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 34:
                format["name"] = "360P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
#endif
            case 18:
                format["name"] = QString("%1 AVC1").arg(tr("Normal"));
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 36:
                format["name"] = "240P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            case 17:
                format["name"] = "144P AVC1";
                format["service"] = Services::YouTube;
                formats << format;
                break;
            default:
                break;
            }
        }
    }

    if (formats.isEmpty()) {
        emit error(tr("Unable to retrieve formats. Access may be restricted"));
    }
    else {
        emit gotVideoFormats(formats);
    }
}

void UrlGrabber::getAvailableDailymotionVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://www.dailymotion.com/embed/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkDailymotionVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantMap info = Json::parse(response.section("var info = ", 1, 1).section(";\n", 0, 0)).toMap();
    QVariantList formats;

    QStringList keys;
    keys << "stream_h264_hd1080_url"
         << "stream_h264_hd_url"
         << "stream_h264_hq_url"
         << "stream_h264_url"
         << "stream_h264_ld_url";

    QStringList names;
    names << "1080P AVC1"
          << "720P AVC1"
          << "480P AVC1"
          << "348P AVC1"
          << "240P AVC1";

    for (int i = 0; i < keys.size(); i++){
        QUrl url = info.value(keys.at(i)).toUrl();

        if (!url.isEmpty()) {
            QVariantMap format;
            format["name"] = names.at(i);
            format["service"] = Services::Dailymotion;
            format["url"] = url;
            formats << format;
        }
    }

    if (formats.isEmpty()) {
        qDebug() << response;
        QString errorString = info.value("error").toMap().value("message").toString();
        emit error(errorString.isEmpty() ? tr("Unable to retrieve video formats. Access may be restricted") : errorString);
    }
    else {
        emit gotVideoFormats(formats);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoVideoFormats(const QString &id) {
    this->setBusy(true, tr("Retrieving available video formats"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoVideoFormats()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoVideoFormats() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("\"h264\":", 1, 1).section(",\"hls\":", 0, 0);
    QVariantMap formatMap = Json::parse(params).toMap();
    QVariantList formats;

    QStringList keys;
    keys << "hd"
         << "sd"
         << "mobile";

    QStringList names;
    names << "720 AVC1"
          << "360P AVC1"
          << tr("Mobile");

    for (int i = 0; i < keys.size(); i++){
        QUrl url = formatMap.value(keys.at(i)).toMap().value("url").toUrl();

        if (!url.isEmpty()) {
            QVariantMap format;
            format["name"] = names.at(i);
            format["service"] = Services::Vimeo;
            format["url"] = url;
            formats << format;
        }
    }

    if (formats.isEmpty()) {
        qDebug() << response;
        emit error(tr("Unable to retrieve video formats. Access may be restricted"));
    }
    else {
        emit gotVideoFormats(formats);
    }

    reply->deleteLater();
}

void UrlGrabber::getSubtitlesUrl(int service, const QString &id, const QString &language) {
    switch (service) {
    case Services::YouTube:
        this->getYouTubeSubtitlesUrl(id, language);
        return;
    case Services::Dailymotion:
        this->getDailymotionSubtitlesUrl(id, language);
        return;
    case Services::Vimeo:
        this->getVimeoSubtitlesUrl(id, language);
        return;
    default:
        qWarning() << "UrlGrabber::getSubtitlesUrl(): No/invalid service specied";
        return;
    }
}

void UrlGrabber::getYouTubeSubtitlesUrl(const QString &id, const QString &language) {
    this->setBusy(true, tr("Loading subtitles"));
    m_language = language;
    QUrl url("https://video.google.com/timedtext?hl=en&type=list&v=" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkYouTubeSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkYouTubeSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement transcriptEl = doc.documentElement();
    QDomNodeList trackNodes = transcriptEl.elementsByTagName("track");

    int i = 0;
    bool found = false;

    while ((!found) && (i < trackNodes.size())) {
        QDomElement trackEl = trackNodes.at(i).toElement();

        if (trackEl.attribute("lang_code") == m_language) {
            found = true;
        }

        i++;
    }

    if (found) {
        QUrl url("https://www.youtube.com/api/timedtext?fmt=srt");
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("v", QUrlQuery(reply->request().url()).queryItemValue("v"));
        query.addQueryItem("lang", m_language);
#else
        url.addQueryItem("v", reply->request().url().queryItemValue("v"));
        url.addQueryItem("lang", m_language);
#endif
        emit gotSubtitlesUrl(url);
    }
    else {
        emit error(tr("No subtitles found"));
    }

    reply->deleteLater();
}

void UrlGrabber::getDailymotionSubtitlesUrl(const QString &id, const QString &language) {
    this->setBusy(true, tr("Loading subtitles"));
    m_language = language;
    QUrl url(QString("https://api.dailymotion.com/video/%1/subtitles?fields=id,language,url").arg(id));
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkDailymotionSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList list = Json::parse(response).toMap().value("list").toList();

    int i = 0;
    QUrl url;

    while ((url.isEmpty()) && (i < list.size())) {
        QVariantMap map = list.at(i).toMap();

        if (map.value("language") == m_language) {
            url = map.value("url").toUrl();
        }

        i++;
    }

    if (!url.isEmpty()) {
        emit gotSubtitlesUrl(url);
    }
    else {
        emit error(tr("No subtitles found"));
    }

    reply->deleteLater();
}

void UrlGrabber::getVimeoSubtitlesUrl(const QString &id, const QString &language) {
    this->setBusy(true, tr("Loading subtitles"));
    m_language = language;
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVimeoSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkVimeoSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("\"text_tracks\":", 1, 1).section(",\"cookie_domain\":", 0, 0);
    QVariantList list = Json::parse(params).toList();

    int i = 0;
    QUrl url;

    while ((url.isEmpty()) && (i < list.size())) {
        QVariantMap map = list.at(i).toMap();

        if (map.value("lang") == m_language) {
            url = QUrl::fromEncoded(map.value("url").toString().section("url=", -1).toUtf8());
        }

        i++;
    }

    if (!url.isEmpty()) {
        emit gotSubtitlesUrl(url);
    }
    else {
        emit error(tr("No subtitles found"));
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableSubtitles(int service, const QString &id) {
    switch (service) {
    case Services::YouTube:
        this->getAvailableYouTubeSubtitles(id);
        return;
    case Services::Dailymotion:
        this->getAvailableDailymotionSubtitles(id);
        return;
    case Services::Vimeo:
        this->getAvailableVimeoSubtitles(id);
        return;
    default:
        qWarning() << "UrlGrabber::getAvailableSubtitles(): No/invalid service specied";
        return;
    }
}

void UrlGrabber::getAvailableYouTubeSubtitles(const QString &id) {
    this->setBusy(true, tr("Retrieving available subtitles"));
    QUrl url("https://video.google.com/timedtext?hl=en&type=list&v=" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAvailableYouTubeSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkAvailableYouTubeSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement transcriptEl = doc.documentElement();
    QDomNodeList trackNodes = transcriptEl.elementsByTagName("track");
    QVariantList list;

    for (int i = 0; i < trackNodes.size(); i++) {
        QDomElement trackEl = trackNodes.at(i).toElement();
        QString language = trackEl.attribute("lang_code");
        QUrl url("https://www.youtube.com/api/timedtext?fmt=srt");
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("v", QUrlQuery(reply->request().url()).queryItemValue("v"));
        query.addQueryItem("lang", m_language);
#else
        url.addQueryItem("v", reply->request().url().queryItemValue("v"));
        url.addQueryItem("lang", m_language);
#endif
        QVariantMap map;
        map["language"] = language;
        map["url"] = url;

        list << map;
    }

    if (list.isEmpty()) {
        emit error(tr("No subtitles found"));
    }
    else {
        emit gotSubtitles(list);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableDailymotionSubtitles(const QString &id) {
    this->setBusy(true, tr("Retrieving available subtitles"));
    QUrl url(QString("https://api.dailymotion.com/video/%1/subtitles?fields=id,language,url").arg(id));
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDailymotionSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkAvailableDailymotionSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QVariantList list = Json::parse(response).toMap().value("list").toList();

    if (list.isEmpty()) {
        emit error(tr("No subtitles found"));
    }
    else {
        emit gotSubtitles(list);
    }

    reply->deleteLater();
}

void UrlGrabber::getAvailableVimeoSubtitles(const QString &id) {
    this->setBusy(true, tr("Retrieving available subtitles"));
    QUrl url("http://player.vimeo.com/video/" + id);
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAvailableVimeoSubtitles()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void UrlGrabber::checkAvailableVimeoSubtitles() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QString response(reply->readAll());
    QString params = response.section("\"text_tracks\":", 1, 1).section(",\"cookie_domain\":", 0, 0);
    QVariantList list = Json::parse(params).toList();
    QVariantList subtitles;

    if (list.isEmpty()) {
        emit error(tr("No subtitles found"));
    }
    else {
        foreach (QVariant variant, list) {
            QVariantMap map = variant.toMap();
            map["language"] = map.value("lang").toString();
            map["url"] = QUrl::fromEncoded(map.value("url").toString().section("url=", -1).toUtf8());
            subtitles << map;
        }

        emit gotSubtitles(subtitles);
    }

    reply->deleteLater();
}
