#include "dailymotion.h"
#include "json.h"
#include "definitions.h"
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QStringList>
#include <QDebug>
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
#include <QQmlEngine>
#else
#include <QDeclarativeEngine>
#endif
#endif

using namespace QtJson;

Dailymotion* Dailymotion::self = 0;

const QByteArray CLIENT_ID("808ed79fa59e10f3952e");
const QByteArray CLIENT_SECRET("5753a7aa0fb0772b628d160aabe3da6059f97375");
const QByteArray REDIRECT_URI("https://sites.google.com/site/marxodian/home/cutetube");

Dailymotion::Dailymotion(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_safeSearch(false),
    m_actionsProcessed(0),
    m_playlistCache(new QList< QSharedPointer<PlaylistItem> >),
    m_subscriptionCache(new QList< QSharedPointer<UserItem> >),
    m_groupCache(new QList< QSharedPointer<GroupItem> >),
    m_playlistCacheLoaded(false),
    m_subscriptionCacheLoaded(false),
    m_groupCacheLoaded(false),
    m_busy(false),
    m_cancelled(false)
{
    if (!self) {
        self = this;
    }

    m_queryOrders[Queries::Relevance] = "relevance";
    m_queryOrders[Queries::Date] = "recent";
    m_queryOrders[Queries::Views] = "visited";
    m_queryOrders[Queries::Rating] = "rated";
}

Dailymotion::~Dailymotion() {
    this->clearCache();
}

Dailymotion* Dailymotion::instance() {
    return !self ? new Dailymotion : self;
}

void Dailymotion::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;
        emit busyChanged(isBusy);
    }

    if (isBusy) {
        this->setCancelled(false);
        emit busy(message, numberOfOperations);
    }
    else if (!this->cancelled()) {
        emit busyProgressChanged(numberOfOperations);
    }
}

void Dailymotion::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, 0);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, 0);
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, 0);
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    emit currentOperationCancelled();
}

QNetworkReply* Dailymotion::createReply(QUrl feed, int offset) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(feed);

    if (userSignedIn()) {
        query.addQueryItem("access_token", this->accessToken());
    }

    if (offset) {
        query.addQueryItem("page", QString::number(offset));
    }

    query.addQueryItem("family_filter", this->safeSearch() ? "true" : "false");
    feed.setQuery(query);
#else
    if (userSignedIn()) {
        feed.addQueryItem("access_token", this->accessToken());
    }

    if (offset) {
        feed.addQueryItem("page", QString::number(offset));
    }

    feed.addQueryItem("family_filter", this->safeSearch() ? "true" : "false");
#endif

    qDebug() << feed;

    QNetworkRequest request(feed);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());

    return this->networkAccessManager()->get(request);
}

QNetworkReply* Dailymotion::createSearchReply(int queryType, const QString &query, int offset, int order, const QString &language) {
    QUrl url;
#if QT_VERSION >= 0x050000
    QUrlQuery urlQuery;

    switch (queryType) {
    case Queries::Videos:
        url.setUrl(DAILYMOTION_VIDEOS_BASE_URL);
        urlQuery.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
        urlQuery.addQueryItem("sort", m_queryOrders.value(order, "relevance"));

        if ((!language.isEmpty()) && (language != "all")) {
            urlQuery.addQueryItem("localization", language);
        }
        break;
    case Queries::Playlists:
        url.setUrl(DAILYMOTION_PLAYLISTS_BASE_URL);
        urlQuery.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
        urlQuery.addQueryItem("sort", "relevance");
        break;
    case Queries::Users:
        url.setUrl(DAILYMOTION_USERS_BASE_URL);
        urlQuery.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
        urlQuery.addQueryItem("sort", "relevance");
        break;
    case Queries::Groups:
        url.setUrl(DAILYMOTION_GROUPS_BASE_URL);
        urlQuery.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
        urlQuery.addQueryItem("sort", "relevance");
        break;
    default:
        qWarning() << "Dailymotion::createSearchReply(): No/invalid query type.";
        break;
    }

    urlQuery.addQueryItem("family_filter", this->safeSearch() ? "true" : "false");
    urlQuery.addQueryItem("limit", "30");
    urlQuery.addQueryItem("search", QString("\"%1\"|%2").arg(query).arg(query.simplified().replace(' ', '+')));

    if (offset) {
        urlQuery.addQueryItem("page", QString::number(offset));
    }

    url.setQuery(urlQuery);
#else
    switch (queryType) {
    case Queries::Videos:
        url.setUrl(DAILYMOTION_VIDEOS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
        url.addQueryItem("sort", m_queryOrders.value(order, "relevance"));

        if ((!language.isEmpty()) && (language != "all")) {
            url.addQueryItem("localization", language);
        }
        break;
    case Queries::Playlists:
        url.setUrl(DAILYMOTION_PLAYLISTS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
        url.addQueryItem("sort", "relevance");
        break;
    case Queries::Users:
        url.setUrl(DAILYMOTION_USERS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
        url.addQueryItem("sort", "relevance");
        break;
    case Queries::Groups:
        url.setUrl(DAILYMOTION_GROUPS_BASE_URL);
        url.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
        url.addQueryItem("sort", "relevance");
        break;
    default:
        qWarning() << "Dailymotion::createSearchReply(): No/invalid query type.";
        break;
    }

    url.addQueryItem("family_filter", this->safeSearch() ? "true" : "false");
    url.addQueryItem("limit", "30");
    url.addQueryItem("search", QString("\"%1\"|%2").arg(query).arg(query.simplified().replace(' ', '+')));

    if (offset) {
        url.addQueryItem("page", QString::number(offset));
    }
#endif
    qDebug() << url;

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());

    return this->networkAccessManager()->get(request);
}

QUrl Dailymotion::authUrl() const {
    QUrl url("https://api.dailymotion.com/oauth/authorize");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("client_id", CLIENT_ID);
    query.addQueryItem("response_type", "code");
    query.addQueryItem("redirect_uri", REDIRECT_URI);
    query.addQueryItem("display", "mobile");
    query.addQueryItem("scope", "read+userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups");
    url.setQuery(query);
#else
    url.addQueryItem("client_id", CLIENT_ID);
    url.addQueryItem("response_type", "code");
    url.addQueryItem("redirect_uri", REDIRECT_URI);
    url.addQueryItem("display", "mobile");
    url.addQueryItem("scope", "read+userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups");
#endif
    return url;
}

void Dailymotion::setAccount(const QString &user, const QString &token, const QString &refresh) {
    if (user != this->username()) {
        this->setUsername(user);
    }

    this->setAccessToken(token);
    this->setRefreshToken(refresh);
    this->clearCache();

    if ((user.isEmpty()) && (!token.isEmpty())) {
        emit newAccountSet();
    }
}

void Dailymotion::signIn(const QString &displayName, const QUrl &response) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(response);
    QString code = query.queryItemValue("code");
#else
    QString code = response.queryItemValue("code");
#endif
    if (code.isEmpty()) {
#if QT_VERSION >= 0x050000
        QString errorString = query.queryItemValue("error");
#else
        QString errorString = response.queryItemValue("error");
#endif
        if (errorString == "access_denied") {
            emit info(tr("You have denied access to your Dailymotion account"));
        }
        else {
            emit error(tr("Unable to authorise access to your Dailymotion account"));
        }
    }
    else {
        this->setBusy(true, tr("Signing in"));
        m_user = displayName;
        QUrl url("https://api.dailymotion.com/oauth/token");

        QByteArray body("code=" + code.toUtf8() +
                        "&client_id=" + CLIENT_ID +
                        "&client_secret=" + CLIENT_SECRET +
                        "&redirect_uri=" + REDIRECT_URI +
                        "&grant_type=authorization_code");

        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
        request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
        QNetworkReply *reply = this->networkAccessManager()->post(request, body);
        this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
        this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
    }
}

void Dailymotion::signIn(const QString &displayName, const QString &user, const QString &pass) {
    this->setBusy(true, tr("Signing in"));
    m_user = displayName;
    QUrl url("https://api.dailymotion.com/oauth/token");

    QByteArray body("username=" + user.toUtf8() +
                    "&password=" + pass.toUtf8() +
                    "&client_id=" + CLIENT_ID +
                    "&client_secret=" + CLIENT_SECRET +
                    "&scope=userinfo+manage_videos+manage_comments+manage_playlists+manage_subscriptions+manage_favorites+manage_groups"
                    "&redirect_uri=" + REDIRECT_URI +
                    "&grant_type=password");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Dailymotion::checkIfSignedIn() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit error(tr("Cannot parse server response"));
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if ((token.isEmpty()) || (refresh.isEmpty())) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }
        else {
            emit alert(tr("You are signed in to your Dailymotion account"));
            emit signedIn(this->username(), token, refresh);
        }
    }
}

void Dailymotion::refreshAccessToken() {
    QUrl url("https://api.dailymotion.com/oauth/token");

    QByteArray body("client_id=" + CLIENT_ID +
                    "&client_secret=" + CLIENT_SECRET +
                    "&refresh_token=" + this->refreshToken().toUtf8() +
                    "&grant_type=refresh_token");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTokenRefresh()));
}

void Dailymotion::checkTokenRefresh() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit error(tr("Cannot parse server response"));
        emit refreshError();
        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, 0);
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if (token.isEmpty()) {
            emit error(tr("Unable to refresh access token"));
            emit refreshError();
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, 0);
        }
        else {
            this->setAccessToken(token);
            this->setRefreshToken(refresh);
            emit accessTokenRefreshed(token, refresh);
        }
    }
}

void Dailymotion::clearCache() {
    m_playlistCache->clear();
    m_subscriptionCache->clear();
    m_groupCache->clear();

    this->setPlaylistsLoaded(false);
    this->setSubscriptionsLoaded(false);
    this->setGroupsLoaded(false);
}

void Dailymotion::signOut() {
    this->setBusy(true, tr("Signing out"));
    QUrl url("https://api.dailymotion.com/logout");
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", "OAuth " + this->accessToken().toUtf8());
    QNetworkReply* reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedOut()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Dailymotion::checkIfSignedOut() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        this->setAccount();
        emit info(tr("You have signed out of your Dailymotion account"));
    }
    else {
        emit error(tr("Unable to sign out of your Dailymotion account. Please visit the Dailymotion website to revoke access"));
    }

    reply->deleteLater();
}

void Dailymotion::postRequest(const QUrl &url, const QByteArray &data) {
    qDebug() << url;
    qDebug() << data;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    request.setRawHeader("Authorization", "OAuth " + this->accessToken().toUtf8());
    QNetworkReply* reply = this->networkAccessManager()->post(request, data);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Dailymotion::deleteRequest(const QUrl &url) {
    qDebug() << url;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", "OAuth " + this->accessToken().toUtf8());
    QNetworkReply* reply = this->networkAccessManager()->deleteResource(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Dailymotion::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (userSignedIn())) {
        this->refreshAccessToken();
    }
    else {
        bool ok;
        QString response(reply->readAll());
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit postFailed(statusText);
        }
        else {
            QVariantMap error = result.value("error").toMap();

            if (!error.isEmpty()) {
                if ((error.value("type").toString() == "video_already_exists")) {
                    emit postSuccessful(QVariantMap());
                }
                else {
                    QString errorString = error.value("message").toString();
                    emit postFailed(errorString);
                }
            }
            else {
                emit postSuccessful(result);
            }
        }
    }

    reply->deleteLater();
}

void Dailymotion::getPlaylists(int offset) {
    this->setPlaylistsLoaded(false);
    QUrl url(DAILYMOTION_PLAYLISTS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("page", QString::number(offset));
    query.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    query.addQueryItem("access_token", accessToken());
    url.setQuery(query);
#else
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", accessToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void Dailymotion::addPlaylists() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error(tr("Error parsing server response"));
            emit allPlaylistsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                PlaylistItem *playlist = new PlaylistItem;
                playlist->loadDailymotionPlaylist(entries.at(i).toMap());
                m_playlistCache->append(QSharedPointer<PlaylistItem>(playlist));
            }

            if (res.value("has_more").toBool()) {
                this->getPlaylists(res.value("page").toInt() + 1);
            }
            else {
                this->setPlaylistsLoaded(true);
                emit allPlaylistsLoaded();
            }
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist) {
    m_playlistCache->insert(0, playlist);
    emit playlistAddedToCache(0);
}

QSharedPointer<PlaylistItem> Dailymotion::removePlaylistFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    QSharedPointer<PlaylistItem> playlist;

    while ((!removed) && (i < m_playlistCache->size())) {
        removed = (m_playlistCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        playlist = m_playlistCache->takeAt(i - 1);
        emit playlistRemovedFromCache(i - 1);
    }

    return playlist;
}

void Dailymotion::updatePlaylistVideoCount(const QString &id, int change) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i).data()->id() == id);
        i++;
    }

    if (updated) {
        QSharedPointer<PlaylistItem> playlist =  m_playlistCache->at(i - 1);
        playlist.data()->setVideoCount(playlist.data()->videoCount() + change);
        emit playlistUpdated(i - 1);
    }
}

void Dailymotion::getGroups(int offset) {
    this->setGroupsLoaded(false);
    QUrl url(DAILYMOTION_GROUPS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("page", QString::number(offset));
    query.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
    query.addQueryItem("access_token", accessToken());
    url.setQuery(query);
#else
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
    url.addQueryItem("access_token", accessToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void Dailymotion::addGroups() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error(tr("Error parsing server response"));
            emit allGroupsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                GroupItem *group = new GroupItem;
                group->loadDailymotionGroup(entries.at(i).toMap(), true);
                m_groupCache->append(QSharedPointer<GroupItem>(group));
            }

            if (res.value("has_more").toBool()) {
                this->getGroups(res.value("page").toInt() + 1);
            }
            else {
                this->setGroupsLoaded(true);
                emit allGroupsLoaded();
            }
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getGroups()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewGroupToCache(QSharedPointer<GroupItem> group) {
    m_groupCache->insert(0, group);
    emit groupAddedToCache(0);
    emit groupMembershipChanged(group.data()->id(), true);
}

QSharedPointer<GroupItem> Dailymotion::removeGroupFromCache(const QString &id) {
    int i = 0;
    bool removed = false;
    QSharedPointer<GroupItem> group;

    while ((!removed) && (i < m_groupCache->size())) {
        removed = (m_groupCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        group = m_groupCache->takeAt(i - 1);
        emit groupRemovedFromCache(i - 1);
        emit groupMembershipChanged(id, false);
    }

    return group;
}

bool Dailymotion::memberOfGroup(const QString &groupId) {
    if (!groupId.isEmpty()) {
        m_actionId = groupId;
    }

    if (this->groupsLoaded()) {
        int i = 0;
        bool member = false;

        while ((i < m_groupCache->size()) && (!member)) {
            member = (m_groupCache->at(i).data()->id() == m_actionId);
            i++;
        }

        this->disconnect(this, SIGNAL(allGroupsLoaded()), this, SLOT(memberOfGroup()));

        emit groupMembershipChanged(m_actionId, member);

        return member;
    }
    else {
        this->getGroups();
        this->connect(this, SIGNAL(allGroupsLoaded()), this, SLOT(memberOfGroup()));

        return false;
    }
}

void Dailymotion::getSubscriptions(int offset) {
    this->setSubscriptionsLoaded(false);
    QUrl url(DAILYMOTION_SUBSCRIPTIONS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("page", QString::number(offset));
    query.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
    query.addQueryItem("access_token", accessToken());
    url.setQuery(query);
#else
    url.addQueryItem("page", QString::number(offset));
    url.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
    url.addQueryItem("access_token", accessToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addSubscriptions()));
}

void Dailymotion::addSubscriptions() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getSubscriptions()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error(tr("Error parsing server response"));
            emit allSubscriptionsLoaded();
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                UserItem *user = new UserItem;
                user->loadDailymotionUser(entries.at(i).toMap(), true);
                m_subscriptionCache->append(QSharedPointer<UserItem>(user));
            }

            if (res.value("has_more").toBool()) {
                this->getSubscriptions(res.value("page").toInt() + 1);
            }
            else {
                this->setSubscriptionsLoaded(true);
                emit allSubscriptionsLoaded();
            }
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getSubscriptions()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::addNewSubscriptionToCache(QSharedPointer<UserItem> user) {
    m_subscriptionCache->insert(0, user);
    emit subscriptionAddedToCache(0);
    emit subscriptionChanged(user.data()->id(), true);
}

QSharedPointer<UserItem> Dailymotion::removeSubscriptionFromCache(const QString &id) {
    int i = 0;
    bool removed = false;
    QSharedPointer<UserItem> user;

    while ((!removed) && (i < m_subscriptionCache->size())) {
        removed = (m_subscriptionCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        user = m_subscriptionCache->takeAt(i - 1);
        emit subscriptionRemovedFromCache(i - 1);
        emit subscriptionChanged(id, false);
    }


    return user;
}

bool Dailymotion::subscribedToChannel(const QString &userId) {
    if (!userId.isEmpty()) {
        m_actionId = userId;
    }

    int i = 0;
    bool subscribed = false;

    while ((i < m_subscriptionCache->size()) && (!subscribed)) {
        subscribed = (m_subscriptionCache->at(i).data()->id() == m_actionId);
        i++;
    }

    this->disconnect(this, 0, this, SLOT(subscribedToChannel()));

    emit subscriptionChanged(m_actionId, subscribed);

    return subscribed;
}

void Dailymotion::getFullVideo(QString id) {
    this->setBusy(true, tr("Retrieving video details"));

    if (id.size() > 6) {
        id = id.section('/', -1).section('_', 0, 0);
    }

    QUrl url("https://api.dailymotion.com/video/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    query.addQueryItem("family_filter", "false");
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    url.addQueryItem("family_filter", "false");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Dailymotion::checkFullVideo() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            VideoItem *video = new VideoItem;
            video->loadDailymotionVideo(result);
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(video, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);
#endif
            emit gotVideo(video);
#else
            emit gotVideo(QSharedPointer<VideoItem>(video));
#endif
        }
        else {
            emit error(tr("Unable to retrieve video"));
#ifdef QML_USER_INTERFACE
            this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
            this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
        }
    }
    else {
        emit error(tr("Unable to retrieve video"));
#ifdef QML_USER_INTERFACE
        this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
        this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    }

    reply->deleteLater();
}

void Dailymotion::getVideoMetadata(const QString &id) {
    QUrl url("https://api.dailymotion.com/video/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_VIDEO_METADATA_FIELDS);
    query.addQueryItem("access_token", this->accessToken());
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_VIDEO_METADATA_FIELDS);
    url.addQueryItem("access_token", this->accessToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVideoMetadata()));
}

void Dailymotion::checkVideoMetadata() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideoMetadata()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (!ok) {
            emit error(tr("Error parsing server response"));
        }
        else if (result.value("error").isNull()) {
            QVariantMap metadata;
            metadata.insert("title", result.value("title").toString());
            metadata.insert("description", result.value("description").toString());
            metadata.insert("tags", result.value("tags").toString());
            metadata.insert("category", result.value("channel").toString());
            metadata.insert("commentsPermission", result.value("allow_comments").toString());
            metadata.insert("isPrivate", result.value("private").toBool());
            emit gotVideoMetadata(metadata);
        }
        else {
            QString errorString = result.value("error").toMap().value("message").toString();
            emit error(errorString.isEmpty() ? tr("Unable to retrieve video metadata") : errorString);
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getVideoMetadata()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::getCurrentUserProfile() {
    QUrl url("https://api.dailymotion.com/me");
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
    query.addQueryItem("access_token", accessToken());
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
    url.addQueryItem("access_token", accessToken());
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUserProfile()));
}

void Dailymotion::checkCurrentUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getCurrentUserProfile()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            UserItem *user = new UserItem;
            user->loadDailymotionUser(result);

            if (this->username() != user->id()) {
                this->setUsername(user->id());
            }
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
            emit gotUser(user);
#else
            emit gotUser(QSharedPointer<UserItem>(user));
#endif
        }
        else {
            emit error(tr("Unable to retrieve user profile"));
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getCurrentUserProfile()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void Dailymotion::getUserProfile(const QString &id) {
    QUrl url("https://api.dailymotion.com/user/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_USER_PROFILE_FIELDS);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void Dailymotion::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            UserItem *user = new UserItem;
            user->loadDailymotionUser(result);

            if (this->subscriptionsLoaded()) {
                this->subscribedToChannel(user->id());
            }
            else if (this->userSignedIn()) {
                m_actionId = user->id();
                this->getSubscriptions();
                this->connect(this, SIGNAL(allSubscriptionsLoaded()), this, SLOT(subscribedToChannel()));
            }
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
            emit gotUser(user);
#else
            emit gotUser(QSharedPointer<UserItem>(user));
#endif
        }
        else {
            emit error(tr("Unable to retrieve user profile"));
        }
    }
    else {
        emit error(tr("Unable to retrieve user profile"));
    }

    reply->deleteLater();
}

void Dailymotion::updateVideoMetadata(const QVariantMap &metadata) {
    m_metadataAction = metadata;
    this->updateVideoMetadata();
}

void Dailymotion::updateVideoMetadata() {
    QUrl url("https://api.dailymotion.com/video/" + m_metadataAction.value("videoId").toString());
    QByteArray data("title=" + m_metadataAction.value("title").toString().toUtf8()
                    + "&description=" + m_metadataAction.value("description").toString().toUtf8()
                    + "&tags=" + m_metadataAction.value("tags").toString().toUtf8()
                    + "&channel=" + m_metadataAction.value("category").toString().toUtf8()
                    + "&allow_comments" + m_metadataAction.value("commentsPermission").toString().toUtf8()
                    + "&private=" + m_metadataAction.value("isPrivate").toString().toUtf8());

    this->postRequest(url, data);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoMetadataUpdated()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(updateVideoMetadata()));
}

void Dailymotion::onVideoMetadataUpdated() {
    emit videoMetadataUpdated(m_metadataAction.value("videoId").toString(), m_metadataAction);
    emit alert(tr("Video metadata updated"));

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoMetadataUpdated()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(updateVideoMetadata()));
}

void Dailymotion::addToFavourites(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->addToFavourites();
        this->setBusy(true, tr("Adding video(s) to favourites"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Dailymotion::addToFavourites() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://api.dailymotion.com/me/favorites/" + m_actionIdList.first());
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToFavourites(QVariantMap)));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::onAddedToFavourites(const QVariantMap &response) {
    Q_UNUSED(response)

    if (!m_actionIdList.isEmpty()) {
        emit favouriteChanged(m_actionIdList.takeFirst(), true);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToFavourites();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to favourites"));

            this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToFavourites(QVariantMap)));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToFavourites()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Dailymotion::deleteFromFavourites(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromFavourites();
        this->setBusy(true, tr("Deleting video(s) from favourites"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Dailymotion::deleteFromFavourites() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://api.dailymotion.com/me/favorites/" + m_actionIdList.first());
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromFavourites()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::onDeletedFromFavourites() {
    if (!m_actionIdList.isEmpty()) {
        emit favouriteChanged(m_actionIdList.takeFirst(), false);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromFavourites();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from favourites"));

            this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromFavourites()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromFavourites()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Dailymotion::deleteFromUploads(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromUploads();
        this->setBusy(true, tr("Deleting video(s) from uploads"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Dailymotion::deleteFromUploads() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://api.dailymotion.com/video/" + m_actionIdList.first());
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoDeleted()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromUploads()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::onVideoDeleted() {
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromUploads(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromUploads();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from uploads"));

            this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onVideoDeleted()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromUploads()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Dailymotion::addToPlaylist(const QStringList &videoIds, const QString &playlistId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = playlistId;
        m_actionsProcessed = 0;
        this->addToPlaylist();
        this->setBusy(true, tr("Adding video(s) to playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Dailymotion::addToPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(QString("https://api.dailymotion.com/video/%1/playlists/%2").arg(m_actionIdList.first()).arg(m_actionId));
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToPlaylist(QVariantMap)));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        this->disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }
}

void Dailymotion::onAddedToPlaylist(const QVariantMap &response) {
    Q_UNUSED(response)

    if (this->playlistsLoaded()) {
        this->updatePlaylistVideoCount(m_actionId, 1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit addedToPlaylist(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onAddedToPlaylist(QVariantMap)));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addToPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Dailymotion::deleteFromPlaylist(const QStringList &videoIds, const QString &playlistId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = playlistId;
        m_actionsProcessed = 0;
        this->deleteFromPlaylist();
        this->setBusy(true, tr("Deleting video(s) from playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Dailymotion::deleteFromPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(QString("https://api.dailymotion.com/video/%1/playlists/%2").arg(m_actionIdList.first()).arg(m_actionId));
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromPlaylist()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Dailymotion::onDeletedFromPlaylist() {    
    if (this->playlistsLoaded()) {
        this->updatePlaylistVideoCount(m_actionId, -1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromPlaylist(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onDeletedFromPlaylist()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deleteFromPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Dailymotion::createPlaylist(const QVariantMap &playlist, const QStringList &videoIds) {
    m_metadataAction = playlist;
    m_actionIdList = videoIds;
    this->createPlaylist();
}

void Dailymotion::createPlaylist() {
    QUrl url("https://api.dailymotion.com/me/playlists");
    QByteArray data("name=" + m_metadataAction.value("title").toString().toUtf8()
                    + "&description=" + m_metadataAction.value("description").toString().toUtf8());

    this->postRequest(url, data);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistCreated(QVariantMap)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(createPlaylist()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::onPlaylistCreated(const QVariantMap &response) {
    m_actionId = response.value("id").toString();

    if (!m_actionIdList.isEmpty()) {
        m_actionsProcessed = 0;

        if (this->playlistsLoaded()) {
            this->getPlaylistForCache(m_actionId);
            this->connect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
        }
        else {
            this->addToPlaylist();
        }
    }
    else {
        if (this->playlistsLoaded()) {
            this->getPlaylistForCache(m_actionId);
        }

        emit alert(tr("Playlist created"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistCreated(QVariantMap)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(createPlaylist()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::getPlaylistForCache(const QString &id) {
    QUrl url("https://api.dailymotion.com/playlist/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCachePlaylist()));
}

void Dailymotion::checkCachePlaylist() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            PlaylistItem *playlist = new PlaylistItem;
            playlist->loadDailymotionPlaylist(result);
            this->addNewPlaylistToCache(QSharedPointer<PlaylistItem>(playlist));
        }
        else {
            emit error(tr("Error parsing server response"));
        }
    }
    else {
        emit error(tr("Unable to retrieve playlist details"));
        this->disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }

    reply->deleteLater();
}

void Dailymotion::deletePlaylist(const QString &playlistId) {
    m_actionId = playlistId;
    this->deletePlaylist();
}

void Dailymotion::deletePlaylist() {
    QUrl url("https://api.dailymotion.com/playlist/" + m_actionId);
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistDeleted()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deletePlaylist()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::onPlaylistDeleted() {
    emit alert(tr("Playlist deleted"));

    if (this->playlistsLoaded()) {
        this->removePlaylistFromCache(m_actionId);
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onPlaylistDeleted()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(deletePlaylist()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void Dailymotion::joinGroup(const QString &groupId) {
    m_actionId = groupId;
    this->joinGroup();
}

void Dailymotion::joinGroup() {
    QUrl url(QString("https://api.dailymotion.com/group/%1/members/%2").arg(m_actionId).arg(this->username()));
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupJoined()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(joinGroup()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::onGroupJoined() {
    if (this->groupsLoaded()) {
        this->getGroupForCache(m_actionId);
    }

    emit alert(tr("You have joined this group"));

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupJoined()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(joinGroup()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::getGroupForCache(const QString &id) {
    QUrl url("https://api.dailymotion.com/group/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_GROUP_FIELDS);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCacheGroup()));
}

void Dailymotion::checkCacheGroup() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            GroupItem *group = new GroupItem;
            group->loadDailymotionGroup(result, true);
            this->addNewGroupToCache(QSharedPointer<GroupItem>(group));
        }
        else {
            emit error(tr("Error parsing server response"));
        }
    }
    else {
        emit error(tr("Unable to retrieve group details"));
    }

    reply->deleteLater();
}

void Dailymotion::leaveGroup(const QString &groupId) {
    m_actionId = groupId;
    this->leaveGroup();
}

void Dailymotion::leaveGroup() {
    QUrl url(QString("https://api.dailymotion.com/group/%1/members/%2").arg(m_actionId).arg(this->username()));
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupLeft()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(leaveGroup()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::onGroupLeft() {
    if ((!this->groupsLoaded()) || (!this->removeGroupFromCache(m_actionId).isNull())) {
        emit alert(tr("You have left this group"));
    }
    else {
        emit error(tr("Unable to leave group"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onGroupLeft()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(leaveGroup()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Dailymotion::addComment(const QVariantMap &comment) {
    m_metadataAction = comment;
    this->addComment();
}

void Dailymotion::addComment() {
    QUrl url(QString("https://api.dailymotion.com/video/%1/comments").arg(m_metadataAction.value("videoId").toString()));
    QByteArray data("message=" + m_metadataAction.value("body").toString().toUtf8());
    this->postRequest(url, data);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onCommentAdded(QVariantMap)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addComment()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Dailymotion::onCommentAdded(const QVariantMap &response) {
    QString id = response.value("id").toString();

    if (!id.isEmpty()) {
        this->getAddedComment(id);
    }
    else {
        emit error(tr("Error parsing server response"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onCommentAdded(QVariantMap)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(addComment()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Dailymotion::getAddedComment(const QString &id) {
    QUrl url("https://api.dailymotion.com/comment/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_COMMENT_FIELDS);
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_COMMENT_FIELDS);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAddedComment()));
}

void Dailymotion::checkAddedComment() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (ok) {
        CommentItem *comment = new CommentItem;
        comment->loadDailymotionComment(result);
        emit commentAdded(QSharedPointer<CommentItem>(comment));
        emit alert(tr("Your comment has been added"));
    }
    else {
        emit error(tr("Error parsing server response"));
    }

    reply->deleteLater();
}

void Dailymotion::subscribe(const QString &userId) {
    m_actionId = userId;
    this->subscribe();
}

void Dailymotion::subscribe() {
    QUrl url("https://api.dailymotion.com/me/following/" + m_actionId);
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onSubscribed()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(subscribe()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onSubscribed() {
    if (this->subscriptionsLoaded()) {
        this->getSubscriptionForCache(m_actionId);
    }
    else {
        emit subscriptionChanged(m_actionId, true);
        emit alert(tr("You have subscribed to this channel"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onSubscribed()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(subscribe()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::getSubscriptionForCache(const QString &id) {
    QUrl url("https://api.dailymotion.com/user/" + id);
#if QT_VERSION >= 0x050000
    QUrlQuery query;
    query.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
    url.setQuery(query);
#else
    url.addQueryItem("fields", DAILYMOTION_USER_FIELDS);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCacheSubscription()));
}

void Dailymotion::checkCacheSubscription() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QString response(reply->readAll());

        bool ok;
        QVariantMap result = Json::parse(response, ok).toMap();

        if (ok) {
            UserItem *user = new UserItem;
            user->loadDailymotionUser(result, true);
            this->addNewSubscriptionToCache(QSharedPointer<UserItem>(user));
            emit alert(tr("You have subscribed to '%1'").arg(user->username()));
        }
        else {
            emit error(tr("Error parsing server response"));
        }
    }
    else {
        emit error(tr("Unable to retrieve subscription details"));
    }

    reply->deleteLater();
}

void Dailymotion::unsubscribe(const QString &userId) {
    m_actionId = userId;
    this->unsubscribe();
}

void Dailymotion::unsubscribe() {
    QUrl url("https://api.dailymotion.com/me/following/" + m_actionId);
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onUnsubscribed()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(unsubscribe()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onUnsubscribed() {
    if (this->subscriptionsLoaded()) {
        QSharedPointer<UserItem> user = this->removeSubscriptionFromCache(m_actionId);

        if (!user.isNull()) {
            emit subscriptionChanged(m_actionId, false);
            emit alert(tr("You have unsubscribed to '%1'").arg(user.data()->username()));
        }
        else {
            emit alert(tr("You have unsubscribed to this channel"));
        }
    }
    else {
        emit subscriptionChanged(m_actionId, false);
        emit alert(tr("You have unsubscribed to this channel"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QVariantMap)), this, SLOT(onUnsubscribed()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(unsubscribe()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Dailymotion::onVideoActionError(const QString &errorString) {
    m_actionIdList.clear();
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Dailymotion::onPlaylistActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Dailymotion::onGroupActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Dailymotion::onUserActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Dailymotion::onCommentActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Dailymotion::getVideosFromIds(QStringList ids) {
    this->setBusy(true, QString(), 0);
    QUrl url(DAILYMOTION_VIDEOS_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("ids", ids.join(","));
    query.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    query.addQueryItem("family_filter", safeSearch() ? "true" : "false");
    query.addQueryItem("limit", "100");
    url.setQuery(query);
#else
    url.addQueryItem("ids", ids.join(","));
    url.addQueryItem("fields", DAILYMOTION_VIDEO_FIELDS);
    url.addQueryItem("family_filter", safeSearch() ? "true" : "false");
    url.addQueryItem("limit", "100");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideos()));
}

void Dailymotion::checkFullVideos() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap res = Json::parse(response, ok).toMap();

    if (ok) {
        QVariantList entries = res.value("list").toList();

        if (!entries.isEmpty()) {
            QList< QSharedPointer<VideoItem> > videos;

            for (int i = 0; i < entries.size(); i++) {
                VideoItem *video = new VideoItem;
                video->loadDailymotionVideo(entries.at(i).toMap());
                videos.append(QSharedPointer<VideoItem>(video));
            }

            emit gotVideosFromIds(videos);
        }
    }

    reply->deleteLater();
}
