#ifndef DAILYMOTION_H
#define DAILYMOTION_H

#include "enums.h"
#include "feedurls.h"
#include "videoitem.h"
#include "playlistitem.h"
#include "groupitem.h"
#include "useritem.h"
#include "commentitem.h"
#include <QObject>
#include <QUrl>
#include <QString>
#include <QVariantMap>
#include <QSharedPointer>

class QNetworkAccessManager;
class QNetworkReply;

class Dailymotion : public QObject {
    Q_OBJECT

    Q_PROPERTY(QUrl uploadsFeed
               READ uploadsFeed
               CONSTANT)
    Q_PROPERTY(QUrl favouritesFeed
               READ favouritesFeed
               CONSTANT)
    Q_PROPERTY(QUrl playlistsFeed
               READ playlistsFeed
               CONSTANT)
    Q_PROPERTY(QUrl groupsFeed
               READ groupsFeed
               CONSTANT)
    Q_PROPERTY(QUrl subscriptionsFeed
               READ subscriptionsFeed
               CONSTANT)
    Q_PROPERTY(QUrl newSubscriptionVideosFeed
               READ newSubscriptionVideosFeed
               CONSTANT)
    Q_PROPERTY(QString username
               READ username
               NOTIFY usernameChanged)
    Q_PROPERTY(bool userSignedIn
               READ userSignedIn
               NOTIFY userSignedInChanged)
    Q_PROPERTY(QUrl authUrl
               READ authUrl
               CONSTANT)
    Q_PROPERTY(bool playlistsLoaded
               READ playlistsLoaded
               NOTIFY playlistsLoadedChanged)
    Q_PROPERTY(bool subscriptionsLoaded
               READ subscriptionsLoaded
               NOTIFY subscriptionsLoadedChanged)
    Q_PROPERTY(bool groupsLoaded
               READ groupsLoaded
               NOTIFY groupsLoadedChanged)
    Q_PROPERTY(bool busy
               READ busy
               NOTIFY busyChanged)

public:
    explicit Dailymotion(QObject *parent = 0);
    ~Dailymotion();
    inline QNetworkAccessManager* networkAccessManager() const { return m_nam; }
    void setNetworkAccessManager(QNetworkAccessManager *manager) { m_nam = manager; }
    inline QString username() const { return m_user; }
    inline bool userSignedIn() const { return !this->accessToken().isEmpty(); }
    inline QUrl uploadsFeed() const { return DAILYMOTION_UPLOADS_FEED; }
    inline QUrl favouritesFeed() const { return DAILYMOTION_FAVOURITES_FEED; }
    inline QUrl playlistsFeed() const { return DAILYMOTION_PLAYLISTS_FEED; }
    inline QUrl groupsFeed() const { return DAILYMOTION_GROUPS_FEED; }
    inline QUrl subscriptionsFeed() const { return DAILYMOTION_SUBSCRIPTIONS_FEED; }
    inline QUrl newSubscriptionVideosFeed() const { return DAILYMOTION_NEW_SUBSCRIPTION_VIDEOS_FEED; }
    QNetworkReply* createReply(QUrl feed, int offset = 0);
    QNetworkReply* createSearchReply(int queryType,
                                     const QString &query,
                                     int offset = 1,
                                     int order = Queries::Relevance,
                                     const QString &language = "all");
    QUrl authUrl() const;
    inline bool safeSearch() const { return m_safeSearch; }
    inline QList< QSharedPointer<PlaylistItem> > * playlists() const { return m_playlistCache; }
    inline QList< QSharedPointer<UserItem> > * subscriptions() const { return m_subscriptionCache; }
    inline QList< QSharedPointer<GroupItem> > * groups() const { return m_groupCache; }
    inline bool playlistsLoaded() const { return m_playlistCacheLoaded; }
    inline bool subscriptionsLoaded() const { return m_subscriptionCacheLoaded; }
    inline bool groupsLoaded() const { return m_groupCacheLoaded; }
    inline bool busy() const { return m_busy; }
    inline bool cancelled() const { return m_cancelled; }

    static Dailymotion* instance();

public slots:
    void signIn(const QString &displayName, const QUrl &response);
    void signIn(const QString &displayName, const QString &user, const QString &pass);
    void getPlaylists(int offset = 1);
    void getSubscriptions(int offset = 1);
    void getGroups(int offset = 1);
    inline void setSafeSearch(bool safe) { m_safeSearch = safe; }
    void signOut();
    void setAccount(const QString &user = QString(), const QString &token = QString(), const QString &refresh = QString());
    void deleteFromUploads(const QStringList &videoIds);
    void addToFavourites(const QStringList &videoIds);
    void deleteFromFavourites(const QStringList &videoIds);
    void addToPlaylist(const QStringList &videoIds, const QString &playlistId);
    void deleteFromPlaylist(const QStringList &videoIds, const QString &playlistId);
    void createPlaylist(const QVariantMap &playlist, const QStringList &videoIds = QStringList());
    void deletePlaylist(const QString &playlistId);
    void joinGroup(const QString &groupId);
    void leaveGroup(const QString &groupId);
    void addComment(const QVariantMap &comment);
    void subscribe(const QString &userId);
    void unsubscribe(const QString &userId);
    void updateVideoMetadata(const QVariantMap &metadata);
    void refreshAccessToken();
    void getFullVideo(QString id);
    void getCurrentUserProfile();
    void getUserProfile(const QString &id);
    bool subscribedToChannel(const QString &userId = QString());
    void getVideosFromIds(QStringList ids);
    void getVideoMetadata(const QString &id = QString());
    bool memberOfGroup(const QString &groupId = QString());
    void cancelCurrentOperation();

private:
    inline void setUsername(const QString &user) { m_user = user; emit usernameChanged(user); }
    inline QString accessToken() const { return m_token; }
    inline void setAccessToken(const QString &token) { m_token = token; emit userSignedInChanged(); }
    inline QString refreshToken() const { return m_refreshToken; }
    inline void setRefreshToken(const QString &token) { m_refreshToken = token; }
    void clearCache();
    void setPlaylistsLoaded(bool loaded) { m_playlistCacheLoaded = loaded; emit playlistsLoadedChanged(); }
    void setSubscriptionsLoaded(bool loaded) { m_subscriptionCacheLoaded = loaded; emit subscriptionsLoadedChanged(); }
    void setGroupsLoaded(bool loaded) { m_groupCacheLoaded = loaded; emit groupsLoadedChanged(); }
    void addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist);
    QSharedPointer<PlaylistItem> removePlaylistFromCache(const QString &id);
    void addNewSubscriptionToCache(QSharedPointer<UserItem> user);
    QSharedPointer<UserItem> removeSubscriptionFromCache(const QString &id);
    void getSubscriptionForCache(const QString &id);
    void getPlaylistForCache(const QString &id);
    void updatePlaylistVideoCount(const QString &id, int change);
    void addNewGroupToCache(QSharedPointer<GroupItem> group);
    QSharedPointer<GroupItem> removeGroupFromCache(const QString &id);
    void getGroupForCache(const QString &id);
    void getAddedComment(const QString &id);
    void setBusy(bool isBusy, const QString &message = QString(), int numberOfOperations = 1);
    inline void setCancelled(bool cancelled) { m_cancelled = cancelled; }

private slots:
    void checkIfSignedIn();
    void checkIfSignedOut();
    void postRequest(const QUrl &url, const QByteArray &data = QByteArray());
    void deleteRequest(const QUrl &url);
    void postFinished();
    void deleteFromUploads();
    void onVideoDeleted();
    void updateVideoMetadata();
    void onVideoMetadataUpdated();
    void addToFavourites();
    void onAddedToFavourites(const QVariantMap &response);
    void deleteFromFavourites();
    void onDeletedFromFavourites();
    void addToPlaylist();
    void onAddedToPlaylist(const QVariantMap &response);
    void deleteFromPlaylist();
    void onDeletedFromPlaylist();
    void createPlaylist();
    void onPlaylistCreated(const QVariantMap &response);
    void deletePlaylist();
    void onPlaylistDeleted();
    void subscribe();
    void onSubscribed();
    void unsubscribe();
    void onUnsubscribed();
    void joinGroup();
    void onGroupJoined();
    void leaveGroup();
    void onGroupLeft();
    void addComment();
    void onCommentAdded(const QVariantMap &response);
    void onVideoActionError(const QString &errorString);
    void onPlaylistActionError(const QString &errorString);
    void onGroupActionError(const QString &errorString);
    void onUserActionError(const QString &errorString);
    void onCommentActionError(const QString &errorString);
    void checkFullVideo();
    void checkTokenRefresh();
    void checkCurrentUserProfile();
    void checkUserProfile();
    void addPlaylists();
    void addSubscriptions();
    void addGroups();
    void checkFullVideos();
    void checkCachePlaylist();
    void checkCacheGroup();
    void checkCacheSubscription();
    void checkVideoMetadata();
    void checkAddedComment();

signals:
    void alert(const QString &message);
    void busy(const QString &message, int numberOfOperations = 1);
    void busyProgressChanged(int progress);
    void busyChanged(bool isBusy);
    void info(const QString &message);
    void error(const QString &errorString);
    void usernameChanged(const QString &username);
    void userSignedInChanged();
    void postSuccessful(const QVariantMap &response);
    void postFailed(const QString &errorString);
    void deletedFromUploads(const QString &videoId);
    void videoMetadataUpdated(const QString &videoId, const QVariantMap &metadata);
    void addedToPlaylist(const QString &videoId, const QString &playlistId);
    void deletedFromPlaylist(const QString &videoId, const QString &playlistId);
    void commentAdded(QSharedPointer<CommentItem> comment);
    void subscriptionChanged(const QString &userId, bool subscribed);
    void groupMembershipChanged(const QString &groupId, bool member);
    void favouriteChanged(const QString &videoId, bool favourite);
    void signedIn(const QString &displayName, const QString &token, const QString &refresh);
    void newAccountSet();
    void accessTokenRefreshed(const QString &token, const QString &refresh);
    void refreshError();
    void gotVideo(QSharedPointer<VideoItem> video);
    void gotVideo(VideoItem *video);
    void gotUser(QSharedPointer<UserItem> user);
    void gotUser(UserItem *user);
    void playlistsLoadedChanged();
    void subscriptionsLoadedChanged();
    void groupsLoadedChanged();
    void allPlaylistsLoaded();
    void allSubscriptionsLoaded();
    void allGroupsLoaded();
    void playlistAddedToCache(int row);
    void playlistRemovedFromCache(int row);
    void playlistUpdated(int row);
    void subscriptionAddedToCache(int row);
    void subscriptionRemovedFromCache(int row);
    void groupAddedToCache(int row);
    void groupRemovedFromCache(int row);
    void gotVideosFromIds(QList< QSharedPointer<VideoItem> > videos);
    void gotVideoMetadata(const QVariantMap &metadata);
    void currentOperationCancelled();

private:
    static Dailymotion *self;

    QNetworkAccessManager *m_nam;
    QString m_user;
    QString m_token;
    QString m_refreshToken;
    bool m_safeSearch;
    QStringList m_actionIdList;
    QString m_actionId;
    QVariantMap m_metadataAction;
    int m_actionsProcessed;
    QList< QSharedPointer<PlaylistItem> > *m_playlistCache;
    QList< QSharedPointer<UserItem> > *m_subscriptionCache;
    QList< QSharedPointer<GroupItem> > *m_groupCache;
    bool m_playlistCacheLoaded;
    bool m_subscriptionCacheLoaded;
    bool m_groupCacheLoaded;
    bool m_busy;
    bool m_cancelled;
    QHash<int, QString> m_queryOrders;
};

#endif // DAILYMOTION_H
