#include "transferitem.h"
#include "utils.h"
#include "settings.h"
#include "networkaccessmanager.h"
#include "connection.h"
#include "definitions.h"
#include "audioconverter.h"
#include "urlgrabber.h"
#include "youtube.h"
#include <QNetworkReply>
#include <QDir>

#ifdef MEEGO_EDITION_HARMATTAN
TransferUI::Client* TransferItem::tuiClient = 0;
#endif
static const qint64 MIN_FRAGMENT_SIZE = 1024 * 2048;
static const QRegExp ILLEGAL_CHARS_RE("[\"@&~=\\/:?#!|<>*^]");

TransferItem::TransferItem(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_reply(0),
    m_grabber(0),
    m_converter(0),
    m_transferType(Transfers::Download),
    m_service(Services::YouTube),
    m_videoFormat(Videos::Unknown),
    m_priority(Transfers::NormalPriority),
    m_size(0),
    m_resumePosition(0),
    m_transferredBytes(0),
    m_progress(0),
    m_speed(0),
    m_status(Transfers::Paused),
    m_convertible(false),
    m_checkedIfConvertible(false),
    m_convert(false),
    m_downloadSubtitles(false),
    m_subtitlesLanguage("en"),
    m_preferredConnections(1)
{
#ifdef MEEGO_EDITION_HARMATTAN
    if (!tuiClient) {
        tuiClient = new TransferUI::Client;
        tuiClient->init();
    }

    m_tuiTransfer = tuiClient->registerTransfer(QString(), TransferUI::Client::TRANSFER_TYPES_DOWNLOAD);
    m_tuiTransfer->waitForCommit();
    m_tuiTransfer->setTargetName(this->serviceString());
    m_tuiTransfer->setIcon("icon-m-content-videos");
    m_tuiTransfer->setCanPause(true);
    m_tuiTransfer->markPaused();
    m_tuiTransfer->commit();

    this->connect(m_tuiTransfer, SIGNAL(start()), this, SLOT(queue()));
    this->connect(m_tuiTransfer, SIGNAL(pause()), this, SLOT(pause()));
    this->connect(m_tuiTransfer, SIGNAL(cancel()), this, SLOT(cancel()));
    this->connect(m_tuiTransfer, SIGNAL(repairError()), this, SLOT(queue()));
#endif
    this->connect(&m_file, SIGNAL(writeCompleted()), this, SLOT(onFileWriteCompleted()));
    this->connect(&m_file, SIGNAL(error()), this, SLOT(onFileError()));
}

TransferItem::~TransferItem() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_tuiTransfer) {
        if (tuiClient) {
            tuiClient->removeTransfer(m_tuiTransfer->transferId());
        }

        delete m_tuiTransfer;
        m_tuiTransfer = 0;
    }
#endif
}

QVariant TransferItem::data(int role) const {
    switch (role) {
    case TitleRole:
        return this->title();
    case FileNameRole:
        return this->fileName();
    case ThumbnailUrlRole:
        return this->thumbnailUrl();
    case TransferTypeRole:
        return this->transferType();
    case TransferTypeStringRole:
        return this->transferTypeString();
    case ServiceRole:
        return this->service();
    case ServiceStringRole:
        return this->serviceString();
    case PriorityRole:
        return this->priority();
    case PriorityStringRole:
        return Transfers::priorityString(this->priority());
    case SizeRole:
        return this->size();
    case PositionRole:
        return this->position();
    case ProgressRole:
        return this->progress();
    case StatusRole:
        return this->status();
    case StatusStringRole:
        return this->statusString();
    case ConvertibleToAudioRole:
        return this->convertibleToAudio();
    case ConvertToAudioRole:
        return this->convertToAudio();
    case DownloadSubtitlesRole:
        return this->downloadSubtitles();
    case SubtitlesLanguageRole:
        return this->subtitlesLanguage();
    case PreferredConnectionsRole:
        return this->preferredConnections();
    case MaximumConnectionsRole:
        return this->maximumConnections();
    case IdRole:
        return this->id();
    case VideoIdRole:
        return this->videoId();
    default:
        return QVariant();
    }
}

QMap<int, QVariant> TransferItem::itemData() const {
    QMap<int, QVariant> map;
    map[TitleRole] = this->title();
    map[FileNameRole] = this->fileName();
    map[ThumbnailUrlRole] = this->thumbnailUrl();
    map[TransferTypeRole] = this->transferType();
    map[TransferTypeStringRole] = this->transferTypeString();
    map[ServiceRole] = this->service();
    map[ServiceStringRole] = this->serviceString();
    map[PriorityRole] = this->priority();
    map[PriorityStringRole] = this->priorityString();
    map[SizeRole] = this->size();
    map[PositionRole] = this->position();
    map[ProgressRole] = this->progress();
    map[StatusRole] = this->status();
    map[StatusStringRole] = this->statusString();
    map[ConvertibleToAudioRole] = this->convertibleToAudio();
    map[ConvertToAudioRole] = this->convertToAudio();
    map[DownloadSubtitlesRole] = this->downloadSubtitles();
    map[SubtitlesLanguageRole] = this->subtitlesLanguage();
    map[PreferredConnectionsRole] = this->preferredConnections();
    map[MaximumConnectionsRole] = this->maximumConnections();
    map[IdRole] = this->id();
    map[VideoIdRole] = this->videoId();

    return map;
}

QVariantMap TransferItem::itemDataWithRoleNames() const {
    QVariantMap map;
    map["title"] = this->title();
    map["fileName"] = this->fileName();
    map["thumbnailUrl"] = this->thumbnailUrl();
    map["transferType"] = this->transferType();
    map["transferTypeString"] = this->transferTypeString();
    map["service"] = this->service();
    map["serviceString"] = this->serviceString();
    map["priority"] = this->priority();
    map["priorityString"] = this->priorityString();
    map["size"] = this->size();
    map["position"] = this->position();
    map["progress"] = this->progress();
    map["status"] = this->status();
    map["statusString"] = this->statusString();
    map["convertibleToAudio"] = this->convertibleToAudio();
    map["convertToAudio"] = this->convertToAudio();
    map["downloadSubtitles"] = this->downloadSubtitles();
    map["subtitlesLanguage"] = this->subtitlesLanguage();
    map["preferredConnections"] = this->preferredConnections();
    map["maximumConnections"] = this->maximumConnections();
    map["id"] = this->id();
    map["videoId"] = this->videoId();

    return map;
}

bool TransferItem::setData(int role, const QVariant &value) {
    switch (role) {
    case PriorityRole:
        this->setPriority(static_cast<Transfers::Priority>(value.toInt()));
        break;
    case StatusRole:
        switch (value.toInt()) {
        case Transfers::Queued:
            this->queue();
            break;
        case Transfers::Paused:
            this->pause();
            break;
        case Transfers::Cancelled:
            this->cancel();
            break;
        }

        break;
    case ConvertToAudioRole:
        this->setConvertToAudio((this->convertibleToAudio()) && (value.toBool()));
        break;
    case PreferredConnectionsRole:
        this->setPreferredConnections(value.toInt());
        break;
    case FileNameRole:
        this->setFileName(value.toString());
        break;
    default:
        return false;
    }

    return true;
}

Transfers::TransferType TransferItem::transferType() const {
    return m_transferType;
}

void TransferItem::setTransferType(Transfers::TransferType type) {
    m_transferType = type;
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_tuiTransfer) {
        switch (type) {
        case Transfers::Upload:
            m_tuiTransfer->setTransferType(TransferUI::Client::TRANSFER_TYPES_UPLOAD);
            return;
        default:
            m_tuiTransfer->waitForCommit();
            m_tuiTransfer->setTransferType(TransferUI::Client::TRANSFER_TYPES_DOWNLOAD);
            m_tuiTransfer->setCanPause(true);
            m_tuiTransfer->commit();
            return;
        }
    }
#endif
}

QString TransferItem::transferTypeString() const {
    return Transfers::transferTypeString(this->transferType());
}

QString TransferItem::id() const {
    return m_id;
}

void TransferItem::setId(const QString &id) {
    m_id = id;
}

QString TransferItem::videoId() const {
    return m_videoId;
}

void TransferItem::setVideoId(const QString &id) {
    m_videoId = id;
}

QUrl TransferItem::thumbnailUrl() const {
    return m_thumbnailUrl;
}

void TransferItem::setThumbnailUrl(const QUrl &url) {
    m_thumbnailUrl = url;
}

QString TransferItem::fileName() const {
    return m_fileName;
}

void TransferItem::setFileName(const QString &fileName) {
    if (fileName != this->fileName()) {
        m_fileName = fileName;

        switch (this->transferType()) {
        case Transfers::Download:
            m_fileName.replace(ILLEGAL_CHARS_RE, "_");
            break;
        default:
            break;
        }

        emit fileNameChanged(m_fileName);
    }
}

QString TransferItem::downloadPath() const {
    return m_downloadPath;
}

void TransferItem::setDownloadPath(const QString &path) {
    m_downloadPath = path.endsWith('/') ? path : path + '/';
}

Services::VideoService TransferItem::service() const {
    return m_service;
}

void TransferItem::setService(Services::VideoService service) {
    m_service = service;
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_tuiTransfer) {
        m_tuiTransfer->setTargetName(this->serviceString());
    }
#endif
}

QString TransferItem::serviceString() const {
    return Services::serviceString(this->service());
}

Videos::VideoFormat TransferItem::videoFormat() const {
    return m_videoFormat;
}

void TransferItem::setVideoFormat(Videos::VideoFormat format) {
    m_videoFormat = format;
}

QString TransferItem::title() const {
    return m_title;
}

void TransferItem::setTitle(const QString &title) {
    m_title = title;
#ifdef MEEGO_EDITION_HARMATTAN
    if (m_tuiTransfer) {
        m_tuiTransfer->setName(title);
    }
#endif
}

QVariantMap TransferItem::uploadMetadata() const {
    return m_uploadMetadata;
}

void TransferItem::setUploadMetadata(const QVariantMap &metadata) {
    m_uploadMetadata = metadata;
}

Transfers::Priority TransferItem::priority() const {
    return m_priority;
}

void TransferItem::setPriority(Transfers::Priority priority) {
    if (priority != this->priority()) {
        m_priority = priority;
        emit priorityChanged(priority);
    }
}

QString TransferItem::priorityString() const {
    return Transfers::priorityString(this->priority());
}

qint64 TransferItem::size() const {
    return m_size;
}

void TransferItem::setSize(qint64 size) {
    if ((size > 0) && (size != this->size())) {
        m_size = size;
        emit sizeChanged(size);
#ifdef MEEGO_EDITION_HARMATTAN
        if (m_tuiTransfer) {
            m_tuiTransfer->setSize(size);
        }
#endif
    }
}

qint64 TransferItem::resumePosition() const {
    return m_resumePosition;
}

void TransferItem::setResumePosition(qint64 position) {
    if (position != this->resumePosition()) {
        m_resumePosition = position;
        emit positionChanged(position);

        if ((this->position() > 0) && (this->size() > 0)) {
            this->setProgress(position * 100 / this->size());
        }
    }
}

qint64 TransferItem::position() const {
    return this->resumePosition() + m_transferredBytes;
}

int TransferItem::progress() const {
    return m_progress;
}

void TransferItem::setProgress(int progress) {
    if (progress != this->progress()) {
        m_progress = progress;
        emit progressChanged(progress);
#ifdef MEEGO_EDITION_HARMATTAN
        if ((tuiClient) && (tuiClient->isTUIVisible())) {
            if (m_tuiTransfer) {
                m_tuiTransfer->setProgress(float(progress) / 100);
            }
        }
#endif
    }
}

int TransferItem::speed() const {
    switch (this->status()) {
    case Transfers::Downloading:
    case Transfers::Uploading:
        if ((m_transferredBytes > 0) && (m_transferTime.elapsed() > 0)) {
            return m_transferredBytes / m_transferTime.elapsed();
        }

        break;
    default:
        break;
    }

    return 0;
}

Transfers::Status TransferItem::status() const {
    return m_status;
}

void TransferItem::setStatus(Transfers::Status status) {
    if (status != this->status()) {
        m_status = status;
#ifdef MEEGO_EDITION_HARMATTAN
        switch (status) {
        case Transfers::Queued:
            if (m_tuiTransfer) m_tuiTransfer->setPending(this->statusString());
            break;
        case Transfers::Downloading:
        case Transfers::Uploading:
            if (m_tuiTransfer) m_tuiTransfer->markResumed();
            m_transferTime.start();
            break;
        case Transfers::Completed:
            if (m_tuiTransfer) m_tuiTransfer->markCompleted();
            break;
        case Transfers::Cancelled:
            if (m_tuiTransfer) m_tuiTransfer->markCancelled();
            switch (this->transferType()) {
            case Transfers::Upload:
                break;
            default:
                this->removeFiles();
                break;
            }

            break;
        case Transfers::Paused:
            if (m_tuiTransfer) m_tuiTransfer->markPaused();

            if (m_grabber) {
                m_grabber->cancelCurrentOperation();
                this->disconnect(m_grabber, SIGNAL(error(QString)), this, 0);
            }

            break;
        case Transfers::Failed:
            if (m_tuiTransfer) m_tuiTransfer->markRepairableFailure(tr("Failed"), this->statusInfo(), tr("Retry"));

            if (m_grabber) {
                m_grabber->cancelCurrentOperation();
                this->disconnect(m_grabber, SIGNAL(error(QString)), this, 0);
            }

            break;
        default:
            break;
        }
#else
        switch (status) {
        case Transfers::Downloading:
        case Transfers::Uploading:
            m_transferTime.start();
            break;
        case Transfers::Cancelled:
            switch (this->transferType()) {
            case Transfers::Upload:
                break;
            default:
                this->removeFiles();
                break;
            }

            break;
        case Transfers::Paused:
        case Transfers::Failed:
            if (m_grabber) {
                m_grabber->cancelCurrentOperation();
                this->disconnect(m_grabber, SIGNAL(error(QString)), this, 0);
            }

            break;
        default:
            break;
        }
#endif
        emit statusChanged(status);
    }
}

QString TransferItem::statusString() const {
    switch (this->status()) {
    case Transfers::Failed:
        return QString("%1 - %2").arg(Transfers::statusString(this->status())).arg(this->statusInfo());
    default:
        return Transfers::statusString(this->status());
    }
}

QString TransferItem::statusInfo() const {
    return m_statusInfo;
}

void TransferItem::setStatusInfo(const QString &info) {
    if (info != this->statusInfo()) {
        m_statusInfo = info;
        emit statusInfoChanged(info);
    }
}

bool TransferItem::convertibleToAudio() const {
    if ((!m_checkedIfConvertible) && (!this->fileName().isEmpty())) {
        m_convertible = QFile::exists("/usr/bin/ffmpeg");
        m_checkedIfConvertible = true;
    }

    return m_convertible;
}

bool TransferItem::convertToAudio() const {
    return m_convert;
}

void TransferItem::setConvertToAudio(bool convert) {
    if (convert != this->convertToAudio()) {
        if ((!convert) || (this->convertibleToAudio())) {
            m_convert = convert;
            emit convertToAudioChanged(convert);
        }
    }
}

bool TransferItem::downloadSubtitles() const {
    return m_downloadSubtitles;
}

void TransferItem::setDownloadSubtitles(bool download, bool overrideGlobalSetting) {
    if (download != this->downloadSubtitles()) {
        m_downloadSubtitles = download;
        emit downloadSubtitlesChanged(download);

        if (overrideGlobalSetting) {
            this->disconnect(Settings::instance(), SIGNAL(downloadSubtitlesChanged(bool)), this, SLOT(onDownloadSubtitlesChanged(bool)));
        }
        else {
            this->connect(Settings::instance(), SIGNAL(downloadSubtitlesChanged(bool)), this, SLOT(onDownloadSubtitlesChanged(bool)));
        }
    }
}

void TransferItem::onDownloadSubtitlesChanged(bool download) {
    if (download != this->downloadSubtitles()) {
        m_downloadSubtitles = download;
        emit downloadSubtitlesChanged(download);
    }
}

QString TransferItem::subtitlesLanguage() const {
    return m_subtitlesLanguage;
}

void TransferItem::setSubtitlesLanguage(const QString &language, bool overrideGlobalSetting) {
    if (language != this->subtitlesLanguage()) {
        m_subtitlesLanguage = language;
        emit subtitlesLanguageChanged(language);

        if (overrideGlobalSetting) {
            this->disconnect(Settings::instance(), SIGNAL(subtitlesLanguageChanged(QString)), this, SLOT(onSubtitlesLanguageChanged(QString)));
        }
        else {
            this->connect(Settings::instance(), SIGNAL(subtitlesLanguageChanged(QString)), this, SLOT(onSubtitlesLanguageChanged(QString)));
        }
    }
}

void TransferItem::onSubtitlesLanguageChanged(const QString &language) {
    this->setSubtitlesLanguage(language, false);
}

int TransferItem::preferredConnections() const {
    return m_preferredConnections;
}

void TransferItem::setPreferredConnections(int pref, bool overrideGlobalSetting) {
    if ((pref != this->preferredConnections()) && (pref > 0) && (pref < this->maximumConnections())) {
        m_preferredConnections = pref;
        emit preferredConnectionsChanged(pref);

        if (overrideGlobalSetting) {
            this->disconnect(Settings::instance(), SIGNAL(maximumConnectionsPerTransferChanged(int,int)), this, SLOT(onMaximumConnectionsChanged(int,int)));
        }
        else {
            this->connect(Settings::instance(), SIGNAL(maximumConnectionsPerTransferChanged(int,int)), this, SLOT(onMaximumConnectionsChanged(int,int)));
        }

        if (this->status() == Transfers::Downloading) {
            if (pref > this->activeConnections()) {
                this->addConnections(pref - this->activeConnections());
            }
            else if (pref < this->activeConnections()) {
                this->removeConnections(this->activeConnections() - pref);
            }
        }
    }
}

int TransferItem::maximumConnections() const {
    switch (this->transferType()) {
    case Transfers::Upload:
        return 1;
    default:
        return MAX_CONNECTIONS;
    }
}

void TransferItem::onMaximumConnectionsChanged(int oldMaximum, int newMaximum) {
    Q_UNUSED(oldMaximum)

    this->setPreferredConnections(newMaximum, false);
}

int TransferItem::activeConnections() const {
    int active = 0;

    foreach (Connection *connection, m_connections) {
        if (connection->status() == Transfers::Downloading) {
            active += 1;
        }
    }

    return active;
}

QList<Connection*> TransferItem::connections() const {
    return m_connections;
}

void TransferItem::loadConnections() {
    if (m_connections.isEmpty()) {
        this->addConnection(0, 0);
    }
    else {
        MetaInfo info;
        info.path = this->downloadPath();
        info.name = this->fileName();
        info.size = this->size();
        info.bytesRemaining = this->size() - this->position();

        m_file.setMetaInfo(info);
        m_file.start(QThread::LowestPriority);

        this->addConnections(this->preferredConnections());
    }
}

void TransferItem::restoreConnection(qint64 start, qint64 end) {
    this->addConnection(start, end, false);
}

void TransferItem::addConnection(qint64 start, qint64 end, bool startWhenAdded) {
    if (!m_nam) {
        m_nam = new NetworkAccessManager(this);
    }

    qDebug() << "Adding connection with range start:" << start << "end:" << end;

    Connection *connection = new Connection(m_nam, this);
    connection->setTransferType(this->transferType());
    connection->setRequest(m_request);
    connection->setContentRange(start, end);

    switch (this->transferType()) {
    case Transfers::Upload:
        connection->setUploadFile(this->fileName());
        break;
    default:
        break;
    }

    m_connections.append(connection);
    this->connect(connection, SIGNAL(dataAvailable(qint64,QByteArray)), this, SLOT(onDataAvailable(qint64,QByteArray)));
    this->connect(connection, SIGNAL(bytesTransferred(qint64)), this, SLOT(onBytesTransferred(qint64)));
    this->connect(connection, SIGNAL(statusChanged(Transfers::Status)), this, SLOT(onConnectionStatusChanged(Transfers::Status)));

    if (m_connections.size() == 1) {
        this->connect(connection, SIGNAL(metaInfoReady(MetaInfo)), this, SLOT(onMetaInfoReady(MetaInfo)));
    }

    if (startWhenAdded) {
        connection->start();
    }
}

void TransferItem::addConnections(int count) {
    qDebug() << "Adding" << count << "connections";

    if (count <= 0) {
        return;
    }

    int added = 0;
    int i = 0;

    while ((added < count) && (i < m_connections.size())) {
        switch (m_connections.at(i)->status()) {
        case Transfers::Downloading:
        case Transfers::Uploading:
            break;
        default:
            m_connections.at(i)->setRequest(m_request);
            m_connections.at(i)->start();
            added++;
            break;
        }

        i++;
    }

    i = 0;

    while ((added < count) && (i < m_connections.size())) {
        qint64 end = m_connections.at(i)->contentRangeEnd();

        if (end > 0) {
            qint64 pos = m_connections.at(i)->position();
            qint64 start = (end - ((end - pos) / 2));

            if ((end - start) > MIN_FRAGMENT_SIZE) {
                m_connections.at(i)->setContentRangeEnd(start);
                this->addConnection(start, end);
                added++;
            }
        }

        i++;
    }
}

void TransferItem::removeConnections(int count) {
    qDebug() << "Removing" << count << "connections";

    if (count <= 0) {
        return;
    }

    int removed = 0;
    int i = 0;

    while ((removed < count) && (i < m_connections.size())) {
        switch (m_connections.at(i)->status()) {
        case Transfers::Downloading:
        case Transfers::Uploading:
            m_connections.at(i)->pause();
            removed++;
            break;
        default:
            break;
        }

        i++;
    }
}

void TransferItem::queue() {
    switch (this->status()) {
    case Transfers::Paused:
    case Transfers::Failed:
        this->setStatus(Transfers::Queued);
        return;
    default:
        return;
    }
}

void TransferItem::start() {
    switch(this->status()) {
    case Transfers::Paused:
    case Transfers::Failed:
    case Transfers::Queued:
        break;
    default:
        return;
    }

    if (!m_nam) {
        m_nam = new NetworkAccessManager(this);
    }

    qint64 bytes = m_transferredBytes;

    if (bytes > 0) {
        m_transferredBytes = 0;
        this->setResumePosition(this->resumePosition() + bytes);
    }

    this->setStatus(Transfers::Connecting);

    switch (this->transferType()) {
    case Transfers::Upload:
        this->startUpload();
        return;
    default:
        this->startDownload();
        return;
    }
}

void TransferItem::pause() {
    switch (this->status()) {
    case Transfers::Uploading:
    case Transfers::Converting:
    case Transfers::Failed:
        return;
    case Transfers::Downloading:
        foreach (Connection *connection, m_connections) {
            connection->pause();
        }

        if (m_reply) {
            m_reply->abort();
        }

        return;
    default:
        this->setStatus(Transfers::Paused);
        return;
    }
}

void TransferItem::cancel() {
    switch (this->status()) {
    case Transfers::Converting:
        return;
    case Transfers::Downloading:
    case Transfers::Uploading:
        foreach (Connection *connection, m_connections) {
            connection->cancel();
        }

        if (m_reply) {
            m_reply->abort();
        }

        return;
    default:
        this->setStatus(Transfers::Cancelled);
        return;
    }
}

void TransferItem::startDownload() {
    qDebug() << "Starting download: " + this->title();

    if (!m_grabber) {
        m_grabber = new UrlGrabber(this);
        m_grabber->setNetworkAccessManager(m_nam);
        this->connect(m_grabber, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), this, SLOT(downloadVideo(QUrl,Videos::VideoFormat)));
        this->connect(m_grabber, SIGNAL(gotSubtitlesUrl(QUrl)), this, SLOT(startSubtitlesDownload(QUrl)));
    }

    switch (this->service()) {
    case Services::YouTube:
        switch (this->videoFormat()) {
        case Videos::Unknown:
            m_grabber->setYouTubeFormats(Settings::instance()->youtubeDownloadFormats());
            break;
        default:
            m_grabber->setYouTubeFormats(QSet<int>() << this->videoFormat());
            break;
        }

        break;
    case Services::Dailymotion:
        switch (this->videoFormat()) {
        case Videos::Unknown:
            m_grabber->setDailymotionFormats(Settings::instance()->dailymotionDownloadFormats());
            break;
        default:
            m_grabber->setDailymotionFormats(QSet<int>() << this->videoFormat());
            break;
        }

        break;
    case Services::Vimeo:
        switch (this->videoFormat()) {
        case Videos::Unknown:
            m_grabber->setVimeoFormats(Settings::instance()->vimeoDownloadFormats());
            break;
        default:
            m_grabber->setVimeoFormats(QSet<int>() << this->videoFormat());
            break;
        }

        break;
    default:
        qWarning() << "TransferItem::startTransfer(): No/invalid service.";
        this->setStatusInfo(tr("Invalid service"));
        this->setStatus(Transfers::Failed);
        return;
    }

    this->connect(m_grabber, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
    m_grabber->getVideoUrl(this->service(), this->videoId());
}

void TransferItem::downloadVideo(const QUrl &url, Videos::VideoFormat format) {
    switch (format) {
    case Videos::Unknown:
        break;
    default:
        this->setVideoFormat(format);
        break;
    }

    m_request = QNetworkRequest(url);
    this->disconnect(m_grabber, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
    this->loadConnections();
}

void TransferItem::onUrlError(const QString &errorString) {
    this->disconnect(m_grabber, SIGNAL(error(QString)), this, SLOT(onUrlError(QString)));
    this->setStatusInfo(errorString);
    this->setStatus(Transfers::Failed);
}

void TransferItem::startUpload() {
    qDebug() << "Starting upload: " + this->title();
    m_reply = YouTube::instance()->createUploadReply(this->uploadMetadata());
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(checkUploadUrl()));
}

void TransferItem::checkUploadUrl() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::OperationCanceledError:
        m_reply->deleteLater();
        m_reply = 0;
        emit statusChanged(Transfers::Cancelled);
        return;
    default:
        break;
    }

    int statusCode = m_reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    switch (statusCode) {
    case 401:
        m_reply->deleteLater();
        m_reply = 0;
        this->connect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startUpload()));
        YouTube::instance()->refreshAccessToken();
        return;
    case 200:
    {
        QUrl url(m_reply->rawHeader("Location"));
        m_reply->deleteLater();
        m_reply = 0;
        this->uploadVideo(url);
        break;
    }
    default:
        this->setStatusInfo(m_reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString());
        m_reply->deleteLater();
        m_reply = 0;
        this->setStatus(Transfers::Failed);
        break;
    }

    this->disconnect(YouTube::instance(), SIGNAL(accessTokenRefreshed(QString)), this, SLOT(startUpload()));
}

void TransferItem::uploadVideo(const QUrl &url) {
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/octet-stream");
    m_request = request;
    this->loadConnections();
}

void TransferItem::onMetaInfoReady(MetaInfo info) {
    this->setSize(info.size);

    switch (this->transferType()) {
    case Transfers::Upload:
        break;
    default:
        if (info.name.size() > this->fileName().size()) {
            this->setFileName(info.name);
        }
        else {
            info.name = this->fileName();
        }

        info.path = this->downloadPath();
        info.name = this->fileName();
        m_file.setMetaInfo(info);
        m_file.start(QThread::LowestPriority);

        if (!m_connections.isEmpty()) {
            m_connections.first()->processData();
        }

        break;
    }

    if (this->preferredConnections() > 1) {
        this->addConnections(this->preferredConnections() - this->activeConnections());
    }
}

void TransferItem::onBytesTransferred(qint64 bytes) {
    m_transferredBytes += bytes;
    emit positionChanged(this->position());
    emit speedChanged(this->speed());

    if (this->size() > 0) {
        this->setProgress(this->position() * 100 / this->size());
    }
}

void TransferItem::onDataAvailable(qint64 offset, const QByteArray &data) {
    m_file.write(offset, data);
}

void TransferItem::onFileError() {
    foreach (Connection *connection, m_connections) {
        connection->pause();
    }

    this->setStatusInfo(m_file.errorString());
    this->setStatus(Transfers::Failed);
}

void TransferItem::onConnectionStatusChanged(Transfers::Status status) {
    switch (status) {
    case Transfers::Downloading:
    case Transfers::Uploading:
        break;
    case Transfers::Completed:
        if (Connection* connection = qobject_cast<Connection*>(this->sender())) {
            this->onConnectionCompleted(connection);
            return;
        }

        break;
    case Transfers::Failed:
        if (this->activeConnections() > 0) {
            return;
        }

        if (Connection* connection = qobject_cast<Connection*>(this->sender())) {
            this->setStatusInfo(connection->errorString());
        }

        break;
    case Transfers::Cancelled:
        foreach (Connection *connection, m_connections) {
            if (connection->status() != status) {
                return;
            }
        }

        break;
    default:
        foreach (Connection *connection, m_connections) {
            if (connection->status() != status) {
                return;
            }
        }

        break;
    }

    this->setStatus(status);
}

void TransferItem::onConnectionCompleted(Connection *connection) {
    if (connection) {
        m_connections.removeOne(connection);
        connection->deleteLater();

        if (m_connections.isEmpty()) {
            switch (this->transferType()) {
            case Transfers::Upload:
                this->setStatus(Transfers::Completed);
                return;
            default:
                if (this->size() <= 0) {
                    this->onFileWriteCompleted();
                }

                return;
            }
        }
        else if ((this->activeConnections() < this->preferredConnections()) && ((this->size() - this->position()) > MIN_FRAGMENT_SIZE)) {
            this->addConnections();
        }
    }
}

void TransferItem::onFileWriteCompleted() {
    switch (this->transferType()) {
    case Transfers::Upload:
        this->setStatus(Transfers::Completed);
        return;
    default:
        this->startThumbnailDownload();
        return;
    }
}

void TransferItem::startThumbnailDownload() {
    QNetworkRequest request(this->thumbnailUrl());
    m_reply = m_nam->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onThumbnailDownloadFinished()));
}

void TransferItem::onThumbnailDownloadFinished() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::OperationCanceledError:
        m_reply->deleteLater();
        m_reply = 0;
        emit statusChanged(Transfers::Cancelled);
        return;
    case QNetworkReply::NoError:
        break;
    default:
        m_reply->deleteLater();
        m_reply = 0;

        if (this->downloadSubtitles()) {
            this->startSubtitlesDownload();
        }
        else if (this->convertToAudio()) {
            this->startAudioConversion();
        }
        else {
            QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
        }

        return;
    }

    QFile file(this->downloadPath() + this->fileName().left(this->fileName().lastIndexOf('.')) + ".jpg");

    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
    }

    m_reply->deleteLater();
    m_reply = 0;

    if (this->downloadSubtitles()) {
        this->startSubtitlesDownload();
    }
    else if (this->convertToAudio()) {
        this->startAudioConversion();
    }
    else {
        QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
    }
}

void TransferItem::startSubtitlesDownload(const QUrl &url) {
    if (url.isEmpty()) {
        if (!m_grabber) {
            m_grabber = new UrlGrabber(this);
            m_grabber->setNetworkAccessManager(m_nam);
            this->connect(m_grabber, SIGNAL(gotVideoUrl(QUrl,Videos::VideoFormat)), this, SLOT(downloadVideo(QUrl,Videos::VideoFormat)));
            this->connect(m_grabber, SIGNAL(gotSubtitlesUrl(QUrl)), this, SLOT(startSubtitlesDownload(QUrl)));
        }

        qDebug() << "Fetching subtitles for" << this->subtitlesLanguage();

        this->connect(m_grabber, SIGNAL(error(QString)), this, SLOT(onSubtitlesError()));
        m_grabber->getSubtitlesUrl(this->service(), this->videoId(), this->subtitlesLanguage());
    }
    else {
        qDebug() << "Downloading subtitles:" << url;

        QNetworkRequest request(url);
        m_reply = m_nam->get(request);
        this->connect(m_reply, SIGNAL(finished()), this, SLOT(onSubtitlesDownloadFinished()));
    }
}

void TransferItem::onSubtitlesDownloadFinished() {
    if (!m_reply) {
        this->setStatusInfo(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    switch (m_reply->error()) {
    case QNetworkReply::OperationCanceledError:
        m_reply->deleteLater();
        m_reply = 0;
        emit statusChanged(Transfers::Cancelled);
        return;
    case QNetworkReply::NoError:
        break;
    default:
        m_reply->deleteLater();
        m_reply = 0;
        QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
        return;
    }

    QFile file(this->downloadPath() + this->fileName().left(this->fileName().lastIndexOf('.')) + ".srt");
    
    if (file.open(QIODevice::WriteOnly)) {
        file.write(m_reply->readAll());
        file.close();
    }

    m_reply->deleteLater();
    m_reply = 0;
    QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
}

void TransferItem::onSubtitlesError() {
    qDebug() << "No subtitles found for" << this->subtitlesLanguage();

    this->disconnect(m_grabber, SIGNAL(error(QString)), this, SLOT(onSubtitlesError()));
    QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
}

void TransferItem::startAudioConversion() {
    this->setStatus(Transfers::Converting);

    if (!m_converter) {
        m_converter = new AudioConverter(this);
        this->connect(m_converter, SIGNAL(finished()), this, SLOT(onAudioConversionFinished()));
        this->connect(m_converter, SIGNAL(error()), this, SLOT(onAudioConversionError()));
    }

    qDebug() << "Converting to audio file";

    m_converter->start(this->downloadPath() + this->fileName(), this->downloadPath());
}

void TransferItem::onAudioConversionFinished() {
    m_file.remove();
    this->setFileName(this->fileName().left(this->fileName().lastIndexOf('.')) + ".m4a");
    QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
}

void TransferItem::onAudioConversionError() {
    qDebug() << "Error converting to audio file:" << m_converter->errorString();

    QMetaObject::invokeMethod(this, "moveFiles", Qt::QueuedConnection);
}

void TransferItem::moveFiles() {
    QDir destinationDir(Settings::instance()->downloadPath());

    if (!destinationDir.mkpath(destinationDir.path())) {
        this->setStatusInfo(tr("Cannot move downloaded files"));
        this->setStatus(Transfers::Failed);
        return;
    }

    QDir downloadDir(this->downloadPath());
    int i = 0;

    foreach (QString oldFileName, downloadDir.entryList(QDir::Files, QDir::Time | QDir::Reversed)) {
        QString fileName = QString("%1/%2%3")
                .arg(destinationDir.path())
                .arg(oldFileName.endsWith(".jpg") ? ".thumbnails/" : "")
                .arg(i == 0 ? oldFileName : QString("%1(%2)%3").arg(oldFileName.left(oldFileName.lastIndexOf('.'))).arg(i).arg(oldFileName.mid(oldFileName.lastIndexOf('.'))));

        while ((destinationDir.exists(fileName)) && (i < 100)) {
            i++;
            fileName = (i == 1 ? QString("%1(%2)%3").arg(fileName.left(fileName.lastIndexOf('.'))).arg(i).arg(fileName.mid(fileName.lastIndexOf('.')))
                               : QString("%1(%2)%3").arg(fileName.left(fileName.lastIndexOf('('))).arg(i).arg(fileName.mid(fileName.lastIndexOf('.'))));
        }

        qDebug() << "Renaming downloaded file to:" << fileName;

        if (!destinationDir.rename(downloadDir.absoluteFilePath(oldFileName), fileName)) {
            this->setStatusInfo(tr("Cannot move downloaded files"));
            this->setStatus(Transfers::Failed);
            return;
        }
    }

    downloadDir.rmdir(downloadDir.path());
    this->setStatus(Transfers::Completed);
}

void TransferItem::removeFiles() {
    m_file.remove();
    QDir dir(this->downloadPath());

    foreach (QString fileName, dir.entryList(QDir::Files)) {
        dir.remove(dir.absoluteFilePath(fileName));
    }

    dir.rmdir(dir.path());
}
