#include "dailymotionplaylistlistmodel.h"
#include "feedurls.h"
#include "json.h"
#include "dailymotion.h"
#include <QNetworkReply>
#include <QNetworkRequest>
#include <QTimer>
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

using namespace QtJson;

DailymotionPlaylistListModel::DailymotionPlaylistListModel(QObject *parent) :
    AbstractPlaylistListModel(parent),
    m_loading(true),
    m_moreResults(true),
    m_error(false),
    m_offset(1)
{
}

void DailymotionPlaylistListModel::clear() {
    AbstractPlaylistListModel::clear();
    this->setLoading(false);
    this->setOffset(1);
    this->setMoreResults(true);
    this->setError(false);
}

void DailymotionPlaylistListModel::reset() {
    if (!this->loading()) {
        this->clear();
        this->getMorePlaylists();
    }
}

void DailymotionPlaylistListModel::setFeed(const QUrl &feed) {
    if (feed != this->feed()) {
        m_feed = feed;
    }
}

void DailymotionPlaylistListModel::getPlaylists(QUrl feed) {
    if (!feed.isEmpty()) {
        this->setFeed(feed);
    }
    else {
        feed = this->feed();
    }
#if QT_VERSION >= 0x050000
    QUrlQuery query(feed);
    query.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
    feed.setQuery(query);
#else
    feed.addQueryItem("fields", DAILYMOTION_PLAYLIST_FIELDS);
#endif
    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createReply(feed, this->offset());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::search(const QString &query, int order) {
    if (!query.isEmpty()) {
        this->setQuery(query);
        this->setOrder(order);
    }

    this->setLoading(true);
    QNetworkReply *reply = Dailymotion::instance()->createSearchReply(Queries::Playlists, this->query(), this->offset(), this->order());
    reply->setParent(this);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void DailymotionPlaylistListModel::addPlaylists() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setLoading(false);
        this->setError(true);
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        this->connect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
        Dailymotion::instance()->refreshAccessToken();
    }
    else {
        QString response(reply->readAll());
        bool ok;
        QVariantMap res = Json::parse(response, ok).toMap();

        if (!ok) {
            this->setLoading(false);
            this->setError(true);
        }
        else {
            QVariantList entries = res.value("list").toList();

            for (int i = 0; i < entries.size(); i++) {
                PlaylistItem *playlist = new PlaylistItem;
                playlist->loadDailymotionPlaylist(entries.at(i).toMap());
                QMetaObject::invokeMethod(this, "appendPlaylist", Qt::QueuedConnection, Q_ARG(QSharedPointer<PlaylistItem>, QSharedPointer<PlaylistItem>(playlist)));
            }

            QTimer::singleShot(1000, this, SLOT(stopLoading()));
            this->setMoreResults(res.value("has_more").toBool());
            this->setOffset(res.value("page").toInt() + 1);
        }

        this->disconnect(Dailymotion::instance(), SIGNAL(accessTokenRefreshed(QString,QString)), this, SLOT(getPlaylists()));
        this->disconnect(Dailymotion::instance(), SIGNAL(refreshError()), this, SLOT(onError()));
    }

    emit countChanged(this->rowCount());

    reply->deleteLater();
}

void DailymotionPlaylistListModel::getMorePlaylists() {
    if ((this->moreResults()) && (!this->loading())) {
        if (!this->query().isEmpty()) {
            this->search();
        }
        else {
            this->getPlaylists();
        }
    }
}
