#include "dailymotionaccountslistmodel.h"
#include "dailymotion.h"
#include "notifications.h"
#include <QSqlRecord>
#include <QSqlField>

DailymotionAccountsListModel::DailymotionAccountsListModel(QObject *parent) :
    QSqlTableModel(parent, QSqlDatabase::database())
{
    m_roleNames[DisplayNameRole] = "displayName";
    m_roleNames[UsernameRole] = "username";
    m_roleNames[AccessTokenRole] = "accessToken";
    m_roleNames[RefreshTokenRole] = "refreshToken";
    m_roleNames[ActiveRole] = "active";
#if QT_VERSION < 0x050000
    this->setRoleNames(m_roleNames);
#endif
    this->setTable("dailymotionAccounts");
    this->setEditStrategy(QSqlTableModel::OnManualSubmit);
    this->select();

    this->connect(Dailymotion::instance(), SIGNAL(signedIn(QString,QString,QString)), this, SLOT(addAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountSelected(QString,QString,QString)), Dailymotion::instance(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountAdded(QString,QString,QString)), Dailymotion::instance(), SLOT(setAccount(QString,QString,QString)));
    this->connect(this, SIGNAL(accountDeleted()), Dailymotion::instance(), SLOT(signOut()));
    this->connect(this, SIGNAL(error(QString)), Notifications::instance(), SLOT(onError(QString)));
}

DailymotionAccountsListModel::~DailymotionAccountsListModel() {
    this->clear();
}

#if QT_VERSION >= 0x050000
QHash<int, QByteArray> DailymotionAccountsListModel::roleNames() const {
    return m_roleNames;
}
#endif

void DailymotionAccountsListModel::setActiveAccount(int row) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(this->data(this->index(row, 0), DisplayNameRole));
    usernameField.setValue(this->data(this->index(row, 0), UsernameRole));
    tokenField.setValue(this->data(this->index(row, 0), AccessTokenRole));
    refreshField.setValue(this->data(this->index(row, 0), RefreshTokenRole));
    activeField.setValue(1);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(activeField);

    this->setRecord(row, rec);

    for (int i = 0; i < this->rowCount(); i++) {
        if (i != row) {
            nameField.setValue(this->data(this->index(i, 0), DisplayNameRole));
            usernameField.setValue(this->data(this->index(i, 0), UsernameRole));
            tokenField.setValue(this->data(this->index(i, 0), AccessTokenRole));
            refreshField.setValue(this->data(this->index(i, 0), RefreshTokenRole));
            activeField.setValue(0);

            rec.clear();
            rec.append(nameField);
            rec.append(usernameField);
            rec.append(tokenField);
            rec.append(refreshField);
            rec.append(activeField);

            this->setRecord(i, rec);
        }
    }

    this->submitAll();
}

void DailymotionAccountsListModel::switchAccount(int row) {
    this->setActiveAccount(row);

    this->database().open();
    QString user = this->data(this->index(row, 0), UsernameRole).toString();
    QString token = this->data(this->index(row, 0), AccessTokenRole).toString();
    QString refresh = this->data(this->index(row, 0), RefreshTokenRole).toString();

    emit accountSelected(user, token, refresh);
}

QVariant DailymotionAccountsListModel::data(const QModelIndex &idx, int role) const {
    return QSqlTableModel::data(this->index(idx.row(), role));
}

QVariant DailymotionAccountsListModel::data(int row, const QByteArray &role) const {
    return this->data(this->index(row, 0), this->roleNames().key(role));
}

void DailymotionAccountsListModel::addAccount(const QString &displayName, const QString &token, const QString &refresh) {
    this->database().open();
    QSqlField nameField("name", QVariant::String);
    QSqlField usernameField("username", QVariant::String);
    QSqlField tokenField("token", QVariant::String);
    QSqlField refreshField("refresh", QVariant::String);
    QSqlField activeField("active", QVariant::Int);
    nameField.setValue(displayName);
    usernameField.setValue(QVariant::String);
    tokenField.setValue(token);
    refreshField.setValue(refresh);
    activeField.setValue(0);

    QSqlRecord rec;
    rec.append(nameField);
    rec.append(usernameField);
    rec.append(tokenField);
    rec.append(refreshField);
    rec.append(activeField);

    this->insertRecord(-1, rec);
    this->submitAll();
    this->setActiveAccount(this->rowCount() - 1);

    emit accountAdded(QString(), token, refresh);
    emit countChanged(this->rowCount());
}

void DailymotionAccountsListModel::deleteAccount(int row) {
    this->database().open();
    bool signOut = this->data(this->index(row, 0), ActiveRole).toBool();

    if ((this->removeRow(row)) && (this->submitAll())) {
        if (signOut) {
            emit accountDeleted();
        }

        emit countChanged(this->rowCount());
    }
    else {
        emit error(tr("Database error. Unable to delete account"));
    }
}
