#include "vimeo.h"
#include "definitions.h"
#include "oauth.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QMap>
#include <QStringList>
#include <QDebug>
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
#include <QQmlEngine>
#else
#include <QDeclarativeEngine>
#endif
#endif
#if QT_VERSION >= 0x050000
#include <QUrlQuery>
#endif

using namespace QtOAuth;

Vimeo* Vimeo::self = 0;

const QString CLIENT_ID("b73846eb250c788a08bda2458be6ae9aebfda98e");
const QString CLIENT_SECRET("6d6a7f4b5f013efe183f12d6e09be31051eb5eed");
const QString REDIRECT_URI("https://sites.google.com/site/marxodian/home/cutetube");

Vimeo::Vimeo(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_safeSearch(false),
    m_actionsProcessed(0),
    m_albumCache(new QList< QSharedPointer<PlaylistItem> >),
    m_groupCache(new QList< QSharedPointer<GroupItem> >),
    m_subscriptionCache(new QList< QSharedPointer<UserItem> >),
    m_albumCacheLoaded(false),
    m_groupCacheLoaded(false),
    m_subscriptionCacheLoaded(false),
    m_busy(false),
    m_cancelled(false)
{
    if (!self) {
        self = this;
    }

    m_queryOrders[Queries::Relevance] = "relevant";
    m_queryOrders[Queries::Date] = "newest";
    m_queryOrders[Queries::Views] = "most_played";
    m_queryOrders[Queries::Rating] = "most_liked";
}

Vimeo::~Vimeo() {
    this->clearCache();
}

Vimeo* Vimeo::instance() {
    return !self ? new Vimeo : self;
}

void Vimeo::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;
        emit busyChanged(isBusy);
    }

    if (isBusy) {
        this->setCancelled(false);
        emit busy(message, numberOfOperations);
    }
    else if (!this->cancelled()) {
        emit busyProgressChanged(numberOfOperations);
    }
}

void Vimeo::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, 0);
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    emit currentOperationCancelled();
}

QNetworkReply* Vimeo::createReply(QUrl feed, int offset) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(feed);

    if (offset) {
        query.addQueryItem("page", QString::number(offset));
    }

    feed.setQuery(query);
#else
    if (offset) {
        feed.addQueryItem("page", QString::number(offset));
    }
#endif
    qDebug() << feed;

    QNetworkRequest request(feed);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", feed, QMap<QString, QString>()));

    return this->networkAccessManager()->get(request);
}

QNetworkReply* Vimeo::createSearchReply(const QString &query, int offset, int order) {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery urlQuery(url);
    urlQuery.addQueryItem("method", "vimeo.videos.search");
    urlQuery.addQueryItem("sort", m_queryOrders.value(order, "relevant"));
    urlQuery.addQueryItem("query", query.simplified());
    urlQuery.addQueryItem("per_page", "30");
    urlQuery.addQueryItem("full_response", "true");

    if (offset) {
        urlQuery.addQueryItem("page", QString::number(offset));
    }

    url.setQuery(urlQuery);
#else
    url.addQueryItem("method", "vimeo.videos.search");
    url.addQueryItem("sort", m_queryOrders.value(order, "relevant"));
    url.addQueryItem("query", query.simplified());
    url.addQueryItem("per_page", "30");
    url.addQueryItem("full_response", "true");

    if (offset) {
        url.addQueryItem("page", QString::number(offset));
    }
#endif
    qDebug() << url;

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));

    return this->networkAccessManager()->get(request);
}

QByteArray Vimeo::getOAuthHeader(const QString &method, const QUrl &url, QMap<QString, QString> params) {
    params.insert("oauth_consumer_key", CLIENT_ID);
    params.insert("oauth_consumer_secret", CLIENT_SECRET);

    if (this->userSignedIn()) {
        params.insert("oauth_token", this->accessToken());
        params.insert("oauth_token_secret", this->tokenSecret());
    }

    return OAuth::createOAuthHeader(method, url, params);
}

void Vimeo::getAuthUrl() {
    QUrl url("https://vimeo.com/oauth/request_token");
    QMap<QString, QString> params;
    params.insert("oauth_callback", REDIRECT_URI);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, params));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAuthUrl()));
}

void Vimeo::checkAuthUrl() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://vimeo.com?" + reply->readAll());
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    QString token = query.queryItemValue("oauth_token");
    QString secret = query.queryItemValue("oauth_token_secret");
#else
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");
#endif
    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining Vimeo authorisation URL"));
    }
    else {
        this->setRequestToken(token);
        this->setRequestSecret(secret);
        QUrl authUrl("https://vimeo.com/oauth/authorize");
#if QT_VERSION >= 0x050000
        QUrlQuery query;
        query.addQueryItem("oauth_token", token);
        query.addQueryItem("permission", "delete");
        authUrl.setQuery(query);
#else
        authUrl.addQueryItem("oauth_token", token);
        authUrl.addQueryItem("permission", "delete");
#endif
        emit gotAuthUrl(authUrl);
    }

    reply->deleteLater();
}

void Vimeo::signIn(const QString &displayName, const QUrl &response) {
#if QT_VERSION >= 0x050000
    QUrlQuery query(response);
    QString verifier = query.queryItemValue("oauth_verifier");
#else
    QString verifier = response.queryItemValue("oauth_verifier");
#endif
    if (verifier.isEmpty()) {
        emit error(tr("Error obtaining Vimeo authorisation verifier"));
    }
    else {
        this->setBusy(true, tr("Signing in"));
        m_user = displayName;
        QUrl url("https://vimeo.com/oauth/access_token");
        QMap<QString, QString> params;
        params.insert("oauth_token", this->requestToken());
        params.insert("oauth_token_secret", this->requestSecret());
        params.insert("oauth_verifier", verifier);
        QNetworkRequest request(url);
        request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
        request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, params));
        QNetworkReply *reply = this->networkAccessManager()->get(request);
        this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
        this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
    }
}

void Vimeo::checkIfSignedIn() {
    this->setBusy(false);
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QUrl url("http://vimeo.com?" + reply->readAll());
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    QString token = query.queryItemValue("oauth_token");
    QString secret = query.queryItemValue("oauth_token_secret");
#else
    QString token = url.queryItemValue("oauth_token");
    QString secret = url.queryItemValue("oauth_token_secret");
#endif
    if ((token.isEmpty()) || (secret.isEmpty())) {
        emit error(tr("Error obtaining Vimeo authorisation"));
    }
    else {
        this->setRequestToken(QString());
        this->setRequestSecret(QString());
        emit signedIn(this->username(), token, secret);
        emit alert(tr("You are signed in to your Vimeo account"));
    }

    reply->deleteLater();
}

void Vimeo::setAccount(const QString &user, const QString &token, const QString &secret) {
    if (user != this->username()) {
        this->setUsername(user);
    }

    this->setAccessToken(token);
    this->setTokenSecret(secret);
    this->clearCache();

    if ((user.isEmpty()) && (!token.isEmpty())) {
        emit newAccountSet();
    }
}

void Vimeo::clearCache() {
    m_albumCache->clear();
    m_groupCache->clear();
    m_subscriptionCache->clear();

    this->setAlbumsLoaded(false);
    this->setGroupsLoaded(false);
    this->setSubscriptionsLoaded(false);
}

void Vimeo::postRequest(const QUrl &url) {
    qDebug() << url;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply* reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Vimeo::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    QDomDocument doc;
    doc.setContent(response);
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        emit postSuccessful(response);
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit postFailed(errorString);
        }
        else {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit postFailed(statusText);
        }
    }

    reply->deleteLater();
}

void Vimeo::getAlbums(int offset) {
    this->setAlbumsLoaded(false);
    QUrl url(VIMEO_ALBUMS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("page", QString::number(offset));
    url.setQuery(query);
#else
    url.addQueryItem("page", QString::number(offset));
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addAlbums()));
}

void Vimeo::addAlbums() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("album");

        for (int i = 0; i < entries.size(); i++) {
            PlaylistItem *album = new PlaylistItem;
            album->loadVimeoAlbum(entries.at(i).toElement());
            m_albumCache->append(QSharedPointer<PlaylistItem>(album));
        }

        QDomElement statNode = responseNode.firstChildElement("albums");

        if ((statNode.attribute("total").toInt() > m_albumCache->size()) && (!entries.isEmpty())) {
            this->getAlbums(statNode.attribute("page").toInt() + 1);
        }
        else {
            this->setAlbumsLoaded(true);
            emit allAlbumsLoaded();
        }
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve albums"));
        }

        this->setAlbumsLoaded(true);
        emit allAlbumsLoaded();
    }

    reply->deleteLater();
}

void Vimeo::addNewAlbumToCache(QSharedPointer<PlaylistItem> album) {
    m_albumCache->insert(0, album);
    emit albumAddedToCache(0);
}

QSharedPointer<PlaylistItem> Vimeo::removeAlbumFromCache(const QString &id) {
    int i = 0;
    bool removed = false;
    QSharedPointer<PlaylistItem> album;

    while ((!removed) && (i < m_albumCache->size())) {
        removed = (m_albumCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        album = m_albumCache->takeAt(i - 1);
        emit albumRemovedFromCache(i - 1);
    }

    return album;
}

void Vimeo::updateAlbumVideoCount(const QString &id, int change) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_albumCache->size())) {
        updated = (m_albumCache->at(i)->id() == id);
        i++;
    }

    if (updated) {
        m_albumCache->at(i - 1).data()->setVideoCount(m_albumCache->at(i - 1).data()->videoCount() + change);
        emit albumUpdated(i - 1);
    }
}

void Vimeo::updateAlbumThumbnail(const QString &id, const QUrl &thumbnailUrl) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_albumCache->size())) {
        updated = (m_albumCache->at(i).data()->id() == id);
        i++;
    }

    if ((updated) && (m_albumCache->at(i - 1).data()->thumbnailUrl().isEmpty())) {
        m_albumCache->at(i - 1).data()->setThumbnailUrl(thumbnailUrl);
        emit albumUpdated(i - 1);
    }
}

void Vimeo::getGroups(int offset) {
    this->setGroupsLoaded(false);
    QUrl url(VIMEO_GROUPS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("user_id", this->username());
    query.addQueryItem("page", QString::number(offset));
    url.setQuery(query);
#else
    url.addQueryItem("user_id", this->username());
    url.addQueryItem("page", QString::number(offset));
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addGroups()));
}

void Vimeo::addGroups() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("group");

        for (int i = 0; i < entries.size(); i++) {
            GroupItem *group = new GroupItem;
            group->loadVimeoGroup(entries.at(i).toElement());
            m_groupCache->append(QSharedPointer<GroupItem>(group));
        }

        QDomElement statNode = responseNode.firstChildElement("groups");

        if ((statNode.attribute("total").toInt() > m_groupCache->size()) && (!entries.isEmpty())) {
            this->getGroups(statNode.attribute("page").toInt() + 1);
        }
        else {
            this->setGroupsLoaded(true);
            emit allGroupsLoaded();
        }
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve groups"));
        }

        this->setGroupsLoaded(true);
        emit allGroupsLoaded();
    }

    reply->deleteLater();
}

void Vimeo::addNewGroupToCache(QSharedPointer<GroupItem> group) {
    m_groupCache->insert(0, group);
    emit groupAddedToCache(0);
    emit groupMembershipChanged(group.data()->id(), true);
}

QSharedPointer<GroupItem> Vimeo::removeGroupFromCache(const QString &id) {
    int i = 0;
    bool removed = false;
    QSharedPointer<GroupItem> group;

    while ((!removed) && (i < m_groupCache->size())) {
        removed = (m_groupCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        group = m_groupCache->takeAt(i - 1);
        emit groupRemovedFromCache(i - 1);
        emit groupMembershipChanged(id, false);
    }

    return group;
}

void Vimeo::getSubscriptions(int offset) {
    this->setSubscriptionsLoaded(false);
    QUrl url(VIMEO_SUBSCRIPTIONS_FEED);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("page", QString::number(offset));
    url.setQuery(query);
#else
    url.addQueryItem("page", QString::number(offset));
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addSubscriptions()));
}

void Vimeo::addSubscriptions() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomNodeList entries = doc.elementsByTagName("contact");

        for (int i = 0; i < entries.size(); i++) {
            UserItem *user = new UserItem;
            user->loadVimeoContact(entries.at(i).toElement(), true);
            m_subscriptionCache->append(QSharedPointer<UserItem>(user));
        }

        QDomElement statNode = responseNode.firstChildElement("contacts");

        if ((statNode.attribute("total").toInt() > m_subscriptionCache->size()) && (!entries.isEmpty())) {
            this->getSubscriptions(statNode.attribute("page").toInt() + 1);
        }
        else {
            this->setSubscriptionsLoaded(true);
            emit allSubscriptionsLoaded();
        }
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve subscriptions"));
        }

        this->setSubscriptionsLoaded(true);
        emit allSubscriptionsLoaded();
    }

    reply->deleteLater();
}

void Vimeo::addNewSubscriptionToCache(QSharedPointer<UserItem> user) {
    m_subscriptionCache->insert(0, user);
    emit subscriptionAddedToCache(0);
    emit subscriptionChanged(user.data()->id(), true);
}

QSharedPointer<UserItem> Vimeo::removeSubscriptionFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    QSharedPointer<UserItem> user;

    while ((!removed) && (i < m_subscriptionCache->size())) {
        removed = (m_subscriptionCache->at(i).data()->id() == id);
        i++;
    }

    if (removed) {
        user = m_subscriptionCache->takeAt(i - 1);
        emit subscriptionRemovedFromCache(i - 1);
        emit subscriptionChanged(id, false);
    }

    return user;
}

void Vimeo::getFullVideo(QString id) {
    this->setBusy(true, tr("Retrieving video details"));
    id = id.section('/', -1);
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.videos.getInfo");
    query.addQueryItem("video_id", id);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.videos.getInfo");
    url.addQueryItem("video_id", id);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void Vimeo::checkFullVideo() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        VideoItem *video = new VideoItem;
        video->loadVimeoVideo(responseNode.firstChildElement("video"));
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(video, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotVideo(video);
#else
        emit gotVideo(QSharedPointer<VideoItem>(video));
#endif
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve video"));
        }
#ifdef QML_USER_INTERFACE
        this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
        this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    }

    reply->deleteLater();
}

void Vimeo::getCurrentUserProfile() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "people.getInfo");
    url.setQuery(query);
#else
    url.addQueryItem("method", "people.getInfo");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUserProfile()));
}

void Vimeo::checkCurrentUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        UserItem *user = new UserItem;
        user->loadVimeoProfile(responseNode.firstChildElement("person"));

        if (this->username() != user->id()) {
            this->setUsername(user->id());
        }

#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotUser(user);
#else
        emit gotUser(QSharedPointer<UserItem>(user));
#endif
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve profile"));
        }
    }

    reply->deleteLater();
}

void Vimeo::getUserProfile(const QString &id) {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "people.getInfo");
    query.addQueryItem("user_id", id);
    url.setQuery(query);
#else
    url.addQueryItem("method", "people.getInfo");
    url.addQueryItem("user_id", id);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void Vimeo::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        UserItem *user = new UserItem;
        user->loadVimeoProfile(responseNode.firstChildElement("person"));
#ifdef QML_USER_INTERFACE
#if QT_VERSION >= 0x050000
            QQmlEngine::setObjectOwnership(user, QQmlEngine::JavaScriptOwnership);
#else
            QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
#endif
        emit gotUser(user);
#else
        emit gotUser(QSharedPointer<UserItem>(user));
#endif
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve profile"));
        }
    }

    reply->deleteLater();
}

void Vimeo::deleteFromUploads(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromUploads();
        this->setBusy(true, tr("Deleting video(s) from uploads"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::deleteFromUploads() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.videos.delete");
        query.addQueryItem("video_id", m_actionIdList.first());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.videos.delete");
        url.addQueryItem("video_id", m_actionIdList.first());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onVideoDeleted() {
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromUploads(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromUploads();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from uploads"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::addToLikes(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->addToLikes();
        this->setBusy(true, tr("Adding video(s) to likes"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::addToLikes() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.videos.setLike");
        query.addQueryItem("video_id", m_actionIdList.first());
        query.addQueryItem("like", "true");
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.videos.setLike");
        url.addQueryItem("video_id", m_actionIdList.first());
        url.addQueryItem("like", "true");
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToLikes(QString)));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onAddedToLikes(const QString &response) {
    Q_UNUSED(response)

    if (!m_actionIdList.isEmpty()) {
        emit likeChanged(m_actionIdList.takeFirst(), true);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToLikes();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to likes"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToLikes(QString)));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::deleteFromLikes(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromLikes();
        this->setBusy(true, tr("Deleting video(s) from likes"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::deleteFromLikes() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.videos.setLike");
        query.addQueryItem("video_id", m_actionIdList.first());
        query.addQueryItem("like", "false");
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.videos.setLike");
        url.addQueryItem("video_id", m_actionIdList.first());
        url.addQueryItem("like", "false");
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromLikes()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onDeletedFromLikes() {    
    if (!m_actionIdList.isEmpty()) {
        emit likeChanged(m_actionIdList.takeFirst(), false);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromLikes();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from likes"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromLikes()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::addToAlbum(const QStringList &videoIds, const QString &albumId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = albumId;
        m_actionsProcessed = 0;
        this->addToAlbum();
        this->setBusy(true, tr("Adding video(s) to album"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::addToAlbum() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.albums.addVideo");
        query.addQueryItem("album_id", m_actionId);
        query.addQueryItem("video_id", m_actionIdList.first());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.albums.addVideo");
        url.addQueryItem("album_id", m_actionId);
        url.addQueryItem("video_id", m_actionIdList.first());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToAlbum(QString)));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        this->disconnect(this, SIGNAL(albumAddedToCache(int)), this, SLOT(addToAlbum()));
    }
}

void Vimeo::onAddedToAlbum(const QString &response) {
    Q_UNUSED(response)

    if (this->albumsLoaded()) {
        this->updateAlbumVideoCount(m_actionId, 1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit addedToAlbum(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToAlbum();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to album"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToAlbum(QString)));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::deleteFromAlbum(const QStringList &videoIds, const QString &albumId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = albumId;
        m_actionsProcessed = 0;
        this->deleteFromAlbum();
        this->setBusy(true, tr("Deleting video(s) from album"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::deleteFromAlbum() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.albums.removeVideo");
        query.addQueryItem("album_id", m_actionId);
        query.addQueryItem("video_id", m_actionIdList.first());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.albums.removeVideo");
        url.addQueryItem("album_id", m_actionId);
        url.addQueryItem("video_id", m_actionIdList.first());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromAlbum()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onDeletedFromAlbum() {
    if (this->albumsLoaded()) {
        this->updateAlbumVideoCount(m_actionId, -1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromAlbum(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromAlbum();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from album"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromAlbum()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::addToWatchLaterAlbum(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->addToWatchLaterAlbum();
        this->setBusy(true, tr("Adding video(s) to album"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::addToWatchLaterAlbum() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.albums.addToWatchLater");
        query.addQueryItem("video_id", m_actionIdList.first());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.albums.addToWatchLater");
        url.addQueryItem("video_id", m_actionIdList.first());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterAlbum(QString)));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onAddedToWatchLaterAlbum(const QString &response) {
    Q_UNUSED(response)

    if (!m_actionIdList.isEmpty()) {
        emit addedToWatchLaterAlbum(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToWatchLaterAlbum();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to 'Watch Later' album"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterAlbum(QString)));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::deleteFromWatchLaterAlbum(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromWatchLaterAlbum();
        this->setBusy(true, tr("Deleting video(s) to album"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::deleteFromWatchLaterAlbum() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.albums.removeFromWatchLater");
        query.addQueryItem("video_id", m_actionIdList.first());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.albums.removeFromWatchLater");
        url.addQueryItem("video_id", m_actionIdList.first());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterAlbum()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void Vimeo::onDeletedFromWatchLaterAlbum() {    
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromWatchLaterAlbum(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromWatchLaterAlbum();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from 'Watch Later' album"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterAlbum()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void Vimeo::createAlbum(const QVariantMap &album, const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_metadataAction = album;
        m_actionIdList = videoIds;
        this->createAlbum();
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void Vimeo::createAlbum() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
        QUrlQuery query(url);
        query.addQueryItem("method", "vimeo.albums.create");
        query.addQueryItem("title", m_metadataAction.value("title").toString());
        query.addQueryItem("description", m_metadataAction.value("description").toString());
        query.addQueryItem("video_id", m_actionIdList.takeFirst());
        url.setQuery(query);
#else
        url.addQueryItem("method", "vimeo.albums.create");
        url.addQueryItem("title", m_metadataAction.value("title").toString());
        url.addQueryItem("description", m_metadataAction.value("description").toString());
        url.addQueryItem("video_id", m_actionIdList.takeFirst());
#endif
        this->postRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAlbumCreated(QString)));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onAlbumActionError(QString)));
    }
}

void Vimeo::onAlbumCreated(const QString &response) {
    Q_UNUSED(response)

    if (!m_actionIdList.isEmpty()) {
        m_actionsProcessed = 0;

        if (this->albumsLoaded()) {
            this->getAlbumForCache();
            this->connect(this, SIGNAL(albumAddedToCache(int)), this, SLOT(addToAlbum()));
        }
        else {
            this->addToAlbum();
        }
    }
    else {
        if (this->albumsLoaded()) {
            this->getAlbumForCache();
        }

        emit alert(tr("Album created"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAlbumCreated(QString)));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onAlbumActionError(QString)));
}

void Vimeo::getAlbumForCache() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.albums.getAll");
    query.addQueryItem("sort", "newest");
    query.addQueryItem("per_page", "1");
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.albums.getAll");
    url.addQueryItem("sort", "newest");
    url.addQueryItem("per_page", "1");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCacheAlbum()));
}

void Vimeo::checkCacheAlbum() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        PlaylistItem *album = new PlaylistItem;
        album->loadVimeoAlbum(responseNode.firstChildElement("albums").firstChildElement("album"));
        this->addNewAlbumToCache(QSharedPointer<PlaylistItem>(album));
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve album"));
        }

        this->disconnect(this, SIGNAL(albumAddedToCache(int)), this, SLOT(addToAlbum()));
    }

    reply->deleteLater();
}

void Vimeo::deleteAlbum(const QString &albumId) {
    m_actionId = albumId;
    this->deleteAlbum();
}

void Vimeo::deleteAlbum() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.albums.delete");
    query.addQueryItem("album_id", m_actionId);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.albums.delete");
    url.addQueryItem("album_id", m_actionId);
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAlbumDeleted()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onAlbumActionError(QString)));
}

void Vimeo::onAlbumDeleted() {
    emit alert(tr("Album deleted"));

    if (this->albumsLoaded()) {
        this->removeAlbumFromCache(m_actionId);
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAlbumDeleted()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onAlbumActionError(QString)));
}

void Vimeo::joinGroup(const QString &groupId) {
    m_actionId = groupId;
    this->joinGroup();
}

void Vimeo::joinGroup() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.groups.join");
    query.addQueryItem("group_id", m_actionId);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.groups.join");
    url.addQueryItem("group_id", m_actionId);
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGroupJoined()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Vimeo::onGroupJoined() {
    if (this->groupsLoaded()) {
        this->getGroupForCache(m_actionId);
    }

    emit alert(tr("You have joined this group"));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGroupJoined()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Vimeo::getGroupForCache(const QString &id) {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.groups.getInfo");
    query.addQueryItem("group_id", id);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.groups.getInfo");
    url.addQueryItem("group_id", id);
#endif
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCacheGroup()));
}

void Vimeo::checkCacheGroup() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        GroupItem *group = new GroupItem;
        group->loadVimeoGroup(responseNode.firstChildElement("groups").firstChildElement("group"));
        this->addNewGroupToCache(QSharedPointer<GroupItem>(group));
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve group"));
        }
    }

    reply->deleteLater();
}

void Vimeo::leaveGroup(const QString &groupId) {
    m_actionId = groupId;
    this->leaveGroup();
}

void Vimeo::leaveGroup() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.groups.leave");
    query.addQueryItem("group_id", m_actionId);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.groups.leave");
    url.addQueryItem("group_id", m_actionId);
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGroupLeft()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Vimeo::onGroupLeft() {
    if ((!this->groupsLoaded()) || (!this->removeGroupFromCache(m_actionId).isNull())) {
        emit alert(tr("You have left this group"));
    }
    else {
        emit error(tr("Unable to leave group"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGroupLeft()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
}

void Vimeo::subscribe(const QString &userId) {
    m_actionId = userId;
    this->subscribe();
}

void Vimeo::subscribe() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.people.addContact");
    query.addQueryItem("user_id", m_actionId);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.people.addContact");
    url.addQueryItem("user_id", m_actionId);
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Vimeo::onSubscribed(const QString &response) {
    Q_UNUSED(response)

    if (this->subscriptionsLoaded()) {
        this->getSubscriptionForCache(m_actionId);
    }
    else {
        emit subscriptionChanged(m_actionId, true);
        emit alert(tr("You have subscribed to this channel"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Vimeo::getSubscriptionForCache(const QString &id) {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.people.getInfo");
    query.addQueryItem("user_id", id);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.people.getInfo");
    url.addQueryItem("user_id", id);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCacheSubscription()));
}

void Vimeo::checkCacheSubscription() {
    QNetworkReply *reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error("Network error");
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        UserItem *user = new UserItem;
        user->loadVimeoProfile(responseNode.firstChildElement("person"));
        this->addNewSubscriptionToCache(QSharedPointer<UserItem>(user));
        emit alert(tr("You have subscribed to '%1'").arg(user->channelName()));
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve subscription"));
        }
    }

    reply->deleteLater();
}

void Vimeo::unsubscribe(const QString &userId) {
    m_actionId = userId;
    this->unsubscribe();
}

void Vimeo::unsubscribe() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.people.removeContact");
    query.addQueryItem("user_id", m_actionId);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.people.removeContact");
    url.addQueryItem("user_id", m_actionId);
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Vimeo::onUnsubscribed() {
    if (this->subscriptionsLoaded()) {
        QSharedPointer<UserItem> user = this->removeSubscriptionFromCache(m_actionId);

        if (!user.isNull()) {
            emit alert(tr("You have unsubscribed to '%1'").arg(user.data()->channelName()));
        }
        else {
            emit subscriptionChanged(m_actionId, false);
            emit alert(tr("You have unsubscribed to this channel"));
        }
    }
    else {
        emit subscriptionChanged(m_actionId, false);
        emit alert(tr("You have unsubscribed to this channel"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void Vimeo::addComment(const QVariantMap &comment) {
    m_metadataAction = comment;
    this->addComment();
}

void Vimeo::addComment() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.videos.comments.addComment");
    query.addQueryItem("video_id", m_metadataAction.value("videoId").toString());
    query.addQueryItem("comment_text", m_metadataAction.value("body").toString());
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.videos.comments.addComment");
    url.addQueryItem("video_id", m_metadataAction.value("videoId").toString());
    url.addQueryItem("comment_text", m_metadataAction.value("body").toString());
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Vimeo::replyToComment(const QVariantMap &comment) {
    m_metadataAction = comment;
    this->replyToComment();
}

void Vimeo::replyToComment() {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.videos.comments.addComment");
    query.addQueryItem("video_id", m_metadataAction.value("videoId").toString());
    query.addQueryItem("reply_to_comment_id", m_metadataAction.value("replyId").toString());
    query.addQueryItem("comment_text", m_metadataAction.value("body").toString());
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.videos.comments.addComment");
    url.addQueryItem("video_id", m_metadataAction.value("videoId").toString());
    url.addQueryItem("reply_to_comment_id", m_metadataAction.value("replyId").toString());
    url.addQueryItem("comment_text", m_metadataAction.value("body").toString());
#endif
    this->postRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Vimeo::onCommentAdded(const QString &response) {
    Q_UNUSED(response)

    this->getAddedComment(m_metadataAction.value("videoId").toString());
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void Vimeo::getAddedComment(const QString &videoId) {
    m_actionId = videoId;
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.video.comments.getList");
    query.addQueryItem("video_id", m_actionId);
    query.addQueryItem("per_page", "1");
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.video.comments.getList");
    url.addQueryItem("video_id", m_actionId);
    url.addQueryItem("per_page", "1");
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAddedComment()));
}

void Vimeo::checkAddedComment() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        CommentItem *comment = new CommentItem;
        comment->loadVimeoComment(responseNode.firstChildElement("comments").firstChildElement("comment"));
        comment->setVideoId(m_actionId);
        emit commentAdded(QSharedPointer<CommentItem>(comment));
    }
    else {
        QString errorString = responseNode.firstChildElement("err").attribute("expl");

        if (!errorString.isEmpty()) {
            emit error(errorString);
        }
        else {
            emit error(tr("Unable to retrieve comment"));
        }
    }

    reply->deleteLater();
}

void Vimeo::onVideoActionError(const QString &errorString) {
    m_actionIdList.clear();
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Vimeo::onAlbumActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onAlbumActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Vimeo::onGroupActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onGroupActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Vimeo::onUserActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Vimeo::onCommentActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void Vimeo::getVideosFromIds(QStringList ids) {
    m_actionIdList = ids;

    if (!m_actionIdList.isEmpty()) {
        this->setBusy(true, QString(), 0);
        this->getVideoFromId(m_actionIdList.takeFirst());
    }
}

void Vimeo::getVideoFromId(const QString &id) {
    QUrl url(VIMEO_BASE_URL);
#if QT_VERSION >= 0x050000
    QUrlQuery query(url);
    query.addQueryItem("method", "vimeo.videos.getInfo");
    query.addQueryItem("video_id", id);
    url.setQuery(query);
#else
    url.addQueryItem("method", "vimeo.videos.getInfo");
    url.addQueryItem("video_id", id);
#endif
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(VERSION_NUMBER).toUtf8());
    request.setRawHeader("Authorization", this->getOAuthHeader("GET", url, QMap<QString, QString>()));
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVideoFromId()));
}

void Vimeo::checkVideoFromId() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomElement responseNode = doc.firstChildElement("rsp");

    if (responseNode.attribute("stat") == "ok") {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        VideoItem *video = new VideoItem;
        video->loadVimeoVideo(responseNode.firstChildElement("video"));
        emit gotVideosFromIds(QList< QSharedPointer<VideoItem> >() << QSharedPointer<VideoItem>(video));
    }
    if (!m_actionIdList.isEmpty()) {
        this->getVideoFromId(m_actionIdList.takeFirst());
    }
    else {
        this->setBusy(false);
    }

    reply->deleteLater();
}
