#include <QDebug>

#include "gcparser.h"

GcParser::GcParser() {
}

GcParser::~GcParser() {
}

QString GcParser::error() const {
  return m_error;
}

bool GcParser::parseFloat(const QMap<QString, QVariant> &map, const QString &key, qreal &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toFloat();
  return true;
}

bool GcParser::parseInt(const QMap<QString, QVariant> &map, const QString &key, int &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toInt();
  return true;
}

// additional waypoints:
// 2000: Parking, 2001: Trailhead, Stage of a Multicache, 
// 2003: Question to answer, 2004: Reference Point, Final Location

bool GcParser::parseCtid(const QMap<QString, QVariant> &map, 
			 const QString &key, Cache::Type &val) {
  int i, ctid;

  const struct { Cache::Type type; int ctid; } tags[] = {
    { Cache::TypeTraditional, 2 }, { Cache::TypeMulti,        3 },
    { Cache::TypeMystery,     8 }, { Cache::TypeVirtual,      4 },
    { Cache::TypeWebcam,     11 }, { Cache::TypeEvent,        6 },
    { Cache::TypeLetterbox,   5 }, { Cache::TypeEarthcache, 137 }, 
    { Cache::TypeWherigo,  1858 }, { Cache::TypeMegaEvent,  453 },
    { Cache::TypeCito,       13 }, { Cache::TypeUnknown,     -1 }
  };

  if(!parseInt(map, key, ctid)) {
    val = Cache::TypeUnknown;
    return false;
  }

  for(i=0;(tags[i].type != Cache::TypeUnknown) && (tags[i].ctid != ctid);i++);
  val = tags[i].type;

  return true;
}

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcParser::subString(QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

bool GcParser::parseRating(const QMap<QString, QVariant> &map, 
			   const QString &key, Rating &rating) {
  QString ratingStr;

  if(!parseString(map, key, ratingStr)) {
    rating.set(0);
    return false;
  }

  rating.set(subString(ratingStr, "alt=\"", "\"").split(" ")[0].toFloat());

  return true;
}

bool GcParser::parseContainer(const QMap<QString, QVariant> &map, 
			      const QString &key, Container &container) {
  QString containerStr;
  int i;

  const struct { Container::Type type; QString str; } tags[] = {
    { Container::ContainerRegular, "Regular" }, 
    { Container::ContainerSmall,     "Small" },
    { Container::ContainerMicro,     "Micro" },
    { Container::ContainerOther,     "Other" },
    { Container::ContainerNotChosen,   "Not" },  // in fact "Not Chosen"
    { Container::ContainerLarge,     "Large" },
    { Container::ContainerVirtual, "Virtual" },
    { Container::ContainerUnknown,       "?" }
  };

  if(!parseString(map, key, containerStr)) {
    container.set(Container::ContainerUnknown);
    return false;
  }

  containerStr = subString(containerStr, "alt=\"", "\"").split(" ")[1];
  container.set(Container::ContainerRegular);

  for(i=0;(tags[i].type != Container::ContainerUnknown) && 
	(tags[i].str != containerStr);i++);

  container.set(tags[i].type);

  return container.isSet();
}

bool GcParser::parseDate(const QMap<QString, QVariant> &map, 
			 const QString &key, QDate &date) {
  QString dateStr;

  if(!parseString(map, key, dateStr)) {
    date = QDate();
    return false;
  }

  QStringList dateParts = dateStr.split("/");

  // we expect month/day/year
  if(dateParts.size() != 3) return false;
  
  // qdate expects y,m,d while gc delivers m/d/y
  date = QDate(dateParts[2].toInt(), dateParts[0].toInt(), dateParts[1].toInt());

  return true;
}

bool GcParser::parseBool(const QMap<QString, QVariant> &map, const QString &key, bool &val) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::Bool != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  val = result.toBool();
  return true;
}

bool GcParser::parseString(const QMap<QString, QVariant> &map, const QString &key, QString &str) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  // the value of this key should be a map
  QVariant result = it.value();
  if(QVariant::String != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  str = result.toString();

  return true;
}

bool GcParser::parseList(const QMap<QString, QVariant> &map, const QString &key, QList<QVariant> &list) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  QVariant result = it.value();
  if(QVariant::List != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  list = result.toList();
  return true;
}

bool GcParser::parseMap(const QMap<QString, QVariant> &map, const QString &key, QMap<QString, QVariant> &nmap) {

  QMap<QString, QVariant>::const_iterator it = map.find(key);
  if(it == map.end()) {
    qDebug() << __FUNCTION__ << "Key \"" << key << "\" not found.";
    return false;
  } 

  QVariant result = it.value();
  if(QVariant::Map != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected value type:" << result.type();
    return false;
  }

  nmap = result.toMap();
  return true;
}

bool GcParser::parseOverviewEntry(const QMap<QString, QVariant> &map, Cache &cache) {
  
  //  qDebug() << __FUNCTION__ << map;

  // set coordinates
  qreal lat, lon;
  if(parseFloat(map, "lat", lat) && parseFloat(map, "lon", lon)) {
    QGeoCoordinate coo(lat, lon);
    cache.setCoordinate(coo);
  }

  // set name and description
  QString nn, gc;
  if(parseString(map, "nn", nn))   cache.setDescription(nn);
  if(parseString(map, "gc", gc))   cache.setName(gc);

  bool f, o, ia;
  if(parseBool(map, "f", f))       cache.setFound(f);
  if(parseBool(map, "o", o))       cache.setOwned(o);
  if(parseBool(map, "ia", ia))     cache.setAvailable(ia);
 
  int id;
  if(parseInt(map, "id", id))      cache.setId(id);

  Cache::Type type;
  if(parseCtid(map, "ctid", type)) cache.setType(type);

  qDebug() << __FUNCTION__ << cache;
  
  return true;
}

bool GcParser::parseOverviewCC(const QList<QVariant> &list, 
			       CacheList &cacheList) {
  bool retval = true;

  QList<QVariant>::const_iterator it = list.begin();
  while(it != list.end()) {

    // the value of this entry should be a map
    if(QVariant::Map != (*it).type()) 
      qDebug() << __FUNCTION__ << "Ignoring unexpected value type:" << (*it).type();
    else {
      Cache cache;
      QMap<QString, QVariant> map = (*it).toMap();
      if(parseOverviewEntry(map, cache))
	cacheList.append(cache);
      else
	retval = false;
    }
    it++;
  }
   
  return retval;
}

bool GcParser::parseOverviewRoot(const QMap<QString, QVariant> &map, 
				 CacheList &cacheList, bool bigArea) {
  // the root overview has 6 fields:
  // c:      int, unknown, always 1
  // cc:     list of caches
  // count:  int, number of entries
  // dist:   float, ??
  // li:     bool, ?? logged in? 
  // pm:     bool, ??

  qDebug() << __FUNCTION__;

  int count = 0;
  if(!parseInt(map, "count", count))
    return false;

  if(count == 0) {
    // if the area is too big, the result will look exactly
    // like one for an area without any cache
    if(bigArea) m_error = tr("Too many caches");
    else        m_error = tr("No caches in this area");

    return false;
  }

  // the list will be empty if more than 500 caches are in this area
  if(count > 500) {
    m_error = tr("Too many caches");
    return false;
  }

  QList<QVariant> list;
  if(!parseList(map, "cc", list))
    return false;

  return parseOverviewCC(list, cacheList);
}

bool GcParser::decodeJson(const QString &data, QMap<QString, QVariant>&map) {
  bool ok;

  // json is a QString containing the data to convert
  QVariant result = Json::parse(data, ok);
  if(!ok) {
    qDebug() << __FUNCTION__ << "Json deconding failed.";
    return false;
  }

  // we expect a qvariantmap
  if(QVariant::Map != result.type()) {
    qDebug() << __FUNCTION__ << "Unexpected result type:" << result.type();
    return false;
  } 

  map = result.toMap();
  return true;
}

// stage two is a complex json encoded map stored under key "cs"
bool GcParser::decodeStage2(const QString &data, QMap<QString, QVariant> &map) {
  QMap<QString, QVariant> tmpMap;

  if(!decodeJson(data, tmpMap))
    return false;

  if(!parseMap(tmpMap, "cs", map))
    return false;

  return true;
}

// stage one is just a single json encoded string stored under key "d"
bool GcParser::decodeStage1(const QString &data, QString &string) {
  QMap<QString, QVariant> map;

  if(!decodeJson(data, map))
    return false;

  if(!parseString(map, "d", string))
    return false;

  return true;
}

bool GcParser::decodeOverview(const QString &data, 
			      CacheList &cacheList, bool bigArea) {

  cacheList.clear();

  // assume for now that parsing failed for unknown reason
  m_error = tr("Error downloading caches");

  QString string;
  if(!decodeStage1(data, string))
    return false;

  QMap<QString, QVariant> map; 
  if(!decodeStage2(string, map))
    return false;

  if(parseOverviewRoot(map, cacheList, bigArea))
    m_error = QString();

  return(m_error.isNull());
}



bool GcParser::parseInfoEntry(const QMap<QString, QVariant> &map, Cache &cache) {

  //  qDebug() << __FUNCTION__ << map;

  // the info entry has 19 fields:
  // bm:     int, the id of the requested cache
  // c:      int, unknown, always 2
  // cb:     string, owner
  // cburl:  string, owner url
  // lg:     int, the id of the requested cache
  // li:     bool, unknown (false)
  // pm:     bool, unknown (false)
  // tc:     int, unknown (1)

  QString cgc, cn;
  if(parseString(map, "cgc", cgc)) cache.setName(cgc);
  if(parseString(map, "cn", cn))   cache.setDescription(cn);

  QString cb;
  if(parseString(map, "cb", cb))   cache.setOwner(cb);

  QDate dh;
  if(parseDate(map, "dh", dh))   cache.setDateOfPlacement(dh);

  Rating difficulty, terrain;
  if(parseRating(map, "d", difficulty))  cache.setDifficulty(difficulty.value());
  if(parseRating(map, "t", terrain))     cache.setTerrain(terrain.value());

  Container container;
  if(parseContainer(map, "cz", container))  cache.setContainer(container.get());

  bool ia;
  if(parseBool(map, "ia", ia)) cache.setAvailable(ia);
 
  int cid;
  if(parseInt(map, "cid", cid)) cache.setId(cid);

  Cache::Type type;
  if(parseCtid(map, "ci", type)) cache.setType(type);

  QString curl, cg;
  if(parseString(map, "curl", curl)) cache.setUrl(curl);
  if(parseString(map, "cg", cg))     cache.setGuid(cg);

  qDebug() << __FUNCTION__ << cache;

  return true;
}

bool GcParser::decodeInfo(const QString &data, Cache &cache) {

  // assume for now that parsing failed for unknown reason
  m_error = tr("Error downloading cache info");

  QString string;
  if(!decodeStage1(data, string))
    return false;

  QMap<QString, QVariant> map; 
  if(!decodeStage2(string, map))
    return false;

  if(!parseInfoEntry(map, cache))
    return false;

  m_error = QString();
  return true;
}
