/* This file is part of Beifahrer.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * Beifahrer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beifahrer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beifahrer. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class LiftDetailWindow : StackableWindow {
	private const string BROWSER_SERVICE = "com.nokia.osso_browser";
	private const string BROWSER_PATH = "/com/nokia/osso_browser";
	private const string BROWSER_IF = "com.nokia.osso_browser";

	private const string CALENDAR_SERVICE = "com.nokia.calendar";
	private const string CALENDAR_PATH = "/com/nokia/calendar";
	private const string CALENDAR_IF = "com.nokia.calendar";

	AdacMitfahrclub adac;
	Lift lift;
	OssoABook.Button button_route;
	OssoABook.Button button_calendar;
	Label label_driver;
	OssoABook.Button button_phone;
	OssoABook.Button button_sms;
	Image image_smoke;
	Label label;

	public LiftDetailWindow (AdacMitfahrclub _adac, Lift _lift) {
		adac = _adac;
		lift = _lift;

		update_lift_details ();

		if (lift.description == null)
			get_lift_details.begin ();
	}

	construct {
		set_title ("Beifahrer");

		var menu = new AppMenu ();
		var add_contact = new Gtk.Button.with_label (_("New contact"));
		add_contact.show ();
		menu.append (add_contact);
		var goto_website = new Gtk.Button.with_label (_("Show website"));
		goto_website.show ();
		menu.append (goto_website);
		set_main_menu (menu);

		var pannable = new PannableArea ();
		pannable.hscrollbar_policy = PolicyType.NEVER;

		var vbox = new VBox (false, 0);

		button_route = new OssoABook.Button (SizeType.FINGER_HEIGHT);
		button_route.style = OssoABook.ButtonStyle.LABEL;

		var table = new Table (5, 2, false);

		button_calendar = new OssoABook.Button (SizeType.FINGER_HEIGHT);
		button_calendar.set_icon_name ("general_calendar");
		button_calendar.set_icon_visible (true);
		button_calendar.style = OssoABook.ButtonStyle.LABEL;
		button_calendar.title = _("Departure time");

		label_driver = new Label (_("Driver:"));
		label_driver.set_alignment (0.0f, 0.5f);

		button_phone = new OssoABook.Button (SizeType.FINGER_HEIGHT);
		button_phone.set_icon_name ("general_call");
		button_phone.set_icon_visible (true);
		button_phone.title = _("Phone");

		button_sms = new OssoABook.Button (SizeType.FINGER_HEIGHT);
		button_sms.set_icon_name ("general_sms");
		button_sms.set_icon_visible (true);
		button_sms.title = _("SMS");

		image_smoke = new Image ();

		label = new Label ("");
		label.set_alignment (0.0f, 0.5f);
		label.set_line_wrap (true);

		table.attach (button_calendar, 1, 2, 0, 1, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, 0, 0);
		table.attach (label_driver, 0, 1, 1, 2, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, MARGIN_DEFAULT, 0);
		table.attach (button_phone, 1, 2, 1, 2, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, 0, 0);
		table.attach (button_sms, 1, 2, 2, 3, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, 0, 0);
		table.attach (image_smoke, 1, 2, 3, 4, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, 0, 0);
		table.attach (label, 0, 2, 4, 5, AttachOptions.FILL | AttachOptions.EXPAND, AttachOptions.FILL, MARGIN_DEFAULT, 0);

		vbox.pack_start (button_route, true, true, 0);
		vbox.pack_start (table, true, true, 0);

		pannable.add_with_viewport (vbox);

		add (pannable);

		add_contact.clicked.connect (on_add_contact_clicked);
		goto_website.clicked.connect (on_goto_website_clicked);
		button_phone.clicked.connect (on_button_phone_clicked);
		button_sms.clicked.connect (on_button_sms_clicked);
	}

	private async void get_lift_details () {
		Hildon.gtk_window_set_progress_indicator (this, 1);
		yield adac.update_lift_details (lift);
		Hildon.gtk_window_set_progress_indicator (this, 0);

		update_lift_details ();
	}

	private void update_lift_details () {
		button_route.title = _("From %s to %s").printf (lift.city_from, lift.city_to);
		if (lift.city_via.length () > 0) {
			string route = _("via ");
			foreach (string via in lift.city_via)
				route += via + " ";
			button_route.value = route;
		} else {
			button_route.value = "";
		}

		button_calendar.value = "%s, %s".printf (lift.date, lift.time);

		label_driver.set_text (_("Driver: ") + lift.name);
		if (lift.cell != null) {
			button_phone.title = _("Cell");
			string cell_number = E.normalize_phone_number (lift.cell);
			button_phone.value = cell_number;
			button_sms.value = cell_number;
		} else if (lift.phone != null) {
			button_phone.title = _("Phone");
			string phone_number = E.normalize_phone_number (lift.phone);
			button_phone.value = phone_number;
			button_sms.value = phone_number;
		}

		string lift_text = _("Free places: %d\n").printf (lift.places);
		lift_text += (lift.price != null) ? _("Price: %s\n").printf (lift.price) : _("(no price given)\n");
		if (LiftFlags.SMOKER in lift.flags) try {
			var pixbuf = IconTheme.get_default ().load_icon ("beifahrer_smoker", 32, IconLookupFlags.NO_SVG);
			image_smoke.pixbuf = pixbuf;
			lift_text += _("smoker ");
		} catch (Error e) {
		} else if (LiftFlags.NON_SMOKER in lift.flags) try {
			var pixbuf = IconTheme.get_default ().load_icon ("beifahrer_non_smoker", 32, IconLookupFlags.NO_SVG);
			image_smoke.pixbuf = pixbuf;
			lift_text += _("non-smoker ");
		} catch (Error e) {
		}
		if (LiftFlags.ADAC_MEMBER in lift.flags)
			lift_text += _("ADAC member ");
		if (LiftFlags.WOMEN_ONLY in lift.flags)
			lift_text += _("only women ");
		lift_text += "\n\n" + lift.description;

		if (lift.modified != null)
			lift_text += _("\nLast changed: ") + lift.modified;

		label.set_text (lift_text);
	}

	void on_button_phone_clicked () {
		var uri = "tel://";
		if (lift.cell != null)
			uri += E.normalize_phone_number (lift.cell);
		else if (lift.phone != null)
			uri += E.normalize_phone_number (lift.phone);
		try {
			var action = URIAction.get_default_action_by_uri (uri);
			if (action != null) {
				action.open (uri);
			} else {
				
				Banner.show_information (this, null, _("Couldn't call %s URI").printf (uri));
			}
		} catch (Error e) {
			if (e is URIError) {
				Banner.show_information (this, null, _("Error: %s").printf (e.message));
			}
		}
	}

	void on_button_sms_clicked () {
		var uri = "sms://";
		if (lift.cell != null)
			uri += E.normalize_phone_number (lift.cell);
		else if (lift.phone != null)
			uri += E.normalize_phone_number (lift.phone);
		try {
			var action = URIAction.get_default_action_by_uri (uri);
			if (action != null) {
				action.open (uri);
			} else {
				
				Banner.show_information (this, null, _("Couldn't open %s URI").printf (uri));
			}
		} catch (Error e) {
			if (e is URIError) {
				Banner.show_information (this, null, _("Error: %s").printf (e.message));
			}
		}
	}

	void on_add_contact_clicked () {
		var contact = new OssoABook.Contact ();
		var attr_fn = new E.VCardAttribute (null, E.EVC_FN);
		contact.add_attribute_with_value ((owned) attr_fn, lift.name);
		if (lift.cell != null) {
			var attr_cell = new E.VCardAttribute (null, E.EVC_TEL);
			var param = new E.VCardAttributeParam (E.EVC_TYPE);
			attr_cell.add_param_with_value ((owned) param, "CELL");
			contact.add_attribute_with_value ((owned) attr_cell, E.normalize_phone_number (lift.cell));
		}
		if (lift.phone != null) {
			var attr_phone = new E.VCardAttribute (null, E.EVC_TEL);
			var param = new E.VCardAttributeParam (E.EVC_TYPE);
			attr_phone.add_param_with_value ((owned) param, "VOICE");
			contact.add_attribute_with_value ((owned) attr_phone, E.normalize_phone_number (lift.phone));
		}
		var dialog = new OssoABook.TemporaryContactDialog.with_contact (this, contact);
		dialog.show ();
	}

	void on_goto_website_clicked () {
		var url = "http://mitfahrclub.adac.de" + lift.href;

		try {
			var conn = DBus.Bus.get (DBus.BusType.SESSION);

			dynamic DBus.Object browser = conn.get_object (BROWSER_SERVICE, BROWSER_PATH, BROWSER_IF);
			browser.open_new_window (url, false);
		} catch (Error e) {
			stderr.printf ("Error: %s\n", e.message);
			Banner.show_information (this, null, _("Failed to open browser."));
		}
	}
}
