#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm


import re
import subprocess

import time

class DataSource(object):
    def __init__(self):
        self.providedValues = self.getProvidedValues()
        self.lastValues = {}
        for key in self.providedValues:
            self.markUpdated(key, None, 0)

    def markUpdated(self, key, value, timestamp):
        self.lastValues[key] = (value, timestamp)

    def timestampAdjust(self, type):
        return 0

class DataSourceInternal(DataSource):
    def getProvidedValues(self):
        return {'internal.graph.break': 'none'}

    def read(self, type):
        """
        Type can be: 'periodical' or 'startup' 
        """
        if type == 'startup':
            return {'internal.graph.break': None}
        return {}

    def decideStoreableValues(self, available, changed):
        if available.has_key('internal.graph.break'):
            return {'internal.graph.break': None}
        return {}

    def timestampAdjust(self, type):
        if type == 'internal.graph.break':
            return -1
        return 0

class DataSourceHal(DataSource):
    def getProvidedValues(self):
        return {'hal.battery.charge_level.percentage': '%',
                'hal.battery.reporting.current': 'mAh',
                'hal.battery.reporting.design': 'mAh',
                'hal.battery.voltage.current': 'mV',
                'hal.battery.voltage.design': 'mV',
                'hal.battery.rechargeable.is_charging': 'bool',
                'hal.battery.rechargeable.is_discharging': 'bool'}

    def read(self, type):
        """
        Type can be: 'periodical' or 'startup' 
        """
        process = subprocess.Popen(['lshal', '-u',
                                    '/org/freedesktop/Hal/devices/bme'],
                                   stdout=subprocess.PIPE)

        output = process.stdout.readlines()
        process.wait()
        process.stdout.close()

        output = self.dictionarize(output)

        infos = ['battery.charge_level.percentage',
                 'battery.reporting.current',
                 'battery.voltage.current',
                 'battery.rechargeable.is_charging',
                 'battery.rechargeable.is_discharging']
        if type == 'periodical':
            pass
        elif type == 'startup':
            infos.append('battery.reporting.design')
            infos.append('battery.voltage.design')
        else:
            raise 'Unknown type'

        ret = {}

        for info in infos:
            if output.has_key(info):
                infoType = self.providedValues['hal.' + info]
                if infoType in ('%', 'mAh', 'mV'):
                    processedValue = int(output[info])
                elif infoType == 'bool':
                    if output[info] == 'true':
                        processedValue = True
                    else:
                        processedValue = False
                else:
                    processedValue = output[info]
                ret['hal.' + info] = processedValue
                    
        return ret

    # Decides which values should be stored.
    # May include more data than what is actually
    # reported as changed, if values are old.
    def decideStoreableValues(self, available, changed):
        changedAndAvailable = []
        for k in changed:
            if available.has_key(k):
                changedAndAvailable.append(k)

        ret = {}
        for k in changedAndAvailable:
            ret[k] = available[k]

        now = int(time.time())

        additional = []
        for k in available.keys():
            if k in ('hal.battery.charge_level.percentage',
                     'hal.battery.reporting.current',
                     'hal.battery.voltage.current'):
                if self.lastValues[k][1] < now - 900 \
			and self.lastValues[k][0] != available[k]:
                    additional.append(k)
            if k in ('hal.battery.rechargeable.is_charging',
                     'hal.battery.rechargeable.is_discharging'):
                 if self.lastValues[k][0] != available[k]:
                    additional.append(k)

        for k in additional:
            ret[k] = available[k]

        chrg = 'hal.battery.rechargeable.is_charging'
        dchrg = 'hal.battery.rechargeable.is_discharging'
        if available.has_key(chrg) and available.has_key(dchrg) \
                and available[chrg] and not available[dchrg]:
            # Don't return percentage and energy, they're not being
            # updated during charge
            if (self.lastValues[dchrg][0] == True):
                # Except, if we just started charging. Then we want to record
                # the last known values.
                # (This should fix charge time start inaccuracy)
                for key in ('hal.battery.charge_level.percentage',
                            'hal.battery.reporting.current'):
                    if not ret.has_key(key) and available.has_key(key):
                        ret[key] = available[key]
            else:
                # Don't record
                ret.pop('hal.battery.charge_level.percentage', None)
                ret.pop('hal.battery.reporting.current', None)

        return ret 

    def dictionarize(self, lines):
        ret = {}
        regex = re.compile('(\S+) = (\S+)')
        for line in lines:
            match = regex.search(line)
            if match:
                ret[match.group(1)] = match.group(2)
        return ret

