/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "audiolevelgraphicsscene.h"

#include <QGraphicsLineItem>
#include <QGraphicsView>
#include <QDebug>


/*! 
  The constructor initializes the scaling factors, sets up the threshold line
  and sets up the associated QGraphicsView object. Furthermore, it stores the
  reference to the application settings.
*/
AudioLevelGraphicsScene::AudioLevelGraphicsScene(QGraphicsView *graphicsView, Settings *settings, QObject *parent)
    : QGraphicsScene(parent), itsSettings(settings)
{
    // initialize view
    graphicsView->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    graphicsView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    graphicsView->setScene(this);

    // setup geometry
    //itsWidth = graphicsView->geometry().width();
    itsWidth = itsSettings->GRAPH_WIDTH;
    //int height = graphicsView->geometry().height();
    int height = itsSettings->GRAPH_HEIGHT;
    itsScale = (float)height / (itsSettings->GRAPH_MAX_VALUE - itsSettings->GRAPH_MIN_VALUE);

    // set scene size
    setSceneRect(0, 0, itsWidth, -height);
    graphicsView->setSceneRect(0, 0, itsWidth, -height);

    // initialize display
    Clear();
}


/*!
  AddValue takes the given audio data point and adds it to the graph.
  If the graph reaches the end of the screen on its x-axes, the graph is cleared
  to start over at the left again.
*/
void AudioLevelGraphicsScene::AddValue(float value)
{
    // add current item
	// values below the threshold are painted black, values above in red
    addLine(itsCurrentPosition, 0, itsCurrentPosition, GetYPos(value),
            value >= itsSettings->THRESHOLD_VALUE ? QPen(Qt::red) : QPen(Qt::black));

    // increment data pointer
    if (++itsCurrentPosition >= itsWidth) {
        // reached end of screen, clear display
        Clear();
    }
}


/*!
  Clear removes all objects from the graph and redraws the limit line.
*/
void AudioLevelGraphicsScene::Clear()
{
    // remove all items
    foreach(QGraphicsItem *item, items()) {
        removeItem(item);
        delete item;
    }

    // add threshold line again
    addLine(0, GetYPos(itsSettings->THRESHOLD_VALUE), itsWidth,
            GetYPos(itsSettings->THRESHOLD_VALUE), QPen(Qt::gray));

    // start again at the left
    itsCurrentPosition = 0;
}


/*!
  GetYPos derives the scaled y-value in pixels from the input value
*/
float AudioLevelGraphicsScene::GetYPos(float value)
{
    return -(value-itsSettings->GRAPH_MIN_VALUE)*itsScale;
}
