/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "usernotifier.h"

#include <QDebug>


UserNotifier::UserNotifier(Settings *settings, QObject *parent) :
    QObject(parent), itsSettings(settings)
{
    // register to receive incoming calls
    bool result = QDBusConnection::systemBus().connect("com.nokia.csd.Call", "/com/nokia/csd/call", "com.nokia.csd.Call", "Coming", this, SLOT(receiveCall(const QDBusMessage&)));
    if (result == false)
        qWarning() << tr("Cannot connect to incoming calls: ") << QDBusConnection::systemBus().lastError();

    // register to receive call establishment notification
    result = QDBusConnection::systemBus().connect("com.nokia.csd.Call", "/com/nokia/csd/call/1", "com.nokia.csd.Call.Instance", "AudioConnect", this, SLOT(callEstablished(const QDBusMessage&)));
    if (result == false)
        qCritical() << tr("Cannot connect to call establishment notifications: ") << QDBusConnection::systemBus().lastError();

    // setup call timer
    itsCallTimer = new QTimer(this);
    itsCallTimer->setSingleShot(true);
    connect(itsCallTimer, SIGNAL(timeout()), this, SLOT(callSetupTimer()));
}


bool UserNotifier::Notify()
{
    // get DBUS system bus
    if (!QDBusConnection::systemBus().isConnected()) {
        qCritical() << tr("Cannot connect to DBUS system bus.");
        return false;
    }

    // initiate call
    QDBusMessage msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call", // --dest
            "/com/nokia/csd/call", // destination object path
            "com.nokia.csd.Call", // message name (w/o method)
            "CreateWith" // method
        );
    msg << itsSettings->itsPhonenumber;
    msg << 0;
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);
    if (reply.type() == QDBusMessage::ErrorMessage) {
        qCritical() << tr("Call initiation failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    qDebug() << tr("Call initiated with result ") << reply;

    // start timer to abort call if not answered
    itsCallTimer->start(itsSettings->CALL_SETUP_TIMER);

    return true;
}


void UserNotifier::callEstablished(const QDBusMessage &msg)
{
    bool flag0 = msg.arguments()[0].toBool();
    bool flag1 = msg.arguments()[1].toBool();

    // is this the start or end of the call?
    if (flag0 && flag1) {
        // start of call
        // restart the call timer with a longer timeout, once we got an active call
        qDebug() << tr("Call established, extending call timeout.");
        itsCallTimer->start(itsSettings->CALL_HOLD_TIMER);
    }
    else if (!flag0 && !flag1) {
        // end of call
        // clear potential running timer
        itsCallTimer->stop();

        // signal the end of the notification process
        qDebug() << tr("Call terminated, signal end of notification.");
        emit notifyFinished();
    }
}


void UserNotifier::callSetupTimer()
{
    // terminate call after this timeout
    qDebug() << tr("Call setup timeout triggered. Releasing call.");
    handleCall(false);

    // signal the end of the notification process
    emit notifyFinished();
}


void UserNotifier::receiveCall(const QDBusMessage &msg)
{
    QList<QVariant> lst = msg.arguments();
    QString caller = lst[1].toString();
    qDebug() << tr("Receive call from ") << caller;

    if (itsSettings->itsRejectIncomingCalls) {
        // handle incoming calls
        if (caller == itsSettings->itsPhonenumber)
            handleCall(true);
        else
            handleCall(false);
    }
    else {
        // do not handle incomin calls
    }
}


bool UserNotifier::handleCall(const bool takeCall) const
{
    QDBusMessage msg;

    // this should be "if (takeCall)" but answering a call is not possible yet:
    // QDBusError("com.nokia.csd.Call.Error.NotAllowed", "Not Allowed Error")
    if (false)
        msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call/1",    // destination object path
            "com.nokia.csd.Call.Instance",  // message name (w/o method)
            "Answer"                    // method
        );
    else
        msg = QDBusMessage::createMethodCall(
            "com.nokia.csd.Call",       // --dest
            "/com/nokia/csd/call",      // destination object path
            "com.nokia.csd.Call",       // message name (w/o method)
            "Release"                   // method
        );
    QDBusMessage reply = QDBusConnection::systemBus().call(msg);

    if (reply.type() == QDBusMessage::ErrorMessage) {
        qWarning() << tr("Call handling failed: ") << QDBusConnection::systemBus().lastError();
        return false;
    }
    qDebug() << tr("Call") << (takeCall ? tr("taken") : tr("dropped")) << tr("with result") << reply;

    return true;
}


/*
void UserNotifier::MessagePopup()
{
    //dbus-send --type=method_call --dest=org.freedesktop.Notifications /org/freedesktop/Notifications org.freedesktop.Notifications.SystemNoteInfoprint string:"NOTIFICATION"
}
*/


/*
bool UserNotifier::NotifySMS()
{
    // Qt mobility messaging interface
    QMap<QString, QMessageAccountId> accountDetails;
    QMessageService theService;
    QMessageManager manager;
    foreach(const QMessageAccountId &id, manager.queryAccounts()){
        QMessageAccount account(id);
        accountDetails.insert(account.name(), account.id());
    }
    QMessage theMessage;
    theMessage.setType(QMessage::Sms);
    theMessage.setParentAccountId(accountDetails["SMS"]);
    theMessage.setTo(QMessageAddress(QMessageAddress::Phone,"0404242552"));
    theMessage.setBody("KUKKUUUULUUruu");
    bool success = theService.send(theMessage);
}
*/
