/*
 * 3g2g-mode-selection-applet: Lets you switch the network access mode
 *
 * Copyright (C) 2009 Faheem Pervez <trippin1@gmail.com>. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *      
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *      
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 */

/* mode-selection-applet.c */

#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <gtk/gtk.h>
#include <hildon/hildon.h>
#include <libhildondesktop/libhildondesktop.h>

/* libcsnet D-Bus definitions */
#define NETWORK_BUS_NAME		"com.nokia.phone.net"
#define NETWORK_INTERFACE		"Phone.Net"
#define NETWORK_PATH			"/com/nokia/phone/net"
#define NETWORK_PATH_GET_RADIO_ACCESS	"get_selected_radio_access_technology"
#define NETWORK_PATH_SET_RADIO_ACCESS	"set_selected_radio_access_technology"

#define SSC_DBUS_NAME  "com.nokia.phone.SSC"
#define SSC_DBUS_IFACE SSC_DBUS_NAME
#define SSC_DBUS_PATH  "/com/nokia/phone/SSC"
#define SSC_MODEM_STATE_SIG  "modem_state_changed_ind"
#define SSC_MODEM_GETSTATE_METHOD "get_modem_state"
#define SSC_MODEM_ONLINE  "online"

/* Start of GObject boilerplate shit */
#define MODE_TYPE_SELECTION_APPLET (mode_selection_applet_get_type ())

#define MODE_SELECTION_APPLET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionApplet))

#define MODE_SELECTION_APPLET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletClass))

#define MODE_IS_SELECTION_APPLET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
						MODE_TYPE_SELECTION_APPLET))

#define MODE_IS_SELECTION_APPLET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), \
						MODE_TYPE_SELECTION_APPLET))

#define MODE_SELECTION_APPLET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletClass))

#define MODE_SELECTION_APPLET_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), \
						MODE_TYPE_SELECTION_APPLET, ModeSelectionAppletPrivate))

typedef struct _ModeSelectionApplet ModeSelectionApplet;
typedef struct _ModeSelectionAppletClass ModeSelectionAppletClass;
typedef struct _ModeSelectionAppletPrivate ModeSelectionAppletPrivate;

struct _ModeSelectionApplet
{
	HDStatusMenuItem parent;
};

struct _ModeSelectionAppletClass
{
	HDStatusMenuItemClass parent_class;
};

struct _ModeSelectionAppletPrivate
{
	GtkWidget *status_button;

	DBusGConnection *dbus_conn;
	DBusGProxy *phonet_proxy;
	DBusGProxy *ssc_proxy;
};
/* End of GObject boilerplate shit */

enum
{
	MODE_DUAL,
	MODE_GSM,
	MODE_3G,
};

const struct
{
	const gchar *msgid;
	const gchar *fallback_string;
} items[] =
{
	{"conn_va_phone_network_mode_b", "Dual"},
	{"conn_va_phone_network_mode_2g", "GSM"},
	{"conn_va_phone_network_mode_3g", "3G"},
};

HD_DEFINE_PLUGIN_MODULE (ModeSelectionApplet, mode_selection_applet, HD_TYPE_STATUS_MENU_ITEM)

static inline G_CONST_RETURN gchar* g_dgettext_localised_or_english (const gchar* domainname, const gchar* msgid, const gchar *fallback_string)
{
	G_CONST_RETURN gchar *retval = g_dgettext (domainname, msgid);

	return G_UNLIKELY (!g_strcmp0 (retval, msgid)) ? fallback_string : retval;
}

static guint mode_selection_applet_get_selected_radio_access_technology (ModeSelectionApplet *plugin)
{
	ModeSelectionAppletPrivate *priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);
	guint retval;
	guchar network_access;
	gint some_next_value;

	dbus_g_proxy_call (priv->phonet_proxy, NETWORK_PATH_GET_RADIO_ACCESS, NULL, G_TYPE_INVALID, G_TYPE_UCHAR, &network_access, G_TYPE_INT, &some_next_value, G_TYPE_INVALID);

	retval = network_access;

	return retval;
}

static void mode_selection_applet_set_selected_radio_access_technology (ModeSelectionApplet *plugin, const guchar mode)
{
	ModeSelectionAppletPrivate *priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);
	dbus_g_proxy_call_no_reply (priv->phonet_proxy, NETWORK_PATH_SET_RADIO_ACCESS, G_TYPE_UCHAR, mode, G_TYPE_INVALID);
}

static void mode_selection_applet_set_button_value (ModeSelectionApplet *plugin)
{
	g_return_if_fail (plugin);
	ModeSelectionAppletPrivate *priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);

	const guint radio_access = mode_selection_applet_get_selected_radio_access_technology (plugin);
	const gchar *label = g_dgettext_localised_or_english ("osso-connectivity-ui", items[radio_access].msgid, items[radio_access].fallback_string);

	hildon_button_set_value (HILDON_BUTTON (priv->status_button), label);
}

static void mode_selection_applet_on_button_clicked (GtkWidget *button G_GNUC_UNUSED, ModeSelectionApplet *plugin)
{
	switch (mode_selection_applet_get_selected_radio_access_technology (plugin))
	{
		case MODE_DUAL:
		case MODE_GSM:
			mode_selection_applet_set_selected_radio_access_technology (plugin, MODE_3G);
			break;

		case MODE_3G:
			mode_selection_applet_set_selected_radio_access_technology (plugin, MODE_GSM);
			break;
	}
}

static void mode_selection_applet_update_on_phone_state_changed (DBusGProxy *object, gchar *status, ModeSelectionApplet *plugin)
{
	if (!g_ascii_strncasecmp (SSC_MODEM_ONLINE, status, strlen (SSC_MODEM_ONLINE)))
	{
		mode_selection_applet_set_button_value (plugin);
		gtk_widget_show_all (GTK_WIDGET (plugin));
	}
	else
		if (GTK_WIDGET_VISIBLE (GTK_WIDGET (plugin)))
			gtk_widget_hide_all (GTK_WIDGET (plugin));

	if (object == NULL)
		g_free (status);
}

static void mode_selection_applet_finalize (GObject *object)
{
	ModeSelectionApplet *plugin = MODE_SELECTION_APPLET (object);
	ModeSelectionAppletPrivate *priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);

	if (priv->ssc_proxy)
	{
		dbus_g_proxy_disconnect_signal (priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_CALLBACK(mode_selection_applet_update_on_phone_state_changed), plugin);
		g_object_unref (G_OBJECT (priv->ssc_proxy));
	}

	if (priv->phonet_proxy)
		g_object_unref (G_OBJECT (priv->phonet_proxy));

	/* if (priv->dbus_conn)
		dbus_g_connection_unref (priv->dbus_conn); */ /* Allegedly, hd_status_plugin_item_get_dbus_g_connection () returns a shared connection. */

	G_OBJECT_CLASS (mode_selection_applet_parent_class)->finalize (object);
}

static void mode_selection_applet_class_init (ModeSelectionAppletClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = (GObjectFinalizeFunc) mode_selection_applet_finalize;

	g_type_class_add_private (klass, sizeof (ModeSelectionAppletPrivate));
}

static void mode_selection_applet_class_finalize (ModeSelectionAppletClass *klass G_GNUC_UNUSED)
{
}

static void mode_selection_applet_on_method_call_finished (DBusGProxy *proxy, DBusGProxyCall *call_id, ModeSelectionApplet *plugin)
{
	gchar *phone_state = NULL;

	dbus_g_proxy_end_call (proxy, call_id, NULL, G_TYPE_STRING, &phone_state, G_TYPE_INVALID);
	mode_selection_applet_update_on_phone_state_changed (NULL, phone_state, plugin);
}

static void mode_selection_applet_init (ModeSelectionApplet *plugin)
{
	GtkWidget *image = NULL;
	ModeSelectionAppletPrivate *priv = MODE_SELECTION_APPLET_GET_PRIVATE (plugin);

	priv->dbus_conn = hd_status_plugin_item_get_dbus_g_connection (HD_STATUS_PLUGIN_ITEM (&plugin->parent), DBUS_BUS_SYSTEM, NULL);
	g_return_if_fail (priv->dbus_conn);

	priv->phonet_proxy = dbus_g_proxy_new_for_name (priv->dbus_conn, NETWORK_BUS_NAME, NETWORK_PATH, NETWORK_INTERFACE);
	g_return_if_fail (priv->phonet_proxy);

	priv->ssc_proxy = dbus_g_proxy_new_for_name (priv->dbus_conn, SSC_DBUS_NAME, SSC_DBUS_PATH, SSC_DBUS_IFACE);
	g_return_if_fail (priv->phonet_proxy);

	priv->status_button = hildon_button_new_with_text (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH, HILDON_BUTTON_ARRANGEMENT_VERTICAL, g_dgettext_localised_or_english ("osso-connectivity-ui", "conn_fi_phone_network_mode", "Network mode"), NULL);
	hildon_button_set_style (HILDON_BUTTON (priv->status_button), HILDON_BUTTON_STYLE_PICKER);

	if (gtk_icon_theme_has_icon (gtk_icon_theme_get_default (), "3g2g-applet-signal"))
		image = gtk_image_new_from_icon_name ("3g2g-applet-signal", HILDON_ICON_SIZE_FINGER);
	else
	{
		GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/3g2g-applet-signal.png", NULL);
		if (G_LIKELY (pixbuf))
		{
			image = gtk_image_new_from_pixbuf (pixbuf);
			g_object_unref (pixbuf);
		}
	}
	if (G_LIKELY (image))
		hildon_button_set_image (HILDON_BUTTON (priv->status_button), image);

	hildon_button_set_alignment (HILDON_BUTTON (priv->status_button), 0, 0, 0, 0);

	g_signal_connect (G_OBJECT (priv->status_button), "clicked", G_CALLBACK (mode_selection_applet_on_button_clicked), plugin);

	gtk_container_add (GTK_CONTAINER (plugin), priv->status_button);

	dbus_g_proxy_add_signal (priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_TYPE_STRING, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (priv->ssc_proxy, SSC_MODEM_STATE_SIG, G_CALLBACK (mode_selection_applet_update_on_phone_state_changed), plugin, NULL);

	dbus_g_proxy_begin_call (priv->ssc_proxy, SSC_MODEM_GETSTATE_METHOD, (DBusGProxyCallNotify) mode_selection_applet_on_method_call_finished, plugin, NULL, G_TYPE_INVALID);
}

