/*
 * Copyright (C) 2008 Collabora Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "config.h"

#include "CString.h"
#include "Noncopyable.h"
#include "NotImplemented.h"
#include "ResourceHandle.h"
#include "ResourceHandleClient.h"
#include "ResourceRequest.h"
#include "ResourceResponse.h"
#include "webkitmarshal.h"
#include "webkitwebdownload.h"
#include "webkitprivate.h"
#include <glib/gstdio.h>

using namespace WebKit;
using namespace WebCore;

extern "C" {

class DownloadClient : Noncopyable, public WebCore::ResourceHandleClient {
    public:
        DownloadClient(WebKitWebDownload*);

        virtual void didReceiveResponse(ResourceHandle*, const ResourceResponse&);
        virtual void didReceiveData(ResourceHandle*, const char*, int, int);
        virtual void didFinishLoading(ResourceHandle*);
        virtual void didFail(ResourceHandle*, const ResourceError&);
        virtual void wasBlocked(ResourceHandle*);
        virtual void cannotShowURL(ResourceHandle*);

    private:
        WebKitWebDownload* m_download;
};

enum {
    /* normal signals */
    STARTED,
    PROGRESS_UPDATE,
    FINISHED,
    ERROR,
    LAST_SIGNAL
};

static guint webkit_web_download_signals[LAST_SIGNAL] = { 0, };

enum {
    PROP_0,

    PROP_NETWORK_REQUEST,
    PROP_LOCAL_URI,
    PROP_SUGGESTED_FILENAME,
    PROP_CURRENT_SIZE,
    PROP_TOTAL_SIZE,
};

G_DEFINE_TYPE(WebKitWebDownload, webkit_web_download, G_TYPE_OBJECT);

static void webkit_web_download_finalize(GObject* object)
{
    WebKitWebDownload* download = WEBKIT_WEB_DOWNLOAD(object);
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);

    /* We don't call webkit_web_download_cancel() because we don't want to emit
     * signals when finalising an object. */
    if (priv->resourceHandle) {
        if (priv->state == WEBKIT_WEB_DOWNLOAD_STATE_STARTED) {
            priv->resourceHandle->setClient(0);
            priv->resourceHandle->cancel();
        }
        priv->resourceHandle.release();
    }

    delete priv->downloadClient;
    delete priv->networkResponse;

    g_free(priv->localUri);
    g_free(priv->suggestedFilename);
    if (priv->outputChannel)
        g_io_channel_unref(priv->outputChannel);
    g_object_unref(priv->networkRequest);

    G_OBJECT_CLASS(webkit_web_download_parent_class)->finalize(object);
}

static void webkit_web_download_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec)
{
    WebKitWebDownload* download = WEBKIT_WEB_DOWNLOAD(object);

    switch(prop_id) {
    case PROP_NETWORK_REQUEST:
        g_value_set_object(value, webkit_web_download_get_network_request(download));
        break;
    case PROP_LOCAL_URI:
        g_value_set_string(value, webkit_web_download_get_local_uri(download));
        break;
    case PROP_SUGGESTED_FILENAME:
        g_value_set_string(value, webkit_web_download_get_suggested_filename(download));
        break;
    case PROP_CURRENT_SIZE:
        g_value_set_uint64(value, webkit_web_download_get_current_size(download));
        break;
    case PROP_TOTAL_SIZE:
        g_value_set_uint64(value, webkit_web_download_get_total_size(download));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    }
}

static void webkit_web_download_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec *pspec)
{
    WebKitWebDownload* download = WEBKIT_WEB_DOWNLOAD(object);
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);

    switch(prop_id) {
    case PROP_NETWORK_REQUEST:
        priv->networkRequest = (WebKitNetworkRequest*)g_value_dup_object(value);
        /* This is safe as network-request is a construct only property and suggestedFilename
         * is initially null. */
        priv->suggestedFilename = g_path_get_basename(webkit_network_request_get_uri(priv->networkRequest));
        break;
    case PROP_LOCAL_URI:
        webkit_web_download_set_local_uri(download, g_value_get_string(value));
        break;
    case PROP_SUGGESTED_FILENAME:
        webkit_web_download_set_suggested_filename(download, g_value_get_string(value));
        break;
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    }
}

static void webkit_web_download_class_init(WebKitWebDownloadClass* downloadClass)
{
    /*
     * implementations of virtual methods
     */
    GObjectClass* objectClass = G_OBJECT_CLASS(downloadClass);
    objectClass->finalize = webkit_web_download_finalize;
    objectClass->get_property = webkit_web_download_get_property;
    objectClass->set_property = webkit_web_download_set_property;

    /**
     * WebKitWebView::finished:
     * @download: the object on which the signal is emitted
     *
     * The file has been successfully downloaded.
     */
    webkit_web_download_signals[FINISHED] = g_signal_new("finished",
            G_TYPE_FROM_CLASS(downloadClass),
            (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
            0,
            NULL,
            NULL,
            webkit_marshal_VOID__VOID,
            G_TYPE_NONE, 0,
            G_TYPE_NONE);

    /**
     * WebKitWebView::started:
     * @download: the object on which the signal is emitted
     *
     * The download has started.  Happens just before the first progress-update signal.
     */
    webkit_web_download_signals[STARTED] = g_signal_new("started",
            G_TYPE_FROM_CLASS(downloadClass),
            (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
            0,
            NULL,
            NULL,
            webkit_marshal_VOID__VOID,
            G_TYPE_NONE, 0,
            G_TYPE_NONE);

    /**
     * WebKitWebView::progress-update:
     * @download: the object on which the signal is emitted
     * @current_bytes: the current count of bytes downloaded
     * @total_bytes: the total bytes count in the downloaded file, aka file size.
     *  
     * Indicates progress in the download.
     */
    webkit_web_download_signals[PROGRESS_UPDATE] = g_signal_new("progress-update",
            G_TYPE_FROM_CLASS(downloadClass),
            (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
            0,
            NULL,
            NULL,
            webkit_marshal_VOID__UINT_UINT,
            G_TYPE_NONE, 2,
            G_TYPE_UINT,
            G_TYPE_UINT);

    /**
     * WebKitWebView::progress-update:
     * @download: the object on which the signal is emitted
     * @current_bytes: the current count of bytes downloaded
     * @total_bytes: the total bytes count in the downloaded file, aka file size.
     *  
     * Indicates progress in the download.
     */
    webkit_web_download_signals[ERROR] = g_signal_new("error",
            G_TYPE_FROM_CLASS(downloadClass),
            (GSignalFlags)(G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION),
            0,
            g_signal_accumulator_true_handled,
            NULL,
            webkit_marshal_BOOLEAN__INT_INT_STRING,
            G_TYPE_BOOLEAN, 3,
            G_TYPE_INT,
            G_TYPE_INT,
            G_TYPE_STRING);

    /*
     * properties
     */
    g_object_class_install_property(objectClass,
                                    PROP_NETWORK_REQUEST,
                                    g_param_spec_object(
                                    "network-request",
                                    "Network Request",
                                    "The network request for the URI that should be downloaded",
                                    WEBKIT_TYPE_NETWORK_REQUEST,
                                    (GParamFlags)(WEBKIT_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY)));

    g_object_class_install_property(objectClass,
                                    PROP_LOCAL_URI,
                                    g_param_spec_string(
                                    "local-uri",
                                    "Local URI",
                                    "The local URI where to save the file",
                                    "",
                                    WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(objectClass,
                                    PROP_SUGGESTED_FILENAME,
                                    g_param_spec_string(
                                    "suggested-filename",
                                    "Suggested Filename",
                                    "The filename suggested as default when saving",
                                    "",
                                    WEBKIT_PARAM_READWRITE));

    g_object_class_install_property(objectClass,
                                    PROP_CURRENT_SIZE,
                                    g_param_spec_uint64(
                                    "current-size",
                                    "Current Size",
                                    "The length of the data already downloaded",
                                    0, G_MAXUINT64, 0,
                                    WEBKIT_PARAM_READABLE));

    g_object_class_install_property(objectClass,
                                    PROP_CURRENT_SIZE,
                                    g_param_spec_uint64(
                                    "total-size",
                                    "Total Size",
                                    "The total size of the file",
                                    0, G_MAXUINT64, 0,
                                    WEBKIT_PARAM_READABLE));

    g_type_class_add_private(downloadClass, sizeof(WebKitWebDownloadPrivate));
}

static void webkit_web_download_init(WebKitWebDownload* download)
{
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);

    priv->downloadClient = new DownloadClient(download);
    priv->currentSize = 0;
    priv->state = WEBKIT_WEB_DOWNLOAD_STATE_CREATED;
}

WebKitWebDownload* webkit_web_download_new(const gchar* uri)
{
    g_return_val_if_fail(uri, 0);

    WebKitNetworkRequest* request = webkit_network_request_new(uri);
    WebKitWebDownload* download = webkit_web_download_new_from_request(request);
    g_object_unref(request);
    return download;
}

WebKitWebDownload* webkit_web_download_new_from_request(WebKitNetworkRequest* request)
{
    g_return_val_if_fail(request, 0);

    WebKitWebDownload* download = WEBKIT_WEB_DOWNLOAD(g_object_new(WEBKIT_TYPE_WEB_DOWNLOAD, "network-request", request, NULL));
    return download;
}

// Private function.
// FIXME Use WebKitNetworkResponse when we have it
WebKitWebDownload* webkit_web_download_new_from_existing_connection(ResourceHandle* handle, WebKitNetworkRequest* request, const ResourceResponse& response)
{
    g_return_val_if_fail(request, 0);

    WebKitWebDownload* download = webkit_web_download_new_from_request (request);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    priv->resourceHandle = handle;
    priv->networkResponse = new ResourceResponse(response);

    return download;
}

static gboolean webkit_web_download_open_channel_with_tmp_file(WebKitWebDownload* download)
{
    gchar* filename;
    gint fd = g_file_open_tmp("download-XXXXXX", &filename, 0);
    if (fd < 0)
        return FALSE;

    gchar* uri = g_filename_to_uri(filename, 0, 0);
    g_free(filename);
    if (!uri)
        return FALSE;

    webkit_web_download_set_local_uri(download, uri);
    g_free(uri);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    priv->outputChannel = g_io_channel_unix_new(fd);
    if (priv->outputChannel)
        g_io_channel_set_close_on_unref(priv->outputChannel, TRUE);

    if (priv->outputChannel)
        g_io_channel_set_encoding(priv->outputChannel, 0, 0);

    return priv->outputChannel != 0;
}

static gboolean webkit_web_download_open_channel_for_uri(WebKitWebDownload* download, const gchar* uri, gboolean append=FALSE)
{
    g_return_val_if_fail(uri, FALSE);
 
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    gchar* filename = g_filename_from_uri(uri, 0, 0);
    if (!filename)
        return FALSE;

    priv->outputChannel = g_io_channel_new_file(filename, append ? "a" : "w", 0);
    g_free(filename);

    if (priv->outputChannel)
        g_io_channel_set_encoding(priv->outputChannel, 0, 0);

    return priv->outputChannel != 0;
}

static void webkit_web_download_close_channel(WebKitWebDownload* download)
{
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    if (priv->outputChannel) {
        g_io_channel_unref(priv->outputChannel);
        priv->outputChannel = 0;
    }
}

void webkit_web_download_start(WebKitWebDownload* download)
{    
    g_return_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download));

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    g_return_if_fail(priv->state == WEBKIT_WEB_DOWNLOAD_STATE_CREATED);

    if (priv->resourceHandle)
        priv->resourceHandle->setClient(priv->downloadClient);
    else {
        // FIXME use kit(priv->networkRequest) when WebKitNetworkRequest will be finished.
        ResourceRequest request(webkit_network_request_get_uri(priv->networkRequest));
        priv->resourceHandle = ResourceHandle::create(request, priv->downloadClient, 0, false, false, false);
    }

    if (priv->localUri)
        webkit_web_download_open_channel_for_uri(download, priv->localUri);
    else
        webkit_web_download_open_channel_with_tmp_file(download);

    if (!priv->outputChannel) {
        // FIXME Handle errors.
        g_warning("Failed to create the output file: %s", priv->localUri);
        return;
    }
}

void webkit_web_download_cancel (WebKitWebDownload* download)
{
    g_return_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download));

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);

    if (priv->resourceHandle)
        priv->resourceHandle->cancel();

    priv->state = WEBKIT_WEB_DOWNLOAD_STATE_CANCELLED;

    gboolean handled;
    g_signal_emit_by_name(download, "error", 0, WEBKIT_WEB_DOWNLOAD_ERROR_CANCELLED_BY_USER, "User cancelled the download", &handled);
}

const gchar* webkit_web_download_get_uri(WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), NULL);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return webkit_network_request_get_uri(priv->networkRequest);
}

WebKitNetworkRequest* webkit_web_download_get_network_request(WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), NULL);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return priv->networkRequest;
}

static void webkit_web_download_set_response(WebKitWebDownload* download, const ResourceResponse& response)
{
    // FIXME Use WebKitNetworkResponse when it's merged
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    priv->networkResponse = new ResourceResponse(response);
}

const gchar* webkit_web_download_get_suggested_filename(WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), NULL);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return priv->suggestedFilename;
}

void webkit_web_download_set_suggested_filename(WebKitWebDownload* download, const gchar* filename)
{
    g_return_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download));
    g_return_if_fail(filename);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    if (priv->suggestedFilename && !strcmp(priv->suggestedFilename, filename))
        return;

    g_free(priv->suggestedFilename);
    priv->suggestedFilename = g_strdup(filename);
    g_object_notify(G_OBJECT(download), "suggested-filename");
}

const gchar* webkit_web_download_get_local_uri(WebKitWebDownload* download)
{   
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), NULL);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return priv->localUri;
}

void webkit_web_download_set_local_uri(WebKitWebDownload* download, const gchar* localUri)
{   
    g_return_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download));
    g_return_if_fail(localUri);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    if (priv->localUri && !strcmp(priv->localUri, localUri))
        return;

    // FIXME can we have a better check?
    if (priv->state != WEBKIT_WEB_DOWNLOAD_STATE_CREATED && priv->state != WEBKIT_WEB_DOWNLOAD_STATE_CANCELLED) {
        ASSERT(priv->localUri);

        gboolean downloading = priv->outputChannel != 0;
        if (downloading)
            webkit_web_download_close_channel(download);

        gchar* srcFilename = g_filename_from_uri(priv->localUri, 0, 0);
        gchar* destFilename = g_filename_from_uri(localUri, 0, 0);
        if (!srcFilename || !destFilename)
            return;
        if (g_rename(srcFilename, destFilename) < 0) {
            g_warning("Impossible to move the download from %s to %s", srcFilename, destFilename);
            g_free(srcFilename);
            g_free(destFilename);
            return;
        }

        g_free(srcFilename);
        g_free(destFilename);

        if (downloading) {
            if (!webkit_web_download_open_channel_for_uri(download, localUri, TRUE)) {
                webkit_web_download_cancel(download);
                return;
            }
        }
    }

    g_free(priv->localUri);
    priv->localUri = g_strdup(localUri);
    g_object_notify(G_OBJECT(download), "local-uri");
}

WebKitWebDownloadState webkit_web_download_get_state (WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), WEBKIT_WEB_DOWNLOAD_STATE_ERROR);
    
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return priv->state;
}

// FIXME document that this can be different from expectedContentLength().
guint64 webkit_web_download_get_total_size (WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), 0);
    
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    if (!priv->networkResponse)
        return 0;

    return MAX(priv->currentSize, priv->networkResponse->expectedContentLength());
}

guint64 webkit_web_download_get_current_size (WebKitWebDownload* download)
{
    g_return_val_if_fail(WEBKIT_IS_WEB_DOWNLOAD(download), 0);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    return priv->currentSize;
}

static void webkit_web_download_received_data(WebKitWebDownload* download, const gchar* data, int length)
{
    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);

    if (priv->currentSize == 0) {
        priv->state = WEBKIT_WEB_DOWNLOAD_STATE_STARTED;
        g_signal_emit_by_name(download, "started");
    }

    ASSERT(priv->outputChannel);
    /* FIXME Make this async? Note that it's not possible with GIOChannel unless you buffer
     * the data in memory. */
    if (g_io_channel_write_chars(priv->outputChannel, data, length, NULL, NULL) != G_IO_STATUS_NORMAL) {
        // FIXME check errors
        g_assert_not_reached();
    }

    priv->currentSize += length;
    g_object_notify(G_OBJECT(download), "current-size");

    ASSERT(priv->networkResponse);
    if (priv->currentSize > priv->networkResponse->expectedContentLength())
        g_object_notify(G_OBJECT(download), "total-size");

    /* FIXME throttle the number of updates?
     * should we remove the previous g_object_notify()s if we are going to throttle
     * the progress updates? */
    g_signal_emit_by_name(download, "progress-update", (guint)priv->currentSize, (guint)webkit_web_download_get_total_size(download));
}

static void webkit_web_download_finished_loading(WebKitWebDownload* download)
{
    webkit_web_download_close_channel(download);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    priv->state = WEBKIT_WEB_DOWNLOAD_STATE_FINISHED;
    g_signal_emit_by_name(download, "finished");
}

static void webkit_web_download_error(WebKitWebDownload* download, const ResourceError& error)
{
    webkit_web_download_close_channel(download);

    WebKitWebDownloadPrivate* priv = WEBKIT_WEB_DOWNLOAD_GET_PRIVATE(download);
    priv->state = WEBKIT_WEB_DOWNLOAD_STATE_ERROR;

    gboolean handled;
    g_signal_emit_by_name(download, "error", 0, WEBKIT_WEB_DOWNLOAD_ERROR_NETWORK, error.localizedDescription().utf8().data(), &handled);
}

DownloadClient::DownloadClient(WebKitWebDownload* download)
    : m_download(download)
{
}

void DownloadClient::didReceiveResponse(ResourceHandle*, const ResourceResponse& response)
{
    webkit_web_download_set_response(m_download, response);
}

void DownloadClient::didReceiveData(ResourceHandle*, const char* data, int length, int lengthReceived)
{
    webkit_web_download_received_data(m_download, data, length);
}

void DownloadClient::didFinishLoading(ResourceHandle*)
{
    webkit_web_download_finished_loading(m_download);
}

void DownloadClient::didFail(ResourceHandle*, const ResourceError& error)
{
    webkit_web_download_error(m_download, error);
}

void DownloadClient::wasBlocked(ResourceHandle*)
{
    // FIXME when we have the new frame loader signals and error handling.
    notImplemented();
}

void DownloadClient::cannotShowURL(ResourceHandle*)
{
    // FIXME when we have the new frame loader signals and error handling.
    notImplemented();
}

}
