# Copyright (C) 2008 Luke Kenneth Casson Leighton <lkcl@lkcl.net>
# Copyright (C) 2008 Martin Soto <soto@freedesktop.org>
# Copyright (C) 2008 Alp Toker <alp@atoker.com>
# Copyright (C) 2009 Adam Dingle <adam@yorba.org>
# Copyright (C) 2009 Jim Nelson <jim@yorba.org>
#
# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Library General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Library General Public License for more details.
#
# You should have received a copy of the GNU Library General Public License
# along with this library; see the file COPYING.LIB.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.

package CodeGeneratorGObject;

my $licenceTemplate = << "EOF";
/*
    This file is part of the WebKit open source project.
    This file has been generated by generate-bindings.pl. DO NOT MODIFY!

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
 */
EOF

my %implIncludes = ();
my %hdrIncludes = ();
my %hdrPropIncludes = ();

# Default constructor
sub new {
    my $object = shift;
    my $reference = { };

    $codeGenerator = shift;
    $outputDir = shift;
    mkdir $outputDir;

    bless($reference, $object);
    return $reference;
}

sub finish {
    my $object = shift;
}

# Params: 'domClass' struct
sub GenerateInterface {
    my $object = shift;
    my $class = shift;
    my $defines = shift;

    my $name = $class->name;

    print "$name\n";
    
    $object->Generate($class);

    # Write private header
    $object->WritePrivateHeader($class);

    # Write public header and implementation
    my $fname = "WebKit_" . $name;
    $fname =~ s/_//g;
    $object->WriteData($fname);
}

# Params: 'idlDocument' struct
sub GenerateModule {
    my $object = shift;
    my $document = shift;

    $module = $document->module;
}

sub GetParentClassName {
    my $class = shift;

    return "WebKitDOMObject" if @{$class->parents} eq 0;
    return "WebKit" . $codeGenerator->StripModule($class->parents(0));
}

sub GetParentGObjectType {
    my $class = shift;

    return "WEBKIT_TYPE_DOM_OBJECT" if @{$class->parents} eq 0;
    return "WEBKIT_TYPE_" . uc(decamelize($codeGenerator->StripModule($class->parents(0))));
}

# Algorithm taken from String::CamelCase 0.01
sub camelize {
    my $s = shift;
    join('', map{ ucfirst $_ } split(/(?<=[A-Za-z])_(?=[A-Za-z])|\b/, $s));
}

# Algorithm taken from String::CamelCase 0.01
sub decamelize {
    my $s = shift;
    $s =~ s#([^a-zA-Z]?)([A-Z]*)([A-Z])([a-z]?)#
        my $fc = pos($s)==0;
        my ($p0,$p1,$p2,$p3) = ($1,lc$2,lc$3,$4);
        my $t = $p0 || $fc ? $p0 : '_';
        $t .= $p3 ? $p1 ? "${p1}_$p2$p3" : "$p2$p3" : "$p1$p2";
        $t;
    #ge;
    $s;
}

sub GetHeaderName {
    my $name = $codeGenerator->StripModule(shift);

    return "WebKit$name";
}

sub GetClassName {
    my $name = $codeGenerator->StripModule(shift);

    return "WebKit$name";
}

sub getIncludeHeader {
    my $type = shift;
    my $name = GetHeaderName($type);

    return "" if $type eq "boolean";
    return "" if $type eq "int";
    return "" if $type eq "long";
    return "" if $type eq "short";
    return "" if $type eq "char";
    return "" if $type eq "float";
    return "" if $type eq "double";
    return "" if $type eq "unsigned";
    return "" if $type eq "unsigned char";
    return "" if $type eq "unsigned int";
    return "" if $type eq "unsigned long";
    return "" if $type eq "unsigned long long";
    return "" if $type eq "unsigned short";
    return "" if $type eq "void";
    
    return "Range.h" if $type eq "CompareHow";
    return "" if $type eq "DOMString";
    return "" if $type eq "DOMTimeStamp";
    return "" if $type eq "RGBColor"; # TODO - DOM class wrapper

    return "$name.h";
}

# returns 1 if we need to create a GObject wrapper around the return result
sub IsGDOMClassType {
    my $type = shift;
    my $name = GetClassName($type);

    return 0 if $type eq "float";
    return 0 if $type eq "double";
    return 0 if $type eq "boolean";
    return 0 if $type eq "char";
    return 0 if $type eq "long";
    return 0 if $type eq "short";
    return 0 if $type eq "uchar";
    return 0 if $type eq "unsigned";
    return 0 if $type eq "int";
    return 0 if $type eq "unsigned int";
    return 0 if $type eq "unsigned long";
    return 0 if $type eq "unsigned long long";
    return 0 if $type eq "unsigned short";
    return 0 if $type eq "void";
    
    return 0 if $type eq "CompareHow";
    return 0 if $type eq "DOMString";
    return 0 if $type eq "DOMTimeStamp";
    return 0 if $type eq "RGBColor"; # TODO - DOM class wrapper

    return 1;
}

sub GetPropGType {
    my $type = shift;
    my $name = GetClassName($type);

    return "string" if $type eq "DOMString";
    return "float" if $type eq "float";
    return "notyetsupported" if $type eq "CompareHow";
    return "double" if $type eq "double";
    return "boolean" if $type eq "boolean";
    return "char" if $type eq "char";
    return "long" if $type eq "long";
    return "int" if $type eq "short";
    return "uchar" if $type eq "uchar";
    return "uint" if $type eq "unsigned";
    return "int" if $type eq "int";
    return "uint" if $type eq "unsigned int";
    return "uint64" if $type eq "unsigned long long";
    return "ulong" if $type eq "unsigned long";
    return "ushort" if $type eq "unsigned short";
    return "ulong" if $type eq "RGBColor"; # TODO - DOM class wrapper

    return "object";
}

sub GetGType {
    my $type = shift;
    my $name = GetClassName($type);

    return "gchar *" if $type eq "DOMString";
    return "gfloat" if $type eq "float";
    return "WebCore::Range::CompareHow " if $type eq "CompareHow";
    return "gdouble" if $type eq "double";
    return "gboolean" if $type eq "boolean";
    return "gchar" if $type eq "char";
    return "glong" if $type eq "long";
    return "gshort" if $type eq "short";
    return "guchar" if $type eq "uchar";
    return "guint" if $type eq "unsigned";
    return "gint" if $type eq "int";
    return "guint" if $type eq "unsigned int";
    return "gulong" if $type eq "unsigned long";
    return "guint64" if $type eq "unsigned long long";
    return "gushort" if $type eq "unsigned short";
    return "gulong" if $type eq "RGBColor"; # TODO - DOM class wrapper

    return "void" if $type eq "void";

    return "$name *";
}

sub addPropIncludeHeader {
    my $type = shift;

    my $hdr = getIncludeHeader($type);
    if ($hdr ne "") {
        if (IsGDOMClassType($type)) {
            $hdr = substr($hdr, 0, length($hdr)-2);     # strip off ".h"
            $hdrPropIncludes{"webkit/".$hdr.".h"} = 1;
            if ($hdr ne "WebKitDOMObject") {
                $hdrPropIncludes{"webkit/".$hdr."Private.h"} = 1;
            }
        } else {
            $hdrPropIncludes{$hdr} = 1;
        }
    }
}

sub addIncludeBody {
    my $type = shift;

    my $hdr = getIncludeHeader($type);
    if ($hdr ne "") {
        if (IsGDOMClassType($type)) {
            $implIncludes{"webkit/$hdr"} = 1;
        } else {
            $implIncludes{$hdr} = 1;
        }
    }
}

sub skipAttribute {
    my $attribute = shift;
    
    if ($attribute->signature->extendedAttributes->{"CustomGetter"} ||
        $attribute->signature->extendedAttributes->{"CustomSetter"}) {
        return 1;
    }
    
    my $prop_type = $attribute->signature->type;

    if ($prop_type eq "stylesheets::MediaList" or
        $prop_type eq "CSSVariablesDeclaration" or
        $prop_type eq "EventListener" or
        $prop_type =~ /Constructor$/ or
        $prop_type eq "custom" or
        $prop_type eq "Array" or
        $prop_type eq "DOMTimeStamp") {
        return 1;   # can't handle these yet
    }

    my $type = GetGType($prop_type);
    
    # TODO: SVG
    if ($codeGenerator->IsPodType($type) || $codeGenerator->IsSVGAnimatedType($type)) {
        return 1;
    }
    return 0;
}

sub UsesManualToJSImplementation {
    my $type = shift;

    # skipped out CanvasPixelArray - see JSCanvasPixelArrayCustom.cpp for why.
    return 1 if $type eq "Node" or $type eq "Document" or $type eq "HTMLCollection" or
      $type eq "SVGPathSeg" or $type eq "StyleSheet" or $type eq "CSSRule" or $type eq "CSSValue" or
      $type eq "Event" or $type eq "Element" or $type eq "Text";
    return 0;
}

sub WritePrivateHeader {
    my $object = shift;
    my $class = shift;

    my $interfaceName = $class->name;
    my $className = GetClassName($interfaceName);
    my $filename = "$outputDir/" . $className . "Private.h";
    my $guard = uc(decamelize($className)) . "_PRIVATE_H";
    my $parentClassName = GetParentClassName($class);

    my $hasLegacyParent = $class->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$class->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    
    open(PRIVHEADER, ">$filename") or die "Couldn't open file $filename for writing";
    
    print PRIVHEADER split("\r", $licenceTemplate);
    print PRIVHEADER "\n";
    
    my $text = << "EOF";
#ifndef $guard
#define $guard

#include <glib-object.h>
#include <webkit/${parentClassName}.h>
#include "${interfaceName}.h"
EOF

    print PRIVHEADER $text;
    
    print PRIVHEADER map { "#include \"$_\"\n" } sort keys(%hdrPropIncludes);
    print PRIVHEADER "\n" if keys(%hdrPropIncludes);
    
    $text = << "EOF";
typedef struct _${className} ${className};

namespace WebKit {
    ${className} *
    wrap${interfaceName}(WebCore::${interfaceName} *coreObject);

    WebCore::${interfaceName} *
    core(${className} *request);

EOF

    print PRIVHEADER $text;

    if (${interfaceName} eq "DOMWindow" || (!$hasParent or $class->extendedAttributes->{"GenerateToJS"}) and
        !UsesManualToJSImplementation($interfaceName)) {
        $text = << "EOF";
    gpointer
    toGDOM(WebCore::${interfaceName}* node);

EOF

        print PRIVHEADER $text;
    }

    $text = << "EOF";
} // namespace WebKit

#endif /* ${guard} */

EOF

    print PRIVHEADER $text;
    
    close(PRIVHEADER);
}

sub ClassHasFunction {
        my $class = shift;
        my $name = shift;
        foreach my $function (@{$class->functions}) {
            if ($function->signature->name eq $name) {
                return 1;
            }
        }
    return 0;
}

sub Uniquify {
    my $name = shift;
    my $avoid = shift;
    if ($name eq $avoid) {
        $name = $name . "2";
    }
    return $name;
}

sub ReturnIfFail {
    my $condition = shift;
    my $returnType = shift;
    
    if ($returnType eq "void") {
        return "    g_return_if_fail ($condition);\n";
    } else {
        return "    g_return_val_if_fail ($condition, 0);\n";  # TODO: return proper default result
    }
}

# Generate code for a function.  $kind should be either "" for an ordinary function,
# "get" for an attribute getter or "set" for an attribute setter.
sub GenerateFunction {
    my $object = shift;
    my $interfaceName = shift;
    my $function = shift;
    my $kind = shift;

    my $className = GetClassName($interfaceName);
    my $classCaps = uc(decamelize($interfaceName));

    my $fn_sig_name = $function->signature->name;
    my $fn_sig_type = $function->signature->type;
    my $interface_name = decamelize($interfaceName);
    my $prefix = $kind eq "get" ? "get_" : "";
    my $functionName = "webkit_" . $interface_name . "_" . $prefix . decamelize($fn_sig_name);
    my $returnType = GetGType($fn_sig_type);
    my $ret_is_class_type = IsGDOMClassType($fn_sig_type);

    my $this = $interface_name;
    $this =~ s/.*_//;
    
    my $functionSig = "$className *$this";

    my $call_impl_params = "";
    my $gdom_fn_prefix = "webkit_" . decamelize($fn_sig_type);

    if ($functionName eq "webkit_css_style_declaration_set_property") {
        # reserved name clash in glib!
        $functionName = "webkit_css_style_declaration_set_css_property";
    }
    # skip CompareHow functions for now
    if ($functionName eq "webkit_range_compare_boundary_points") {
        push(@hBody, "\n/* TODO: skipping CompareHow function ${functionName} */\n\n");
        push(@cBody, "\n/* TODO: skipping CompareHow function ${functionName} */\n\n");
        return;
    }

    # skip SVGDocument functions for now
    if ($returnType eq "WebKitVGDocument *") {
        push(@hBody, "\n/* TODO: skipping SVGDocument function ${functionName} */\n\n");
        push(@cBody, "\n/* TODO: skipping SVGDocument function ${functionName} */\n\n");
        return;
    }

    # skip some custom functions for now
    my $is_custom_fn = $function->signature->extendedAttributes->{"Custom"} ||
                       $function->signature->extendedAttributes->{"CustomArgumentHandling"};
    if ($functionName eq "webkit_html_collection_item" ||
        $functionName eq "webkit_html_collection_named_item" ||
        $functionName eq "webkit_xml_http_request_open" ||
        $functionName eq "webkit_xml_http_request_set_request_header" ||
        $functionName eq "webkit_xml_http_request_send" ||
        $functionName eq "webkit_xml_http_request_get_response_header" ||
        $functionName eq "webkit_xml_http_request_override_mime_type" ||
        $functionName eq "webkit_xml_http_request_dispatch_event" ||
        $functionName eq "webkit_element_set_attribute") {
        $is_custom_fn = 0;
    }
    if ($is_custom_fn && 
        $functionName ne "webkit_node_append_child" && 
        $functionName ne "webkit_node_remove_child" && 
        $functionName ne "webkit_node_insert_before" &&
        $functionName ne "webkit_node_replace_child") {
        push(@hBody, "\n/* TODO: custom function ${functionName} */\n\n");
        push(@cBody, "\n/* TODO: custom function ${functionName} */\n\n");
        return;
    }

    foreach my $param (@{$function->parameters}) {
        my $paramIDLType = $param->type;
        if ($paramIDLType eq "Event") {
            push(@hBody, "\n/* TODO: event function ${functionName} */\n\n");
            push(@cBody, "\n/* TODO: event function ${functionName} */\n\n");
            return;
        }
        addIncludeBody($paramIDLType);
        my $paramType = GetGType($paramIDLType);
        my $paramName = Uniquify(decamelize($param->name), $this);

        $functionSig .= ", $paramType $paramName";

        my $param_is_cls_type = IsGDOMClassType($paramIDLType);
        if ($param_is_cls_type) {
            if ($paramIDLType ne "DOMObject") {
                $implIncludes{"webkit/WebKit${paramIDLType}Private.h"} = 1;
            }
        }
        if ($param_is_cls_type || ($paramIDLType eq "DOMString")) {
            $paramName = "_g_" . $paramName;
        }
        if ($call_impl_params) {
            $call_impl_params .= ", $paramName";
        } else {
            $call_impl_params = "$paramName";
        }
    }

    if ($fn_sig_type eq "Event") {
        push(@hBody, "\n/* TODO: event function ${functionName} */\n\n");
        push(@cBody, "\n/* TODO: event function ${functionName} */\n\n");
        return;
    }

    if ($ret_is_class_type and $fn_sig_type ne "DOMObject") {
        $implIncludes{"webkit/WebKit${fn_sig_type}Private.h"} = 1;
        $implIncludes{"webkit/WebKit${fn_sig_type}.h"} = 1;
        $implIncludes{"${fn_sig_type}.h"} = 1;
    }

    if ($fn_sig_type eq 'DOMObject' &&
        ($functionName eq 'webkit_html_canvas_element_get_context' or
         $functionName eq 'webkit_document_get_css_canvas_context')) {
        return;       # too involved for now; skip
    }

    if(@{$function->raisesExceptions}) {
        $functionSig .= ", GError **error";
    }

    push(@hBody, "WEBKIT_API $returnType\n$functionName ($functionSig);\n\n");

    push(@cBody, "$returnType\n$functionName ($functionSig)\n{\n");

    push(@cBody, ReturnIfFail("WEBKIT_IS_$classCaps($this)", $returnType));
    
    # Check that arguments are non-NULL.  For attribute setters, we allow a NULL argument,
    # which can be used to remove an attribute.
    if ($kind ne "set") {
        foreach my $param (@{$function->parameters}) {
            if (!$codeGenerator->IsPrimitiveType($param->type)) {
                my $paramName = Uniquify(decamelize($param->name), $this);
                push(@cBody, ReturnIfFail($paramName, $returnType));
            }
        }
    }
    
    if(@{$function->raisesExceptions}) {
        push(@cBody, ReturnIfFail("!error || !*error", $returnType));
    }

    # The WebKit::core implementations check for NULL already; no need to
    # duplicate effort.
    push(@cBody, "    WebCore::${interfaceName} * item = WebKit::core($this);\n");

    $returnParamName = "";
    foreach my $param (@{$function->parameters}) {
        my $paramIDLType = $param->type;
        my $paramName = Uniquify(decamelize($param->name), $this);

        my $param_is_class_type = IsGDOMClassType($paramIDLType);
        # HACK!
        if ($functionName eq "webkit_xml_http_request_open" && $param->name eq "url") {
            push(@cBody, "    WebCore::KURL _g_${paramName} = WebCore::KURL($paramName);\n");
        } elsif ($paramIDLType eq "DOMString") {
            push(@cBody, "    WebCore::String _g_${paramName} = WebCore::String::fromUTF8($paramName);\n");
        }
        if ($param_is_class_type) {
            my $param_gdom_fn = "webkit_" . decamelize($paramIDLType);
            push(@cBody, "    WebCore::${paramIDLType} * _g_${paramName} = WebKit::core($paramName);\n");
        }
        $returnParamName = "_g_".$paramName if $param->extendedAttributes->{"Return"};
    }

    my $assign = "";
    my $assign_pre = "";
    my $assign_post = "";
    if ($returnType ne "void" && !$is_custom_fn) {
        if ($ret_is_class_type) {
            $assign = "PassRefPtr<WebCore::${fn_sig_type}> g_res = ";
            $assign_pre = "WTF::getPtr(";
            $assign_post = ")";
        } else {
            $assign = "${returnType} res = ";
        }
    }
    my $exceptions = "";
    if (@{$function->raisesExceptions}) {
        push(@cBody, "    WebCore::ExceptionCode ec = 0;\n");
        if (${call_impl_params} ne "") {
            $exceptions = ", ec";
        } else {
            $exceptions = "ec";
        }
    }

    # We need to special-case these Node methods because their C++ signature is different
    # from what we'd expect given their IDL description; see Node.h.
    if ($functionName eq "webkit_node_append_child" ||
        $functionName eq "webkit_node_insert_before" ||
        $functionName eq "webkit_node_replace_child" ||
        $functionName eq "webkit_node_remove_child") {
        my $custom_gdom_node_append_child = << "EOF";
    bool ok = item->${fn_sig_name}(${call_impl_params}${exceptions});
    if (ok)
    {
        ${returnType} res = static_cast<${returnType}>(WebKit::toGDOM($returnParamName));
        return res;
    }
EOF
        push(@cBody, $custom_gdom_node_append_child);
    
        if(@{$function->raisesExceptions}) {
            my $exception_handling = << "EOF";

    WebCore::ExceptionCodeDescription ecdesc;
    WebCore::getExceptionCodeDescription(ec, ecdesc);
    g_set_error(error, g_quark_from_string("GDOM"), ecdesc.code, ecdesc.name);
EOF
            push(@cBody, $exception_handling);
        }
        push(@cBody, "return NULL;");
        push(@cBody, "}\n\n");
        return;
    } elsif ($fn_sig_type eq "DOMObject" &&
         $functionName ne 'webkit_document_get_css_canvas_context') {
        push(@cBody, "    /* TODO - get canvas object (item->${fn_sig_name}(${call_impl_params}${exceptions}));*/\n" );

    } elsif ($fn_sig_type eq "DOMString") {
        push(@cBody, "    ${assign}gdom_gstring_convert(item->${fn_sig_name}(${call_impl_params}${exceptions}));\n" );
    } else {
        push(@cBody, "    ${assign}${assign_pre}item->${fn_sig_name}(${call_impl_params}${exceptions}${assign_post});\n" );
        
        if(@{$function->raisesExceptions}) {
            my $exception_handling = << "EOF";
    if(ec) {
        WebCore::ExceptionCodeDescription ecdesc;
        WebCore::getExceptionCodeDescription(ec, ecdesc);
        g_set_error(error, g_quark_from_string("GDOM"), ecdesc.code, ecdesc.name);
    }
EOF
            push(@cBody, $exception_handling);
        }
    }
    if ($returnType ne "void" && !$is_custom_fn) {

        if ($fn_sig_type ne "DOMObject") {
            if ($ret_is_class_type) {
                push(@cBody, "    ${returnType} res = static_cast<${returnType}>(WebKit::toGDOM(g_res.get()));\n");
            }
        }
        if ($fn_sig_type eq "DOMObject") {
            push(@cBody, "    return NULL; /* TODO: return canvas object */\n");
        } else {
            push(@cBody, "    return res;\n");
        }
    }
    push(@cBody, "\n");

    push(@cBody, "}\n\n");
}

sub CountGetProperties {
    my $class = shift;
    
    my $count = 0;
    
    foreach my $attribute (@{$class->attributes}) {
        if(!skipAttribute($attribute)) {
            # all properties not skipped are either read-only or read-write
            $count++;
        }
    }
    
    return $count;
}

sub CountSetProperties {
    my $class = shift;
    
    my $count = 0;
    
    SKIPPROP:
    foreach my $attribute (@{$class->attributes}) {
        if(skipAttribute($attribute)) {
            next SKIPPROP;
        }
        
        my $writeable = $attribute->type !~ /^readonly/;
        if($writeable && !$custom) {
            # custom properties are read-only for now
            $count++;
        }
    }
    
    return $count;
}

sub GenerateProperties {
    my $object = shift;
    my $class = shift;
    my $classCaps = shift;
    my $lcase_ifacename = shift;
    
    my $interfaceName = $class->name;
    my $className = GetClassName($interfaceName);   # e.g. WebKitElement
    
    # This function is expected to open a G_BEGIN_DECLS block, but does not end it
    push(@cBodyPriv, "G_BEGIN_DECLS\n\n");
    
    my $get_count = CountGetProperties($class);
    my $set_count = CountSetProperties($class);
    
    if(($get_count == 0) && ($set_count == 0)) {
        return ();
    }

  SKIPPROP:
    foreach my $attribute (@{$class->attributes}) {
        if (skipAttribute($attribute)) {
            next SKIPPROP;
        }
        
        my $prop_type = $attribute->signature->type;
        addPropIncludeHeader($prop_type);
    }

    my $implContent = << "EOF";
enum {
    PROP_0,
EOF
    push(@cBodyPriv, $implContent);

    my @txt_install_props = ();
    my @txt_set_props = ();
    my @txt_get_props = ();

    my $txt_get_prop = << "EOF";
static void ${lcase_ifacename}__get_property_fn(GObject * object, guint prop_id,
                     GValue * value,
                     GParamSpec * pspec)
{
    ${className} *self = WEBKIT_${classCaps}(object);
    WebCore::${interfaceName} * gself = WebKit::core(self);
    switch (prop_id) {
EOF
    push(@txt_get_props, $txt_get_prop);

    my $txt_set_prop = << "EOF";
static void ${lcase_ifacename}__set_property_fn(GObject * object, guint prop_id,
                     const GValue * value,
                     GParamSpec * pspec)
{
    ${className} *self = WEBKIT_${classCaps}(object);
    WebCore::${interfaceName} * gself = WebKit::core(self);
    switch (prop_id) {
EOF
    push(@txt_set_props, $txt_set_prop);
    
    # Iterate over the interface attributes and generate a property for
    # each one of them.
  SKIPENUM:
    foreach my $attribute (@{$class->attributes}) {
        if (skipAttribute($attribute)) {
            next SKIPENUM;
        }

        my $prop_name = $attribute->signature->name;
        my $setprop_name_fn = $codeGenerator->WK_ucfirst($prop_name);
        my $getprop_name_fn = $codeGenerator->WK_lcfirst($prop_name);
        my $prop_name_decamel = decamelize($prop_name);
        my $propName = $prop_name_decamel;
        $propName =~ s/_/-/g;
        my $propCaps = uc($prop_name_decamel);
        my $prop_type = $attribute->signature->type;
        my $type = GetGType($prop_type);
        my ${propEnum} = "PROP_${propCaps}";
        my ${prop_gtype} = decamelize($prop_type);
        my ${propGType} = uc($prop_gtype);
        my $lcase_gtypename = "webkit_" . $prop_gtype;

        push(@cBodyPriv, "    ${propEnum},\n");

        my $gtype = GetPropGType($prop_type);
        my $gparamflag = "G_PARAM_READABLE";
        my $writeable = $attribute->type !~ /^readonly/;
        my $const = "read-only ";
        if ($writeable && $custom) {
            $const = "read-only (due to custom functions needed in webkit/gobject)";
            next SKIPENUM;
        }
        if ($writeable && !$custom) {
            $gparamflag = "G_PARAM_READWRITE";
            $const = "read-write ";
        }

        my $_gtype = $gtype;
        if ($gtype eq "ushort") {
            $_gtype = "uint"; # yuk
        }
        $nick = decamelize(${interfaceName}) . "_" . $prop_name_decamel;
        $nick =~ s/_/ /g;
        $nick =~ s/\b(\w)/\U$1/g;   # capitalize each word
        $long = "${const} ${type} ${interfaceName}.${propName}";
        my $txt_install_prop = << "EOF";
    g_object_class_install_property (g_klass, ${propEnum},
            g_param_spec_${_gtype} ("${propName}", /* name */
                "$nick", /* short description */
                "$long", /* longer - could use some extra doc here */
EOF
        push(@txt_install_props, $txt_install_prop);
        $txt_install_prop = "/* TODO! $gtype */\n";

        if ($writeable) {
            my $ok = 0;
            my $convert_fn = "";

            if ($gtype eq "string") {
                $convert_fn = "WebCore::String::fromUTF8";
                $ok = 1;
            }
            elsif (($gtype eq "boolean" and ${interfaceName} ne "DOMWindow") or     # TODO: handle DOMWindow
                   $gtype eq "float" or
                   ($gtype eq "double" and ${interfaceName} ne "DOMWindow") or
                   $gtype eq "uint64" or
                   ($gtype eq "ulong" and ${interfaceName} ne "DOMWindow") or
                   ($gtype eq "long" and ${interfaceName} ne "DOMWindow") or
                   $gtype eq "uint" or
                   $gtype eq "ushort" or
                   $gtype eq "uchar" or
                   $gtype eq "char") {
                $ok = 1;
            }
            # TODO: handle Location
            if ($interfaceName eq "Location") {
                $ok = 0;
            }

            if ($ok) {
                push(@txt_set_props, "    case ${propEnum}:\n    {\n");
                push(@txt_set_props, "        WebCore::ExceptionCode ec = 0;\n") if @{$attribute->setterExceptions};
                
                if($attribute->signature->extendedAttributes->{"ConvertFromString"}) {
                    $convert_fn = "WebCore::String::number";
                }
                
                push(@txt_set_props, "        gself->set${setprop_name_fn}(${convert_fn}(g_value_get_$gtype (value))");

                push(@txt_set_props, ", ec") if @{$attribute->setterExceptions};
                push(@txt_set_props, ");\n");

                push(@txt_set_props, "        break;\n    }\n");
            }
        }

        push(@txt_get_props, "   case ${propEnum}:\n    {\n");

        my $exception = "";
        if (@{$attribute->getterExceptions}) {
            $exception = "ec";
            push(@txt_get_props, "        WebCore::ExceptionCode ec = 0;\n");
        }

        my $convert_fn = "";
        my $post_convert_fn = "";
        my $done = 0;
        if ($gtype eq "string") {
            push(@txt_get_props, "        g_value_take_string (value, gdom_gstring_convert(gself->${getprop_name_fn}(${exception})));\n");
            $done = 1;
        }
        elsif ($gtype eq "object") {
            $txt_get_prop = << "EOF";
        PassRefPtr<WebCore::${prop_type}> ptr = gself->${getprop_name_fn}(${exception});
        g_value_set_object (value, WebKit::toGDOM(ptr.get()));
EOF
            push(@txt_get_props, $txt_get_prop);

            $done = 1;
        }
        
        if($attribute->signature->extendedAttributes->{"ConvertFromString"}) {
            # TODO: Add other conversion functions for different types.  Current
            # IDLs only list longs.
            if($gtype eq "long") {
                $convert_fn = "";
                $post_convert_fn = ".toInt()";
            } else {
                die "Can't convert to type ${gtype}.";
            }
        }

        if (!$done) {
            push(@txt_get_props, "        g_value_set_$_gtype (value, ${convert_fn}gself->${getprop_name_fn}(${exception})${post_convert_fn});\n");
        }

        push(@txt_get_props, "        break;\n    }\n");

        # is this _entirely_ necessary???
        # can it be done with generic g_param_spec_gtype
        # and getting a few things, ignoring the min-max for example?
        if ($gtype eq "int") {
            $txt_install_prop = << "EOF";
                G_MININT,        /* min */
                G_MAXINT, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "boolean") {
            $txt_install_prop = << "EOF";
                FALSE, /* default */
EOF
        } elsif ($gtype eq "float") {
            $txt_install_prop = << "EOF";
                G_MINFLOAT,        /* min */
                G_MAXFLOAT, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "double") {
            $txt_install_prop = << "EOF";
                G_MINDOUBLE,        /* min */
                G_MAXDOUBLE, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "uint64") {
            $txt_install_prop = << "EOF";
                0,        /* min */
                G_MAXUINT64, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "ulong") {
            $txt_install_prop = << "EOF";
                0,        /* min */
                G_MAXULONG, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "long") {
            $txt_install_prop = << "EOF";
                G_MINLONG,        /* min */
                G_MAXLONG, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "uint") {
            $txt_install_prop = << "EOF";
                0,        /* min */
                G_MAXUINT, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "ushort") {
            $txt_install_prop = << "EOF";
                0,        /* min */
                G_MAXUINT16, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "uchar") {
            $txt_install_prop = << "EOF";
                G_MININT8,        /* min */
                G_MAXINT8, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "char") {
            $txt_install_prop = << "EOF";
                0,        /* min */
                G_MAXUINT8, /* max */
                0,        /* default */
EOF
        } elsif ($gtype eq "string") {
            $txt_install_prop = << "EOF";
                "", /* default */
EOF
        } elsif ($gtype eq "object") {
            $txt_install_prop = << "EOF";
                WEBKIT_TYPE_${propGType}, /* gobject type */
EOF
        }
        push(@txt_install_props, $txt_install_prop);
        $txt_install_prop = << "EOF";
                (GParamFlags)${gparamflag}));

EOF
        push(@txt_install_props, $txt_install_prop);
    }

    push(@cBodyPriv, "};\n\n");

    $txt_get_prop = << "EOF";
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}
EOF
    push(@txt_get_props, $txt_get_prop);

    $txt_set_prop = << "EOF";
    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
        break;
    }
}
EOF
    push(@txt_set_props, $txt_set_prop);
    
    if ($get_count > 0) {
        push(@cBodyPriv, @txt_get_props);
    }
    
    if ($set_count > 0) {
        push(@cBodyPriv, @txt_set_props);
    }
    
    return @txt_install_props;
}

# Generate header and implementation
sub Generate {
    my $object = shift;
    my $class = shift;

    my $hasLegacyParent = $class->extendedAttributes->{"LegacyParent"};
    my $hasRealParent = @{$class->parents} > 0;
    my $hasParent = $hasLegacyParent || $hasRealParent;
    my $parentClassName = GetParentClassName($class);
    my $parentGObjType = GetParentGObjectType($class);
    my $interfaceName = $class->name;
    my $className = GetClassName($interfaceName);   # e.g. WebKitElement
    my $hdrName = GetHeaderName($interfaceName);
    
    # Add the default implementation header template
    @cPrefix = split("\r", $licenceTemplate);
    push(@cPrefix, "\n");

    $implIncludes{"webkitmarshal.h"} = 1;
    $implIncludes{"webkitbinding.h"} = 1;
    $implIncludes{"webkit/$hdrName.h"} = 1;
    $implIncludes{"webkit/${hdrName}Private.h"} = 1;
    $implIncludes{"gobject/gstringconvert.h"} = 1;
    $implIncludes{"${interfaceName}.h"} = 1;
    $implIncludes{"ExceptionCode.h"} = 1;
    
    my $parentInterfaceName = substr($parentClassName, length("WebKit"));     # strip WebKit prefix
    my $parentHdrName = GetHeaderName($parentInterfaceName);
    $implIncludes{"webkit/${parentHdrName}.h"} = 1;
    if ($parentHdrName ne "WebKitDOMObject") {
        $implIncludes{"webkit/${parentHdrName}Private.h"} = 1;
    }
    $hdrIncludes{"webkit/${parentHdrName}.h"} = 1;

    # Get correct pass/store types respecting PODType flag
    my $podType = $class->extendedAttributes->{"PODType"};
    my $passType = $podType ? "JSSVGPODTypeWrapper<$podType>*" : "$interfaceName*";

    if (${interfaceName} eq "DOMWindow" || (!$hasParent or $class->extendedAttributes->{"GenerateToJS"})
        and !UsesManualToJSImplementation($interfaceName)) {
        my $converter = << "EOF";
namespace WebKit {
    
gpointer toGDOM(WebCore::$passType obj)
{
    g_return_val_if_fail(obj != NULL, NULL);

    if (gpointer ret = GDOMObjectCache::getDOMObject(obj))
        return ret;

    return GDOMObjectCache::putDOMObject(obj, WebKit::wrap${interfaceName}(obj));
}
    
} // namespace WebKit //

EOF

        push(@cBody, $converter);
    }

    my $implContent = "";

    push(@cBody, "extern \"C\" {\n\n");

    # Add the default header template
    @hPrefix = split("\r", $licenceTemplate);
     
    # Header guard
    my $guard = "__" . $className . "_H__";

    my $classCaps = uc(decamelize($interfaceName));
    my $lcase_ifacename = "webkit_" . decamelize($interfaceName);

    @hPrefixGuard = << "EOF";
#ifndef $guard
#define $guard
EOF

    $implContent = << "EOF";
G_BEGIN_DECLS

EOF

    push(@hBodyPre, $implContent);

    $implContent = << "EOF";

#define WEBKIT_TYPE_${classCaps}            (${lcase_ifacename}_get_type())
#define WEBKIT_${classCaps}(obj)            (G_TYPE_CHECK_INSTANCE_CAST((obj), WEBKIT_TYPE_${classCaps}, ${className}))
#define WEBKIT_${classCaps}_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST((klass),  WEBKIT_TYPE_${classCaps}, $className)Class)
#define WEBKIT_IS_${classCaps}(obj)         (G_TYPE_CHECK_INSTANCE_TYPE((obj), WEBKIT_TYPE_${classCaps}))
#define WEBKIT_IS_${classCaps}_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE((klass),  WEBKIT_TYPE_${classCaps}))
#define WEBKIT_${classCaps}_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS((obj),  WEBKIT_TYPE_${classCaps}, ${className}Class))

struct _${className} {
    ${parentClassName} parent_instance;
};

struct _${className}Class {
    ${parentClassName}Class parent_class;
};

WEBKIT_API GType
${lcase_ifacename}_get_type (void);

EOF

    push(@hBody, $implContent);

    $implContent = << "EOF";

G_BEGIN_DECLS

G_DEFINE_TYPE(${className}, ${lcase_ifacename}, ${parentGObjType});

G_END_DECLS

namespace WebKit {

${className} *
wrap${interfaceName}(WebCore::${interfaceName} *coreObject)
{
    g_return_val_if_fail(coreObject != NULL, NULL);
    
    ${className} *wrapper = WEBKIT_${classCaps}(g_object_new(WEBKIT_TYPE_${classCaps}, NULL));
    
    coreObject->ref();
    WEBKIT_DOM_OBJECT(wrapper)->coreObject = coreObject;

    return wrapper;
}

WebCore::${interfaceName} *
core(${className} *request)
{
    g_return_val_if_fail(request != NULL, NULL);
    
    WebCore::${interfaceName} *coreObject = static_cast<WebCore::${interfaceName} *>(WEBKIT_DOM_OBJECT(request)->coreObject);
    g_assert(coreObject != NULL);
    
    return coreObject;
}

} // namespace WebKit /

EOF
    push(@cBodyPriv, $implContent);

    my @txt_install_props = $object->GenerateProperties($class, $classCaps, $lcase_ifacename);
    
    my $install_get_prop = "// no get properties";
    my $install_set_prop = "// no set properties";
    
    if (CountGetProperties($class) > 0) {
        $install_get_prop = "g_klass->get_property = ${lcase_ifacename}__get_property_fn;";
    }
    
    if (CountSetProperties($class) > 0) {
        $install_set_prop = "g_klass->set_property = ${lcase_ifacename}__set_property_fn;";
    }

    $implContent = << "EOF";

static void ${lcase_ifacename}_finalize(GObject* object)
{
    WebKitDOMObject* dom_object = WEBKIT_DOM_OBJECT(object);
    
    if (dom_object != NULL && dom_object->coreObject != NULL) {
        WebCore::${interfaceName} *coreObject = static_cast<WebCore::${interfaceName} *>(dom_object->coreObject);
        coreObject->deref();
        
        dom_object->coreObject = NULL;
        
        WebKit::GDOMObjectCache::forgetDOMObject(coreObject);
    }

    G_OBJECT_CLASS(${lcase_ifacename}_parent_class)->finalize(object);
}

static void ${lcase_ifacename}_class_init(${className}Class* requestClass)
{
    GObjectClass *g_klass = G_OBJECT_CLASS (requestClass);
    g_klass->finalize = ${lcase_ifacename}_finalize;
    ${install_get_prop}
    ${install_set_prop}

@txt_install_props
}

static void ${lcase_ifacename}_init(${className}* request)
{
}

EOF
    push(@cBodyPriv, $implContent);

    $implIncludes{"webkit/WebKit${interfaceName}Private.h"} = 1;

    $implIncludes{"${interfaceName}.h"} = 1;

    foreach my $function (@{$class->functions}) {
        $object->GenerateFunction($interfaceName, $function, "");
    }

    TOP:
    foreach my $attribute (@{$class->attributes}) {
        if (skipAttribute($attribute)) {
            next TOP;
        }
        
        if ($attribute->signature->name eq "type"
            # This will conflict with the get_type() function we define to return a GType
            # according to GObject conventions.  Skip this for now.
            || $attribute->signature->name eq "URL"     # TODO: handle this
            || $attribute->signature->extendedAttributes->{"ConvertFromString"}    # TODO: handle this
            ) {
            next TOP;
        }
            
        my $attname_upper = $codeGenerator->WK_ucfirst($attribute->signature->name);
        my $getname = "get${attname_upper}";
        my $setname = "set${attname_upper}";
        if (ClassHasFunction($class, $getname) || ClassHasFunction($class, $setname)) {
            # Very occasionally an IDL file defines getter/setter functions for one of its
            # attributes; in this case we don't need to autogenerate the getter/setter.
            next TOP;
        }
        
        # Generate an attribute getter.  For an attribute "foo", this is a function named
        # "get_foo" which calls a DOM class method named foo().
        my $function = new domFunction();
        $function->signature($attribute->signature);
        $function->raisesExceptions($attribute->getterExceptions);
        $object->GenerateFunction($interfaceName, $function, "get");
        
        if ($attribute->type =~ /^readonly/ ||
            $attribute->signature->extendedAttributes->{"Replaceable"}  # can't handle this yet
            ) {
            next TOP;
        }
        
        # Generate an attribute setter.  For an attribute, "foo", this is a function named
        # "set_foo" which calls a DOM class method named setFoo().
        $function = new domFunction();
        
        $function->signature(new domSignature());
        $function->signature->name($setname);
        $function->signature->type("void");
        $function->signature->extendedAttributes($attribute->signature->extendedAttributes);
        
        my $param = new domSignature();
        $param->name("value");
        $param->type($attribute->signature->type);
        my %attributes = ();
        $param->extendedAttributes(attributes);
        my $arrayRef = $function->parameters;
        push(@$arrayRef, $param);
        
        $function->raisesExceptions($attribute->setterExceptions);
        
        $object->GenerateFunction($interfaceName, $function, "set");
    }

    push(@cBody, "} /* extern \"C\" */\n\n");
    push(@cBodyPriv, "G_END_DECLS\n\n");

    push(@hBody, "G_END_DECLS\n\n");
    push(@hBody, "#endif /* $guard */\n");
}

# Internal helper
sub WriteData {
    my ($object, $name) = @_;

    # Write public header.
    my $hdrFName = "$outputDir/" . $name . ".h";
    open(HEADER, ">$hdrFName") or die "Couldn't open file $hdrFName";

    print HEADER @hPrefix;
    print HEADER @hPrefixGuard;
    print HEADER "\n#include \"webkit/webkitdomdefines.h\"\n";
    print HEADER "#include <glib-object.h>\n";
    print HEADER "#include <webkit/webkitdefines.h>\n";
    print HEADER map { "#include \"$_\"\n" } sort keys(%hdrIncludes);
    print HEADER "\n" if keys(%hdrIncludes);
    print HEADER "\n";
    print HEADER @hBodyPre;
    print HEADER @hBody;

    close(HEADER);

    # Write the implementation
    my $implFileName = "$outputDir/" . $name . ".cpp";
    open(IMPL, ">$implFileName") or die "Couldn't open file $implFileName";

    print IMPL @cPrefix;
    print IMPL "#include <glib-object.h>\n";
    print IMPL "#include \"config.h\"\n\n";
    print IMPL "#include <wtf/GetPtr.h>\n";
    print IMPL "#include <wtf/PassRefPtr.h>\n";
    print IMPL map { "#include \"$_\"\n" } sort keys(%implIncludes);
    print IMPL "\n" if keys(%implIncludes);
    print IMPL @cBody;

    print IMPL "\n";
    print IMPL @cBodyPriv;

    close(IMPL);

    %implIncludes = ();
    %hdrIncludes = ();
    @hPrefix = ();
    @hPrefixGuard = ();
    @hBodyPre = ();
    @hBody = ();

    @cPrefix = ();
    @cBody = ();
    @cBodyPriv = ();
}

1;
