#include <hildon/hildon.h>

#include <locale.h>
#include <config.h>
#include <glib/gi18n.h>

#include "userdata.h"
#include "views.h"
#include "view-main.h"
#include "view-settings.h"
#include "settings.h"
#include "app-menu.h"
#include "view-history.h"

#include "glue.h"

static void show_search(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;
    const gchar *search;

    if (gtk_entry_get_text_length(GTK_ENTRY(u->main.entry_search))) {
        search = gtk_entry_get_text(GTK_ENTRY(u->main.entry_search));
        g_print("search with %s\n", search);
        yaspot_show_search(u, search);
    }
}

static void show_playlists(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_stored_playlists(u);
}

/* called from outside gtk thread context */
static void reload_playlists_reply(yaspot_glue_t *t, gpointer response, gpointer userdata) {
    userdata_t *u = (userdata_t*)userdata;

    gdk_threads_enter();
    yaspot_main_set_sensitive(u, TRUE);
    gdk_threads_leave();
}

static void reload_playlists(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_main_set_sensitive(u, FALSE);

    yaspot_playlist_reload(u->session, reload_playlists_reply, u);
}

static void show_queue(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_queue(u);
}

static void show_history(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_history(u);
}

static void show_settings(GtkWidget *widget, gpointer data) {
    userdata_t *u = (userdata_t*)data;

    yaspot_show_settings(u);
}

GtkWidget* yaspot_view_main_new(userdata_t *u) {
    GtkWidget *win;
    GtkWidget *vbox;
    GtkWidget *search_box;
    GtkWidget *playlist_box;

    win = hildon_stackable_window_new();
    gtk_window_set_title(GTK_WINDOW(win), "yaspot");

    u->main.button_search = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                _("Search"),
                                                NULL);

    u->main.button_playlists = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                   HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                   _("Stored Playlists"),
                                                   NULL);

    u->main.button_reload_playlists = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                                  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                                  _("Reload"),
                                                                  NULL);

    u->main.button_queue = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                               HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                               _("Playing Queue"),
                                               NULL);

    u->main.button_history = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                               HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                               _("History"),
                                               NULL);

    u->main.button_settings = hildon_button_new_with_text(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT,
                                                  HILDON_BUTTON_ARRANGEMENT_VERTICAL,
                                                  _("Settings"),
                                                  NULL);

    u->main.entry_search = hildon_entry_new(HILDON_SIZE_AUTO);

    vbox = gtk_vbox_new(FALSE, 0);
    search_box = gtk_hbox_new(FALSE, 0);
    playlist_box = gtk_hbox_new(FALSE, 0);

    gtk_box_pack_start_defaults(GTK_BOX(search_box), u->main.button_search);
    gtk_box_pack_start_defaults(GTK_BOX(search_box), u->main.entry_search);

    gtk_box_pack_start_defaults(GTK_BOX(playlist_box), u->main.button_playlists);
    gtk_box_pack_start_defaults(GTK_BOX(playlist_box), u->main.button_reload_playlists);

    gtk_box_pack_start(GTK_BOX(vbox),
                       search_box,
                       FALSE,
                       FALSE,
                       7);
    gtk_box_pack_start(GTK_BOX(vbox),
                       playlist_box,
                       FALSE,
                       FALSE,
                       7);
    gtk_box_pack_start(GTK_BOX(vbox),
                       u->main.button_queue,
                       FALSE,
                       FALSE,
                       7);
    gtk_box_pack_start(GTK_BOX(vbox),
                       u->main.button_history,
                       FALSE,
                       FALSE,
                       7);

    gtk_box_pack_start(GTK_BOX(vbox),
                       u->main.button_settings,
                       FALSE,
                       FALSE,
                       7);

    gtk_container_add(GTK_CONTAINER(win), vbox);

    /* search button & entry */
    g_signal_connect(u->main.button_search, "clicked", G_CALLBACK(show_search), u);
    g_signal_connect(u->main.entry_search, "activate", G_CALLBACK(show_search), u);

    /* playlists */
    g_signal_connect(u->main.button_playlists, "clicked", G_CALLBACK(show_playlists), u);

    /* playlists */
    g_signal_connect(u->main.button_reload_playlists, "clicked", G_CALLBACK(reload_playlists), u);

    /* playing queue */
    g_signal_connect(u->main.button_queue, "clicked", G_CALLBACK(show_queue), u);

    /* history list */
    g_signal_connect(u->main.button_history, "clicked", G_CALLBACK(show_history), u);

    /* settings dialog */
    g_signal_connect(u->main.button_settings, "clicked", G_CALLBACK(show_settings), u);

    /* x */
    g_signal_connect(G_OBJECT(win), "delete_event",
                     G_CALLBACK(gtk_main_quit), NULL);

    return win;
}

enum {
    BUTTON_LOGIN = 9999,
    BUTTON_OFFLINE
};

void yaspot_display_login(userdata_t *u, gint *quit) {
    GtkWidget *caption_user;
    GtkWidget *caption_pass;
    GtkWidget *entry_user;
    GtkWidget *entry_pass;
    GtkSizeGroup *group;
    GtkWidget *content_area;
    GtkWidget *save_credentials;
    GtkWidget *high_bitrate;
    gchar *username = NULL;
    gchar *password = NULL;
    gint result;
    gint save_cred;
    gint high_br;

    *quit = 0;

    u->login_dialog = gtk_dialog_new();
    gtk_window_set_transient_for(GTK_WINDOW(u->login_dialog), GTK_WINDOW(u->main.window));
    gtk_window_set_title(GTK_WINDOW(u->login_dialog), _("Login"));

    entry_user = hildon_entry_new(HILDON_SIZE_AUTO);
    entry_pass = hildon_entry_new(HILDON_SIZE_AUTO);
    g_object_set (entry_user, "hildon-input-mode", HILDON_GTK_INPUT_MODE_FULL, NULL);
    g_object_set (entry_pass, "hildon-input-mode", HILDON_GTK_INPUT_MODE_FULL, NULL);
    gtk_entry_set_visibility(GTK_ENTRY(entry_pass), FALSE);

    group = gtk_size_group_new(GTK_SIZE_GROUP_BOTH);

    caption_user = hildon_caption_new(group, _("Username"), entry_user, NULL, HILDON_CAPTION_MANDATORY);
    caption_pass = hildon_caption_new(group, _("Password"), entry_pass, NULL, HILDON_CAPTION_MANDATORY);
    g_object_unref(group);

    save_credentials = hildon_check_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label(GTK_BUTTON(save_credentials), _("Save credentials"));
    high_bitrate = hildon_check_button_new(HILDON_SIZE_AUTO_WIDTH | HILDON_SIZE_FINGER_HEIGHT);
    gtk_button_set_label(GTK_BUTTON(high_bitrate), _("High bitrate"));

    /* settings stuff */
    if (!yaspot_settings_save_credentials(&save_cred)) {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(save_credentials), save_cred);

        if (save_cred) {
            username = yaspot_settings_username();
            password = yaspot_settings_password();

            if (username)
                hildon_entry_set_text(HILDON_ENTRY(entry_user), username);
            if (password)
                hildon_entry_set_text(HILDON_ENTRY(entry_pass), password);

            g_free(username);
            g_free(password);
        }
    } else {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(save_credentials), 0);
    }

    if (!yaspot_settings_high_bitrate(&high_br)) {
        hildon_check_button_set_active(HILDON_CHECK_BUTTON(high_bitrate), high_br);
    }

    gtk_dialog_add_button(GTK_DIALOG(u->login_dialog), _("Login"), BUTTON_LOGIN);
    /* OFFLINE disabled for now. */
    /* gtk_dialog_add_button(dialog, "Offline", BUTTON_OFFLINE); */

    content_area = gtk_dialog_get_content_area (GTK_DIALOG (u->login_dialog));

    gtk_box_pack_start(GTK_BOX(content_area),
                       caption_user,
                       FALSE,
                       FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(content_area),
                       caption_pass,
                       FALSE,
                       FALSE,
                       0);
    gtk_box_pack_start(GTK_BOX(content_area),
                       save_credentials,
                       FALSE,
                       FALSE,
                       4);
    gtk_box_pack_start(GTK_BOX(content_area),
                       high_bitrate,
                       FALSE,
                       FALSE,
                       2);

    gtk_widget_show_all(GTK_WIDGET(u->login_dialog));
    result = gtk_dialog_run(GTK_DIALOG(u->login_dialog));

    if (hildon_check_button_get_active(HILDON_CHECK_BUTTON(save_credentials))) {
        yaspot_settings_set_save_credentials(1);
        yaspot_settings_set_username(hildon_entry_get_text(HILDON_ENTRY(entry_user)));
        yaspot_settings_set_password(hildon_entry_get_text(HILDON_ENTRY(entry_pass)));
    } else {
        yaspot_settings_set_save_credentials(0);
        yaspot_settings_set_username("foo");
        yaspot_settings_set_password("bar");
    }

    high_br = hildon_check_button_get_active(HILDON_CHECK_BUTTON(high_bitrate));
    yaspot_settings_set_high_bitrate(high_br);

    switch (result) {
        case BUTTON_LOGIN:
            g_print("try login\n");
            u->connection.status = YASPOT_CONNECTING;

            yaspot_glue_set_credentials(u->session,
                                        hildon_entry_get_text(HILDON_ENTRY(entry_user)),
                                        hildon_entry_get_text(HILDON_ENTRY(entry_pass)));
            yaspot_glue_set_high_bitrate(u->session, !!high_br);

            hildon_gtk_window_set_progress_indicator(GTK_WINDOW(u->login_dialog), TRUE);
            gtk_widget_set_sensitive(GTK_WIDGET(entry_user), FALSE);
            gtk_widget_set_sensitive(GTK_WIDGET(entry_pass), FALSE);
            gtk_widget_set_sensitive(GTK_WIDGET(save_credentials), FALSE);
            gtk_widget_set_sensitive(GTK_WIDGET(high_bitrate), FALSE);

            yaspot_glue_connect(u->session);
            break;

        case BUTTON_OFFLINE:
            g_print("offline not supported\n");
            gtk_widget_destroy(u->login_dialog);
            *quit = 1;
            break;
        default:
            g_print("cancelled -> quit\n");
            gtk_widget_destroy(u->login_dialog);
            u->connection.status = YASPOT_DISCONNECTED;
            *quit = 1;
            break;
    }
}

void yaspot_show_main(userdata_t *u) {
    if (!u->main.window) {
        u->main.window = yaspot_view_main_new(u);
    }

    yaspot_app_menu_attach(u, HILDON_WINDOW(u->main.window));

    gtk_widget_show_all(u->main.window);
}

void yaspot_main_set_sensitive(userdata_t *u, gboolean sensitive) {
    g_assert(u);

    gtk_widget_set_sensitive(u->main.button_search, sensitive);
    gtk_widget_set_sensitive(u->main.entry_search, sensitive);
    gtk_widget_set_sensitive(u->main.button_playlists, sensitive);
    gtk_widget_set_sensitive(u->main.button_reload_playlists, sensitive);
    gtk_widget_set_sensitive(u->main.button_queue, sensitive);
    gtk_widget_set_sensitive(u->main.button_history, sensitive);
    gtk_widget_set_sensitive(u->main.button_settings, sensitive);
    hildon_gtk_window_set_progress_indicator(GTK_WINDOW(u->main.window), !sensitive);
}

// vim: expandtab shiftwidth=4

