#include <hildon/hildon.h>

#include <locale.h>
#include <config.h>
#include <glib/gi18n.h>

#include "userdata.h"
#include "history.h"
#include "views.h"
#include "view-history.h"

static void cancel_cb(GtkWidget *widget, GdkEvent *event, gpointer userdata) {
    userdata_t *u = (userdata_t*)userdata;
    g_assert(u);

    gtk_widget_destroy(u->history_dialog);
    u->history_dialog = NULL;
}

static void selection_changed(HildonTouchSelector *selector, gint column, gpointer userdata) {
    userdata_t *u = (userdata_t*)userdata;
    gint active;
    gint i = 0;
    const yaspot_history_entry_t *e;
    void *state = NULL;

    g_assert(u);

    active = hildon_touch_selector_get_active(selector, column);

    while ((e = yaspot_history_iterate(u->history, &state))) {
        if (i == active)
            break;
        i++;
    }

    if (e) {
        switch (e->operation) {
            case HISTORY_SEARCH:
                yaspot_show_search(u, e->id);
                break;

            case HISTORY_ALBUM:
                yaspot_show_album_by_id(u, e->id, u->main.window);
                break;

            case HISTORY_PLAYLIST:
                yaspot_show_stored_list_by_id(u, e->id, u->main.window);
                break;
        }
    }

    cancel_cb(NULL, NULL, u);
}

static GtkWidget*create_selector() {
    GtkWidget *selector = NULL;
    GtkListStore *list_store = NULL;
    GtkCellRenderer *renderer = NULL;
    HildonTouchSelectorColumn *column = NULL;

    selector = hildon_touch_selector_new();

    list_store = gtk_list_store_new(1, G_TYPE_STRING);

    renderer = gtk_cell_renderer_text_new();
    gtk_cell_renderer_set_fixed_size(renderer, 700, -1);

    column = hildon_touch_selector_append_column(HILDON_TOUCH_SELECTOR (selector),
                                                 GTK_TREE_MODEL(list_store),
                                                 renderer, "text", 0, NULL);

    hildon_touch_selector_column_set_text_column(column, 0);

    g_object_unref(list_store);

    hildon_touch_selector_set_column_selection_mode(HILDON_TOUCH_SELECTOR (selector),
                                                    HILDON_TOUCH_SELECTOR_SELECTION_MODE_SINGLE);

    return selector;
}

void yaspot_show_history(userdata_t *u) {
    GtkWidget *content_area;
    GtkWidget *selector;
    void *state = NULL;
    const yaspot_history_entry_t *e;

    g_assert(u);
    g_assert(!u->history_dialog);

    u->history_dialog = gtk_dialog_new();
    gtk_window_set_transient_for(GTK_WINDOW(u->history_dialog), GTK_WINDOW(u->main.window));
    gtk_window_set_title(GTK_WINDOW(u->history_dialog), _("History"));

    content_area = gtk_dialog_get_content_area(GTK_DIALOG (u->history_dialog));

    selector = create_selector();

    while ((e = yaspot_history_iterate(u->history, &state))) {
        gchar *label;

        switch (e->operation) {
            case HISTORY_SEARCH:
                label = g_strdup_printf(_("Search \"%s\""), e->string);
                break;

            case HISTORY_ALBUM:
                label = g_strdup_printf(_("View album %s"), e->string);
                break;

            case HISTORY_PLAYLIST:
                label = g_strdup_printf(_("View playlist %s"), e->string);
                break;

            default:
                label = NULL;
                break;
        }

        if (label) {
            hildon_touch_selector_append_text(HILDON_TOUCH_SELECTOR(selector), label);
            g_free(label);
        }
    }

    g_signal_connect(G_OBJECT (selector), "changed",
                     G_CALLBACK (selection_changed), u);

    /* cancel */
    g_signal_connect(G_OBJECT(u->history_dialog), "delete_event",
                     G_CALLBACK(cancel_cb), u);

    gtk_widget_set_size_request(content_area, -1, 300);
    gtk_box_pack_start(GTK_BOX(content_area),
                       selector,
                       TRUE,
                       TRUE,
                       0);

    gtk_widget_show_all(GTK_WIDGET(u->history_dialog));
    gtk_dialog_run(GTK_DIALOG(u->history_dialog));

}

// vim: expandtab shiftwidth=4

