#include <glib.h>

#include "history.h"

struct yaspot_history_t {
    GList *list;
    GList *oldest_entry;
};

yaspot_history_t* yaspot_history_new() {
    yaspot_history_t *h;

    h = g_new0(yaspot_history_t, 1);

    return h;
}

static void free_entry(yaspot_history_entry_t *e) {
    g_free(e->string);
    g_free(e->id);
    g_free(e);
}

static void free_cb(gpointer data, gpointer userdata) {
    yaspot_history_entry_t *e = (yaspot_history_entry_t*)data;

    g_assert(e);
    free_entry(e);
}

void yaspot_history_free(yaspot_history_t *h) {
    g_assert(h);

    g_list_foreach(h->list, free_cb, NULL);
    g_list_free(h->list);
    g_free(h);
}

static yaspot_history_entry_t* history_entry_new(yaspot_history_operation op, const char *string, const char *id) {
    yaspot_history_entry_t *e;
    g_assert(string);
    g_assert(id);

    e = g_new0(yaspot_history_entry_t, 1);
    e->operation = op;
    e->string = g_strdup(string);
    e->id = g_strdup(id);
    return e;
}

void yaspot_history_add(yaspot_history_t* h, yaspot_history_operation op, const char *string, const char *id) {
    yaspot_history_entry_t *e = NULL;
    yaspot_history_entry_t *search;
    GList *remove;
    GList *i;
    gint length;
    g_assert(h);
    g_assert(string);
    g_assert(id);

    for (i = g_list_first(h->list); i; i = g_list_next(i)) {
        search = (yaspot_history_entry_t*)i->data;
        if (op == search->operation && 0 == g_strcmp0(id, search->id)) {
            h->list = g_list_remove_link(h->list, i);
            e = search;
            g_list_free(i);
        }
    }

    if (!e)
        e = history_entry_new(op, string, id);

    if (g_list_length(h->list) == YASPOT_HISTORY_LENGTH) {
        remove = h->oldest_entry;
        h->oldest_entry = g_list_previous(h->oldest_entry);
        h->list = g_list_remove_link(h->list, remove);
        g_list_foreach(remove, free_cb, NULL);
        g_list_free(remove);
    }

    h->list = g_list_prepend(h->list, e);

    if (!h->oldest_entry)
        h->oldest_entry = h->list;
}

void test_history(yaspot_history_t *h) {
    yaspot_history_entry_t *e;
    const yaspot_history_entry_t *b;
    GList *i;
    int n;

    for (n = 0, i = g_list_first(h->list); i; n++, i = g_list_next(i)) {
        e = (yaspot_history_entry_t*)i->data;
        g_print("history %d: OP %d DATA %s\n", n, e->operation, e->string);
    }

    void *iter = NULL;
    n = 0;

    while ((b = yaspot_history_iterate(h, &iter)))
        g_print("history %d: OP %d DATA %s ID %s\n", n++, b->operation, b->string, b->id);
}

const yaspot_history_entry_t* yaspot_history_iterate(yaspot_history_t *h, void **state) {
    GList *position;
    g_assert(h);

    if (*state) {
        position = *state;
        position = position->next;
        if (position) {
            *state = (void*)position;
            return position->data;
        } else {
            *state = NULL;
            return NULL;
        }
    } else {
        *state = (void*)h->list;
        if (h->list)
            return h->list->data;
        else
            return NULL;
    }
}

// vim: expandtab shiftwidth=4

